function C = lt (A, B)
%A < B less than.
% C = (A < B) is an element-by-element comparison of A and B.  One or
% both may be scalars.  Otherwise, A and B must have the same size.
%
% See also GrB/le, GrB/gt, GrB/ge, GrB/ne, GrB/eq.

% FUTURE: lt(A,B) for two matrices A and B is slower than it could be.
% See comments in gb_union_op.

% The pattern of C depends on the type of inputs:
% A scalar, B scalar:  C is scalar.
% A scalar, B matrix:  C is full if A<0, otherwise C is a subset of B.
% B scalar, A matrix:  C is full if B>0, otherwise C is a subset of A.
% A matrix, B matrix:  C has the pattern of the set union, A+B.

% SuiteSparse:GraphBLAS, Timothy A. Davis, (c) 2017-2020, All Rights
% Reserved. http://suitesparse.com.  See GraphBLAS/Doc/License.txt.

if (isobject (A))
    A = A.opaque ;
end

if (isobject (B))
    B = B.opaque ;
end

[am, an, atype] = gbsize (A) ;
[bm, bn, btype] = gbsize (B) ;
a_is_scalar = (am == 1) && (an == 1) ;
b_is_scalar = (bm == 1) && (bn == 1) ;
ctype = gboptype (atype, btype) ;

if (a_is_scalar)
    if (b_is_scalar)
        % both A and B are scalars
        C = GrB (gb_union_op ('<', A, B)) ;
    else
        % A is a scalar, B is a matrix
        if (gb_scalar (A) < 0)
            if (~gb_issigned (btype))
                % a < 0, and B has an unsigned type.  C is all true.
                C = GrB (gb_scalar_to_full (bm, bn, 'logical', true)) ;
            else
                % since a < 0, entries not present in B result in a true
                % value, so the result is dense.  Expand A to dense.
                A = gb_scalar_to_full (bm, bn, ctype, A) ;
                C = GrB (gbemult (A, '<', gbfull (B, ctype))) ;
            end
        else
            % since a >= 0, entries not present in B result in a false
            % value, so the result is a sparse subset of B.  select all
            % entries in B > a, then convert to true.
            C = GrB (gbapply ('1.logical', gbselect (B, '>', A))) ;
        end
    end
else
    if (b_is_scalar)
        % A is a matrix, B is a scalar
        b = gb_scalar (B) ;
        if (b < 0 && ~gb_issigned (atype))
            % b is negative, and A has an unsigned type.  C is all false.
            C = GrB (gbnew (am, an, 'logical')) ;
        elseif (b > 0)
            % since b > 0, entries not present in A result in a true
            % value, so the result is dense.  Expand B to a dense matrix.
            B = gb_scalar_to_full (am, an, ctype, B) ;
            C = GrB (gbemult (gbfull (A, ctype), '<', B)) ;
        else
            % since b <= 0, entries not present in A result in a false
            % value, so the result is a sparse subset of A.  Select all
            % entries in A < b, then convert to true.
            C = GrB (gbapply ('1.logical', gbselect (A, '<', B))) ;
        end
    else
        % both A and B are matrices.  C is the set union of A and B.
        C = GrB (gb_union_op ('<', A, B)) ;
    end
end

