% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dmixt.R
\name{dmixt}
\alias{dmixt}
\title{Fast density computation for mixture of multivariate Student's t distributions.}
\usage{
dmixt(X, mu, sigma, df, w, log = FALSE, ncores = 1, isChol = FALSE,
  A = NULL)
}
\arguments{
\item{X}{matrix n by d where each row is a d dimensional random vector. Alternatively \code{X} can be a d-dimensional vector.}

\item{mu}{an (m x d) matrix, where m is the number of mixture components.}

\item{sigma}{as list of m covariance matrices (d x d) on for each mixture component. 
Alternatively it can be a list of m cholesky decomposition of the covariance. 
In that case \code{isChol} should be set to \code{TRUE}.}

\item{df}{a positive scalar representing the degrees of freedom. All the densities in the mixture have the same \code{df}.}

\item{w}{vector of length m, containing the weights of the mixture components.}

\item{log}{boolean set to true the logarithm of the pdf is required.}

\item{ncores}{Number of cores used. The parallelization will take place only if OpenMP is supported.}

\item{isChol}{boolean set to true is \code{sigma} is the cholesky decomposition of the covariance matrix.}

\item{A}{an (optional) numeric matrix of dimension (m x d), which will be used to store the evaluations of each mixture
density over each mixture component. It is useful when m and n are large and one wants to call \code{dmixt()} 
several times, without reallocating memory for the whole matrix each time. NB1: \code{A} will be modified, 
not copied! NB2: the element of \code{A} must be of class "numeric".}
}
\value{
A vector of length n where the i-the entry contains the pdf of the i-th random vector (i.e. the i-th row of \code{X}).
}
\description{
Fast density computation for mixture of multivariate Student's t distributions.
}
\details{
There are many candidates for the multivariate generalization of Student's t-distribution, here we use
         the parametrization described here \url{https://en.wikipedia.org/wiki/Multivariate_t-distribution}. NB: at the moment 
         the parallelization does not work properly on Solaris OS when \code{ncores>1}. Hence, \code{dmixt()} checks if the OS 
         is Solaris and, if this the case, it imposes \code{ncores==1}.
}
\examples{
#### 1) Example use
# Set up mixture density
df <- 6
mu <- matrix(rep(c(1, 2, 10, 20), 2), 2, 2, byrow = TRUE)
sigma <- list(diag(c(1, 10)), matrix(c(1, -0.9, -0.9, 1), 2, 2))
w <- c(0.1, 0.9)

# Simulate
X <- rmixt(1e4, mu, sigma, df, w)

# Evaluate density
ds <- dmixt(X, mu, sigma, w = w, df = df)
head(ds)

##### 2) More complicated example
# Define mixture
set.seed(5135)
N <- 10000
d <- 2
df = 10
w <- rep(1, 2) / 2
mu <- matrix(rep(c(0, 0, 2, 3), 2), 2, 2, byrow = TRUE) 
sigma <- list(matrix(c(1, 0, 0, 2), 2, 2), matrix(c(1, -0.9, -0.9, 1), 2, 2)) 

# Simulate random variables
X <- rmixt(N, mu, sigma, w = w, df = df, retInd = TRUE)

# Plot mixture density
np <- 100
xvals <- seq(min(X[ , 1]), max(X[ , 1]), length.out = np)
yvals <- seq(min(X[ , 2]), max(X[ , 2]), length.out = np)
theGrid <- expand.grid(xvals, yvals) 
theGrid <- as.matrix(theGrid)
dens <- dmixt(theGrid, mu, sigma, w = w, df = df)
plot(X, pch = '.', col = attr(X, "index")+1)
contour(x = xvals, y = yvals, z = matrix(dens, np, np),
        levels = c(0.002, 0.01, 0.02, 0.04, 0.08, 0.15 ), add = TRUE, lwd = 2)

}
\author{
Matteo Fasiolo <matteo.fasiolo@gmail.com>.
}
