/*
 * GWT - General Windowing Toolkit
 *
 * Copyright (C) 1998 MenTaLguY - mentalg@geocities.com
 *                    Rodolphe Ortalo - ortalo@laas.fr
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: id.c,v $
 * Revision 1.3  1998/08/23 18:57:11  ortalo
 * Minor editing (mainly comments).
 *
 * Revision 1.2  1998/08/11 22:17:34  ortalo
 * Adapted to the gwt.h->ggi/gwt.h change. Incorporated the modifications
 * brought by Tristan Wibberley.
 *
 * Revision 1.1  1998/07/09 18:38:39  ortalo
 * Initial integration of libgwt in the GGI repository.
 * This is experimental code, for your eyes only !
 * (Don't execute... ;-)
 *
 */

#include <stdlib.h>
#include <malloc.h>
#include <search.h>
#include <assert.h>

#include "ggi/gwt.h"
#include "internal.h"

/*
 * Private data type for dictionnary
 */
typedef struct {
	gwt_id_t id;
	gwt_window_t win;
} id_record;

/*
 * Static variables
 */
#ifdef USE_THREADS
#include <pthread.h>
static pthread_mutex_t id_database_mutex = PTHREAD_MUTEX_INITIALIZER;
#endif

static gwt_id_t next_id=1;
static void *id_tree=NULL;

/*********************************************************************
 *                        STATIC FUNCTIONS                           *
 *********************************************************************/

/*
 * Static internal functions
 */

static int id_compare(const void *av, const void *bv)
{
#	define a ((id_record *)av)
#	define b ((id_record *)bv)
	
	if ( a->id > b->id ) return 1;
	if ( a->id < b->id ) return -1;
	return 0;

#	undef a
#	undef b
}

/*
 * Locking for the id system
 */
#ifdef USE_THREADS
static int _gwt_id_sys_lock(void)
{
	int status;
	status = pthread_mutex_lock(&id_database_mutex);
	assert(!status);
	return status;
}
#else
#define _gwt_id_sys_lock() 0
#endif

#ifdef USE_THREADS
static int _gwt_id_sys_unlock(void)
{
	int status;
	status = pthread_mutex_unlock(&id_database_mutex);
	assert(!status);
	return status;
}
#else
#define _gwt_id_sys_unlock() 0
#endif

/* find the next free id after the specified id; the id system should already
   have been locked */
static gwt_id_t _gwt_find_next_free_id(gwt_id_t id)
{
	return GWT_NULL_ID; /* FIXME !!! cheap cop-out; never succeeds */
}

/*************************************************************************
 *                    GWT PRIVATE EXPORTED FUNCTIONS                     *
 *************************************************************************/

/*
 * Creation of a new unique id
 */
gwt_id_t gwt_allocate_id()
{
	gwt_id_t retval;
	id_record *recp;	

	recp = (id_record *)malloc(sizeof(id_record));
	if ( recp == NULL ) return GWT_NULL_ID; /* TODO: Return error */
	_gwt_id_sys_lock();
	recp->id = next_id;
	next_id++;
	recp->win = NULL;
	{
		id_record **found_recpp;
		found_recpp = (id_record **)tsearch((const void *)recp,
		                                    (void **)&id_tree, id_compare);
		assert(found_recpp != NULL);
		if ( *found_recpp != recp ) {
			/* the id was not unique; we probably wrapped */
			/* if _gwt_find_next_free_id finds an id for us, great.
			   it could possibly fail (returning GWT_NULL_ID) if all
			   the ids were allocated, which should be extremely
			   unlikely (uh, like, never?), but it is best to be
			   prepared for that remote possibility anyway. */
			retval = _gwt_find_next_free_id(recp->id);
			if ( retval != GWT_NULL_ID ) {
				recp->id = retval;
				found_recpp = (id_record **)tsearch((const void *)recp,
				                                    (void **)&id_tree,
				                                    id_compare);
				assert(found_recpp != NULL);
				assert(*found_recpp == recp);
			} else {
				/* give up */
				free(recp);
			}
		} else {
			retval = recp->id;
		}
	}
	_gwt_id_sys_unlock();
	return retval;
}

/*
 * Associate win to id
 */
int gwt_associate_id(gwt_id_t id, gwt_window_t win)
{
	int status=0;
	id_record rec;
	id_record *recp;

	rec.id = id;
	_gwt_id_sys_lock();
	recp = tfind((const void *)&rec, (const void **)&id_tree, id_compare);
	if ( recp == NULL ) {
		status = -1;
	} else {
		recp = *((id_record **)recp);
		recp->win = win;
	}
	_gwt_id_sys_unlock();
	return status;
}

/*
 * Now try to fetch a window from its id
 */
int gwt_get_window_from_id(gwt_id_t id, gwt_window_t *winp)
{
	int status=0;
	id_record rec;
	id_record *recp;

	assert(winp != NULL);
	rec.id = id;
	_gwt_id_sys_lock();
	recp = tfind((const void *)&rec, (const void **)&id_tree, id_compare);
	if ( recp == NULL ) {
		status = -1; /* TODO: Return real error code */
	} else {
		recp = *((id_record **)recp);
		*winp = recp->win;
	}
	_gwt_id_sys_unlock();
	return status;
}

/*
 * Release an id
 */
void gwt_free_id(gwt_id_t id)
{
	id_record rec;
	id_record *recp;

	rec.id = id;
	_gwt_id_sys_lock();
	recp = tfind((const void *)&rec, (const void **)&id_tree, id_compare);
	assert(recp != NULL)
	{
#ifndef NDEBUG
		id_record **tp;
#endif
		recp = *((id_record **)recp);
#ifndef NDEBUG
		tp = (id_record **) /* yes, this is correct! see below... */
#endif
		tdelete((const void *)recp, (void **)&id_tree, id_compare);
#ifndef NDEBUG
		assert(tp != NULL);
#endif
		free(recp);
	}
	_gwt_id_sys_unlock();
}
