/* 
 *  Graphics context manipulation
 *
 *  Copyright (C) 1998 Thomas Tanner. See CREDITS for details.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "internal2d.h"
#include "util.h"

/*----------------------------------------------------------------------*/

/* function table updating */

void	GGI2DregisterUpdateFunc(ggi_visual_t vis, updatefunc f, ggi_uint mask)
{
	ggi2d_data	*data = GGI2D_DATA(vis);
	
	if (data->modules == MAX_MODULES) {
		DPRINT("Error: too much modules loaded on %p\n",vis);
		return;
	}
	data->module[data->modules].update = f;
	data->module[data->modules].change_mask = mask;
	data->change_mask |= mask;
	data->modules++;
}

void	GGI2DinitGC(ggi_visual_t vis)
{
	GGI2D_GC(vis).arcmode = GGI2D_ARC_SECTOR;
	GGI2D_GC(vis).polymode = GGI2D_POLY_EVENODD;
	GGI2D_GC_DASHSIZE(vis) = 0;
	GGI2D_GC_APPEND(vis) = 0;
	GGI2D_GC_DRAWCOL(vis) = 0; 
	GGI2D_GC_FILLCOL(vis) = 0; 
	GGI2D_GC_TEXTURED(vis) = 0;
	GGI2D_GC_CLIP_X1(vis) = 0; 
	GGI2D_GC_CLIP_Y1(vis) = 0; 
	GGI2D_GC_CLIP_X2(vis) = LIBGGI_VIRTX(vis); 
	GGI2D_GC_CLIP_Y2(vis) = LIBGGI_VIRTY(vis); 
	GGI2D_GC_OP(vis) = GGI2D_SET; 
	GGI2D_ACCELCHANGES(vis) = GGI2D_UPD_ALL; 
}

void	GGI2Dupdate(ggi_visual_t vis, ggi_uint change)
{
	ggi_uint	i;
	ggi2d_data	*data = GGI2D_DATA(vis);
	
	if (change & data->change_mask) {
		for (i = 0;  i < data->modules; i++)
			if (change & data->module[i].change_mask)
				data->module[i].update(vis, change);
	}
	data->gc.accel_changes |= change; 
}

/*----------------------------------------------------------------------*/

/* clipping */

int 	ggi2dSetClip(ggi_visual_t vis, 
			ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2)
{
	_ggiLock(vis->mutex); 
	if (x1 > x2) swap(x1, x2);
	if (y1 > y2) swap(y1, y2);
	if (x1 < 0) x1 = 0;
	if (y1 < 0) y1 = 0;
	if (x2 > LIBGGI_VIRTX(vis)) x2 = LIBGGI_VIRTX(vis);
	if (y2 > LIBGGI_VIRTY(vis)) y2 = LIBGGI_VIRTY(vis);
	GGI2D_GC_CLIP_X1(vis) = x1; 
	GGI2D_GC_CLIP_Y1(vis) = y1; 
	GGI2D_GC_CLIP_X2(vis) = x2; 
	GGI2D_GC_CLIP_Y2(vis) = y2; 
	DPRINT("changing clip rect to ((%d,%d),(%d,%d))\n", x1, y1, x2, y2);
	GGI2Dupdate(vis, GGI2D_UPD_CLIP);    
	_ggiUnlock(vis->mutex);
        return 0;
}

int    	ggi2dPointVisible(ggi_visual_t vis, ggi_sint x, ggi_sint y)
{
	return CLIP(vis,x,y);
}


int    	ggi2dRectVisible(ggi_visual_t vis,
			ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2)
{
	int cx1, cy1, cx2, cy2;
	
	if (x1 > x2) swap(x1, x2);
	if (y1 > y2) swap(y1, y2);
	cx1 = GGI2D_GC_CLIP_X1(vis);
	cy1 = GGI2D_GC_CLIP_Y1(vis);
	cx2 = GGI2D_GC_CLIP_X2(vis);
	cy2 = GGI2D_GC_CLIP_Y2(vis);
	return (x1 < cx2 && y1 < cy2 && x2 > cx1 && y2 > cy1);
}

/*----------------------------------------------------------------------*/

/* manipulation of the GC */

void	ggi2dReset(ggi_visual_t vis)
{
	DPRINT("reset of visual %p\n", vis);
	GGI2DinitGC(vis);
	GGI2Dupdate(vis, GGI2D_UPD_ALL);	
}


int	ggi2dSetArcMode(ggi_visual_t vis, ggi2d_arcmode mode)
{
	_ggiLock(vis->mutex); 
	if (GGI2D_GC(vis).arcmode != mode) {
		GGI2D_GC(vis).arcmode = mode;
		DPRINT("changing arcmode to %d\n", (int)mode);
		GGI2Dupdate(vis, GGI2D_UPD_ARCMODE);
	}
	_ggiUnlock(vis->mutex);
        return 0;
}


ggi2d_arcmode	ggi2dGetArcMode(ggi_visual_t vis)
{
	return GGI2D_GC(vis).arcmode;
}


int	ggi2dSetPolyMode(ggi_visual_t vis, ggi2d_polymode mode)
{
	_ggiLock(vis->mutex); 
	if (GGI2D_GC(vis).polymode != mode) {
		GGI2D_GC(vis).polymode = mode;
		DPRINT("changing polymode to %d\n", (int)mode);
		GGI2Dupdate(vis, GGI2D_UPD_POLYMODE);
	}
	_ggiUnlock(vis->mutex); 
        return 0;
}


ggi2d_polymode	ggi2dGetPolyMode(ggi_visual_t vis)
{
	return GGI2D_GC(vis).polymode;
}


int	ggi2dSetLineDash(ggi_visual_t vis, ggi_uint dash[], ggi_uint size)
{
	_ggiLock(vis->mutex); 
	if (GGI2D_GC_DASHSIZE(vis) * size == 0) { 
		GGI2D_GC_DASH(vis) = dash;
		GGI2D_GC_DASHSIZE(vis) = size;
		DPRINT("changing line dash (size %d)\n", size);
		GGI2Dupdate(vis, GGI2D_UPD_LINEMODE);
	} else
	if (GGI2D_GC_DASHSIZE(vis) != size ||
	    memcmp(GGI2D_GC_DASH(vis), dash, size)) {
		GGI2D_GC_DASH(vis) = dash;
		GGI2D_GC_DASHSIZE(vis) = size;
		DPRINT("changing line dash (size %d)\n", size);
		GGI2Dupdate(vis, GGI2D_UPD_LINE);
	}
	_ggiUnlock(vis->mutex); 
        return 0;
}


int	ggi2dGetLineDash(ggi_visual_t vis, ggi_uint dash[], ggi_uint *size)
{
	if ((*size = GGI2D_GC_DASHSIZE(vis))) 
		dash = GGI2D_GC_DASH(vis);
	else  
		dash = NULL;
        return 0;
}


int	ggi2dSetAppendMode(ggi_visual_t vis, int append)
{
	_ggiLock(vis->mutex); 
	if ((GGI2D_GC_APPEND(vis)) != (append)) {
		GGI2D_GC_APPEND(vis) = append;
		DPRINT("changing append mode to %d\n", append);
		GGI2Dupdate(vis, GGI2D_UPD_APPENDMODE);
	}
	_ggiUnlock(vis->mutex); 
        return 0;
}


int    	ggi2dGetAppendMode(ggi_visual_t vis)
{
	return GGI2D_GC_APPEND(vis);
}


int	ggi2dSetDrawColor(ggi_visual_t vis, ggi_pixel color)
{
	_ggiLock(vis->mutex); 
	if (GGI2D_GC_DRAWCOL(vis) != color) {
		GGI2D_GC_DRAWCOL(vis) = color;
		DPRINT("changing drawcolor to %d\n", color);
		GGI2Dupdate(vis, GGI2D_UPD_DRAWCOL);
	}
	_ggiUnlock(vis->mutex); 
        return 0;
}


ggi_pixel	ggi2dGetDrawColor(ggi_visual_t vis)
{
	return GGI2D_GC_DRAWCOL(vis);
}


int	ggi2dSetFillColor(ggi_visual_t vis, ggi_pixel color)
{
	_ggiLock(vis->mutex); 
	if (GGI2D_GC_TEXTURED(vis)) {
	/* FIXME: release texture */
		GGI2D_GC_TEXTURED(vis) = 0;
		GGI2D_GC_FILLCOL(vis) = color;
		DPRINT("changing fillcolor to %d\n", color);
		GGI2Dupdate(vis, GGI2D_UPD_TEXTUREMODE);	
	} else
	if (GGI2D_GC_FILLCOL(vis) != color) {
		GGI2D_GC_FILLCOL(vis) = color;
		DPRINT("changing fillcolor to %d\n", color);
		GGI2Dupdate(vis, GGI2D_UPD_FILLCOL);	
	}
	_ggiUnlock(vis->mutex); 
        return 0;
}


ggi_pixel	ggi2dGetFillColor(ggi_visual_t vis)
{
	return GGI2D_GC_FILLCOL(vis);
}


int	ggi2dSetFillTexture(ggi_visual_t vis, ggi2d_coord refpoint, 
			ggi2d_image texture)
{
	if (GGI2D_GC_TEXTURED(vis)) {
	/* FIXME: release old texture */
	}	
	/* FIXME: copy texture */
	GGI2D_GC_TEXTURE(vis) = texture;
	refpoint.x %= LIBGGI_VIRTX(vis);
	refpoint.y %= LIBGGI_VIRTY(vis);
	GGI2D_GC_TEXREF(vis).x = refpoint.x;
	GGI2D_GC_TEXREF(vis).y = refpoint.y;
	DPRINT("changing texture to (%d,%d) %p\n", 
		refpoint.x, refpoint.y, texture);
	if (!GGI2D_GC_TEXTURED(vis)) {
			GGI2D_GC_TEXTURED(vis) = 1;
		GGI2Dupdate(vis, GGI2D_UPD_TEXTUREMODE);	
	} else
		GGI2Dupdate(vis, GGI2D_UPD_TEXTURE);
	_ggiUnlock(vis->mutex); 
        return 0;
}


int	ggi2dGetFillTexture(ggi_visual_t vis, ggi2d_coord *refpoint, 
			ggi2d_image *texture)
{
	if (!GGI2D_GC_TEXTURED(vis)) {
		refpoint->x = refpoint->y = 0;
		*texture = NULL;
	} else {
		refpoint->x = GGI2D_GC_TEXREF(vis).x;
		refpoint->y = GGI2D_GC_TEXREF(vis).y;
		*texture = GGI2D_GC_TEXTURE(vis);
	}
	return 0;
}


int	ggi2dSetOperator(ggi_visual_t vis, ggi2d_operator operator)
{
	_ggiLock(vis->mutex); 
	if (GGI2D_GC_OP(vis) != operator) {
		GGI2D_GC_OP(vis) = operator;
		DPRINT("changing operator to %d\n", (int)operator);
		GGI2Dupdate(vis, GGI2D_UPD_OPERATOR);	
	}
	_ggiUnlock(vis->mutex); 
        return 0;
}


ggi2d_operator	ggi2dGetOperator(ggi_visual_t vis)
{
	return GGI2D_GC_OP(vis);
}

