/* $Id: mouse.inc,v 1.6 1998/10/26 04:11:02 ajapted Exp $
***************************************************************************

   Linux_common/mouse.c
   Linux mouse parsing code.

   Copyright (C) 1998  Andrew Apted  <andrew.apted@ggi-project.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   ---------------------------------------------------------------------

   USAGE:

	1. Define MOUSE_FD(vis) to be a variable (type: int) which
	can be used to hold the mouse file descriptor, and initialize
	it to -1 for good measure.
	
	2. Define MOUSE_PRIV(vis) to be a variable (type: void *)
	where some private information can be stored.  Initialize the
	pointer to NULL for good measure.

	3. Call mouse_init(vis, typename).  The will open the mouse
	device and initialize everything.  Returns 0 if successful.

	4. When mouse data is available (e.g. by using select on
	MOUSE_FD(vis)), call mouse_handle_data(vis).  This will
	read data from the mouse device, convert them to GGI mouse
	events, and add them to the event queue.

	5. When finished, call mouse_exit(vis), which will do any
	restorations and close the mouse device.

***************************************************************************
*/

#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <termios.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <linux/kdev_t.h>   /* only needed for MAJOR() macro */
#include <linux/major.h>    /* only needed for MISC_MAJOR */


#define MAX_MOUSE_NAMES  8
#define MAX_MOUSE_TYPES  32

#define MAX_PACKET_BUF	128

struct mouse_hook;

typedef struct mouse_type
{
	const char *names[MAX_MOUSE_NAMES];

	/* Serial parameters.  If the mouse is not a serial mouse
	 * (for example, a busmouse), then default_baud should be < 0.
	 */

	int default_baud;
	int cflag;  int iflag;
	int oflag;  int lflag;

	/* The parser function.
	 *
	 * This returns the number of bytes used by the packet.  It
	 * should return 0 when the packet is too short, and return 1
	 * when the packet is invalid (to resync).  len is guaranteed
	 * to be >= minimum_packet_size.
	 */

	int (* parse_func)(ggi_visual *vis, uint8 *buf, int len);

	int minimum_packet_len;
	
} MouseType;

typedef struct mouse_hook
{
	MouseType *type;

	int button_state;
	int parse_state;
	int left_hand;

	struct termios old_termios;

	/* mouse packet buffer */
	
	int packet_len;
	uint8 packet_buf[MAX_PACKET_BUF];

} MouseHook;

#define MOUSE_HOOK(vis)  ((MouseHook *) MOUSE_PRIV(vis))


#define C_NORM	(CREAD | CLOCAL | HUPCL)
#define I_NORM	(IGNBRK)

#define DPRINT1  DPRINT
#define DPRINT2  DPRINT
#define DPRINT3  DPRINT


/* ---------------------------------------------------------------------- */


/**
 **  Event dispatching routines
 **/

static void mouse_send_movement(ggi_visual *vis, int dx, int dy)
{
	ggi_event mouse_event;

	if (dx || dy) {

		mouse_event.pmove.size   = sizeof(ggi_pmove_event);
		mouse_event.pmove.type   = evPtrRelative;
		mouse_event.pmove.origin = EV_ORIGIN_NONE;
		mouse_event.pmove.target = EV_TARGET_NONE;

		EV_TIMESTAMP(& mouse_event);

		mouse_event.pmove.x = dx;
		mouse_event.pmove.y = dy;

		_ggiEvQueueAdd(vis, &mouse_event);
	}
}

static void mouse_send_buttons(ggi_visual *vis, int buttons, int last)
{
	ggi_event mouse_event;

	int mask;
	int changed = buttons ^ last;

	
	/* change in button state ? */

	for (mask = 1; mask > 0; mask <<= 1)

	if (changed & mask) {

		int state = buttons & mask;

		mouse_event.pbutton.size = sizeof(ggi_pbutton_event);
		mouse_event.pbutton.type = state ?
			evPtrButtonPress : evPtrButtonRelease;
		mouse_event.pmove.origin = EV_ORIGIN_NONE;
		mouse_event.pmove.target = EV_TARGET_NONE;

		EV_TIMESTAMP(& mouse_event);

		mouse_event.pbutton.button = mask;

		_ggiEvQueueAdd(vis, &mouse_event);
	}
}


/**
 **  Mouse Packet Parsers
 **/

static int B_microsoft[16] =
{	0x0, 0x2, 0x1, 0x3, 0x4, 0x6, 0x5, 0x7,
	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_microsoft(ggi_visual *vis, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(vis);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0x40) != 0x40) || ((buf[1] & 0x40) != 0x00)) {

		DPRINT3("Invalid microsoft packet\n");
		return 1;
	}

	buttons = B_microsoft[hand | ((buf[0] & 0x30) >> 4)];

	dx = (sint8) (((buf[0] & 0x03) << 6) | (buf[1] & 0x3f));
	dy = (sint8) (((buf[0] & 0x0c) << 4) | (buf[2] & 0x3f));

	mouse_send_movement(vis, dx, dy);
	mouse_send_buttons(vis, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT3("Got microsoft packet\n");

	return 3;
} 

static MouseType T_microsoft =
{
	{ "Microsoft", "ms", NULL, },
	B1200, C_NORM | CS7, I_NORM, 0, 0,
	M_microsoft, 3 
};


static int B_mousesys[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_mousesys(ggi_visual *vis, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(vis);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if ((buf[0] & 0xf8) != 0x80) {

		DPRINT3("Invalid mousesys packet\n");
		return 1;
	}

	buttons = B_mousesys[hand | (~buf[0] & 0x07)];

	dx =  (sint8) buf[1] + (sint8) buf[3];
	dy = -(sint8) buf[2] - (sint8) buf[4];

	mouse_send_movement(vis, dx, dy);
	mouse_send_buttons(vis, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT3("Got mousesys packet\n");

	return 5;
}

static MouseType T_mousesys =
{
	{ "MouseSystems", "mousesystem", "mousesys", "msc", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_mousesys, 5
};


static int B_logitech[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_logitech(ggi_visual *vis, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(vis);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0xe0) != 0x80) || ((buf[1] & 0x80) != 0x00)) {

		DPRINT3("Invalid logitech packet\n");
		return 1;
	}

	buttons = B_logitech[hand | (buf[0] & 0x07)];

	dx = (buf[0] & 0x10) ?	(sint8)buf[1] : -(sint8)buf[1];
	dy = (buf[0] & 0x08) ? -(sint8)buf[2] :  (sint8)buf[2];

	mouse_send_movement(vis, dx, dy);
	mouse_send_buttons(vis, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT3("Got logitech packet\n");

	return 3;
}

static MouseType T_logitech = 
{
	{ "Logitech", "logi", "log", "Logitech", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_logitech, 3
};


static int B_sun[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_sun(ggi_visual *vis, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(vis);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if ((buf[0] & 0xf8) != 0x80) {

		DPRINT3("Invalid sun packet\n");
		return 1;
	}

	buttons = B_sun[hand | (~buf[0] & 0x07)];

	dx =  (sint8) buf[1];
	dy = -(sint8) buf[2];

	mouse_send_movement(vis, dx, dy);
	mouse_send_buttons(vis, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT3("Got sun packet\n");

	return 3;
}

static MouseType T_sun =
{
	{ "Sun", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_sun, 3
};


static int B_mouseman[16] =
{	0x0, 0x2, 0x1, 0x3, 0x4, 0x6, 0x5, 0x7,
	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_mouseman(ggi_visual *vis, uint8 *buf, int len)
{
	/*  The damned MouseMan has 3/4 bytes packets.	The extra byte
	 *  is only there if the middle button is active.
	 *
	 *  This is what we do: when we get the first 3 bytes, we parse
	 *  the info and send off the events, and set a flag to say we
	 *  have seen the first three bytes.
	 * 
	 *  When we get the fourth byte (maybe with the first three,
	 *  or maybe later on), we check if it is a header byte.
	 *  If so, we return 3, otherwise we parse the buttons in it,
	 *  send off the events, and return 4.
	 *
	 *  Note also that unlike the other mices, the mouseman parser
	 *  stores the RAW buttons in priv->button_state.
	 */

	MouseHook *mhook = MOUSE_HOOK(vis);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0x40) != 0x40) || ((buf[1] & 0x40) != 0x00)) {

		DPRINT3("Invalid mouseman packet\n");
		return 1;
	}

	/* handle the common 3 bytes */

	if (mhook->parse_state == 0) {

		buttons = (mhook->button_state & 0x4) |
			((buf[0] & 0x30) >> 4);

		dx = (sint8) (((buf[0] & 0x03) << 6) | (buf[1] & 0x3f));
		dy = (sint8) (((buf[0] & 0x0c) << 4) | (buf[2] & 0x3f));

		mouse_send_movement(vis, dx, dy);
		mouse_send_buttons(vis, B_mouseman[hand | buttons],
			B_mouseman[hand | mhook->button_state]);

		mhook->button_state = buttons;
		mhook->parse_state  = 1;

		DPRINT3("Got mouseman base packet\n");
	}


	/* now look for extension byte */

	if (len < 4) {
		return 0;
	}

	mhook->parse_state = 0;

	if ((buf[3] & 0xc0) != 0) {
	
		/* 4th byte must be a header byte */
		
		return 3;
	}

	/* handle the extension byte */

	buttons = (mhook->button_state & 0x3) | ((buf[3] & 0x20) >> 3);

	mouse_send_buttons(vis, B_mouseman[hand | buttons],
		B_mouseman[hand | mhook->button_state]);

	mhook->button_state = buttons;

	DPRINT3("Got mouseman extension packet\n");

	return 4;
}

static MouseType T_mouseman =
{
	{  "MouseMan", "mman", NULL, },
	B1200, C_NORM | CS7, I_NORM, 0, 0,
	M_mouseman, 3
};


/* 
 * MMSeries mice use the same protocal as Logitech (and vice-versa)
 */

static MouseType T_mmseries =
{
	{ "MMSeries", "mm", NULL, },
	B1200, C_NORM | CS8 | PARENB | PARODD, I_NORM, 0, 0,
	M_logitech, 3
};
	

/*
 * BusMice use the same protocal as Sun
 */

static MouseType T_busmouse =
{
	{ "BusMouse", "bus", "bm", NULL, },
	-1, 0, 0, 0, 0,
	M_sun, 3
};

static int B_ps2[16] =
{	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7,
	0x0, 0x2, 0x1, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_ps2(ggi_visual *vis, uint8 *buf, int len)
{
	MouseHook *mhook = MOUSE_HOOK(vis);

	int dx, dy, buttons;
	int hand = mhook->left_hand ? 8 : 0;

	/* Check header byte. */
	if ((buf[0] & 0xc0) != 0) {
		DPRINT_EVENTS("Invalid PS2 packet\n");
		return 1;
	}

	buttons = B_ps2[hand | (buf[0] & 0x07)];

	dx = (buf[0] & 0x10) ? buf[1] - 256 : buf[1];
	dy = (buf[0] & 0x20) ? -(buf[2] - 256) : -buf[2];

	mouse_send_movement(vis, dx, dy);
	mouse_send_buttons(vis, buttons, mhook->button_state);

	mhook->button_state = buttons;

	DPRINT_EVENTS("Got PS2 packet\n");

	return 3;
}

static MouseType T_ps2 =
{
 	{ "PS2", "ps", NULL, },
	-1, 0, 0, 0, 0,
 	M_ps2, 3
};



/* ---------------------------------------------------------------------- */


/* The Holy Table of Mouse Parsers
*/

static MouseType *mice_types[MAX_MOUSE_TYPES] =
{
	& T_microsoft, 
	& T_mousesys,
	& T_logitech,
	& T_sun,
	& T_mmseries,
	& T_mouseman,
	& T_busmouse,
	& T_ps2,
	
	NULL,
};

#define DEFAULT_MOUSE  0

/* ---------------------------------------------------------------------- */


/* Parse one packet
 */
 
static int do_parse_packet(ggi_visual *vis)
{
	int used;

#if 0
	{	int i;

		fprintf(stderr, "L/mouse: do_parse_packet [");

		for (i=0; i < (MOUSE_HOOK(vis)->packet_len - 1); i++) {
			fprintf(stderr, "%02x ", 
				MOUSE_HOOK(vis)->packet_buf[i]);
		}

		fprintf(stderr, "%02x]\n", MOUSE_HOOK(vis)->packet_buf[i]);
	}
#endif

	/* call parser function */
	
	used = MOUSE_HOOK(vis)->type->parse_func(vis, 
			MOUSE_HOOK(vis)->packet_buf,
			MOUSE_HOOK(vis)->packet_len);

	DPRINT3("packet used %d bytes\n", used);

	return used;
}


static int find_mouse(char *name)
{
	int m, n;

	for (m=0; (m < MAX_MOUSE_TYPES) && 
	          (mice_types[m] != NULL); m++) {

		MouseType *mtype = mice_types[m];

		for (n=0; (n < MAX_MOUSE_NAMES) && 
		          (mtype->names[n] != NULL); n++) {

			if (strcasecmp(mtype->names[n], name) == 0) {
			
				return m;  /* found it */
			}
		}
	}

	fprintf(stderr, "Unknown mouse type '%s' -- using the default.\n", 
		name);

	return DEFAULT_MOUSE;
}


#if 0
static int find_baud(char *baudname)
{
	switch (atoi(baudname))
	{
		case 9600: return B9600;
		case 4800: return B4800;
		case 2400: return B2400;
		case 1200: return B1200;
	}

	fprintf(stderr, "Baud rate '%s' not supported\n", baudname);
	
	return B1200;  /* !!! */
}
#endif

static int do_mouse_open(ggi_visual *vis, char *filename,
			 int dtr, int rts, int baud)
{
	int fd;

	struct termios new;


	fd = open(filename, O_RDWR | O_NOCTTY);

	if (fd < 0) {
		DPRINT("L/mouse: Failed to open '%s'.\n", filename);
		return -1;
	}

	if (MOUSE_HOOK(vis)->type->default_baud >= 0) {

		/* set up the termios state and baud rate */
	
		tcflush(fd, TCIOFLUSH);

		if (tcgetattr(fd, &MOUSE_HOOK(vis)->old_termios) < 0) {
			DPRINT("tcgetattr failed.\n");
			close(fd);
			return -1;
		}

		new = MOUSE_HOOK(vis)->old_termios;

		if (baud < 0) {
			baud = MOUSE_HOOK(vis)->type->default_baud;
		}
		
		new.c_cflag = MOUSE_HOOK(vis)->type->cflag | baud;
		new.c_iflag = MOUSE_HOOK(vis)->type->iflag;
		new.c_oflag = MOUSE_HOOK(vis)->type->oflag;
		new.c_lflag = MOUSE_HOOK(vis)->type->lflag;
		new.c_cc[VMIN]  = 1;
		new.c_cc[VTIME] = 0;

		if (tcsetattr(fd, TCSANOW, &new) < 0) {
			DPRINT("tcsetattr failed.\n");
			close(fd);
			return -1;
		}

		/* set up RTS and DTR modem lines */

		if ((dtr >= 0) || (rts >= 0))
		{
			unsigned int modem_lines;

			if (ioctl(fd, TIOCMGET, &modem_lines) == 0) {

				if (dtr == 0) modem_lines &= ~TIOCM_DTR;
				if (rts == 0) modem_lines &= ~TIOCM_RTS;

				if (dtr > 0) modem_lines |= TIOCM_DTR;
				if (rts > 0) modem_lines |= TIOCM_RTS;
				
				ioctl(fd, TIOCMSET, &modem_lines);
			}
		}
	}

	return fd;
}


static void do_mouse_close(ggi_visual *vis)
{
	if (tcsetattr(MOUSE_FD(vis),
		      TCSANOW, &MOUSE_HOOK(vis)->old_termios) < 0) {
		DPRINT("tcsetattr failed.\n");
	}

	close(MOUSE_FD(vis));

	MOUSE_FD(vis) = -1;
}


static char *parse_field(char *dst, int max, char *src)
{
	int len=1;   /* includes trailing NUL */

	for (; *src && (*src != ','); src++) {
		if (len < max) {
			*dst++ = *src;
			len++;
		}
	}

	*dst = 0;

	if (*src == ',') {
		src++;
	}
	return src;
}


static void parse_mouse_specifier(char *spec, char *protname,
				  char *devname, char *options)
{
	*protname = *devname = *options = 0;

	/* LISP-haters should shut their eyes now :) */

	if (spec) 
	parse_field(options, 255,
		    parse_field(devname, 255,
				parse_field(protname, 255, spec)));

	/* supply defaults for missing bits */

	if (*devname == 0) {
		strcpy(devname, "/dev/mouse");
	}

	if (*protname == 0) {

		/* Protocol hasn't been specified. So try statting the
		 * file to see if it is a char device with Major == 10.
		 * In this case, the protocol is most likely BusMouse.
		 */

		struct stat m_stat;

		strcpy(protname, "microsoft");

		if ((stat(devname, &m_stat) == 0) &&
		    S_ISCHR(m_stat.st_mode) &&
		    (MAJOR(m_stat.st_rdev) == MISC_MAJOR)) {
		    
			strcpy(protname, "busmouse");
		}
	}
}


static char *parse_opt_int(char *opt, int *val)
{
	*val = 0;

	for (; *opt && isdigit(*opt); opt++) {
		*val = ((*val) * 10) + ((*opt) - '0');
	}

	return opt;
}


static void parse_options(char *opt, int *baud, int *dtr, int *rts)
{
	while (*opt)
	switch (*opt++) {

		case 'b': case 'B':    /* baud */
		{
			opt = parse_opt_int(opt, baud);
			break;
		}

		case 'd': case 'D':    /* dtr */
		{
			opt = parse_opt_int(opt, dtr);
			break;
		}

		case 'r': case 'R':    /* rts */
		{
			opt = parse_opt_int(opt, rts);
			break;
		}

		default:
		{
			fprintf(stderr, "Unknown mouse option "
				"'%c' -- rest ignored.\n", *opt);
			return;
		}
	}
}


/* ---------------------------------------------------------------------- */


int mouse_init(ggi_visual *vis, char *typename)
{
	char protname[256];
	char devname[256];
	char options[256];

	int mindex;
	int dtr=-1, rts=-1, baud=-1;


	/* allocate mouse private structure */

	MOUSE_PRIV(vis) = _ggi_malloc(sizeof(MouseHook));

	/* parse the mouse specifier */
	
	parse_mouse_specifier(typename, protname, devname, options);
	parse_options(options, &baud, &dtr, &rts);

	if (strcmp(protname, "none") == 0) {
		return -1;
	}


 	/* open mouse */

	mindex = find_mouse(protname);

	MOUSE_HOOK(vis)->type = mice_types[mindex];

	if ((MOUSE_FD(vis) = do_mouse_open(vis, devname, 
		dtr, rts, baud)) < 0) {

		fprintf(stderr, "L/mouse: Couldn't open mouse.\n");
		return -1;
	}

	
	DPRINT("L/mouse: init OK.\n");

	return 0;
}


void mouse_exit(ggi_visual *vis)
{
	if (MOUSE_FD(vis) >= 0) {

		do_mouse_close(vis);

		/* free mouse private structure */

		free(MOUSE_PRIV(vis));

	}

	DPRINT("L/mouse: exit OK.\n");
}


void mouse_handle_data(ggi_visual *vis)
{
	int read_len;


	/* read the mouse data */

	read_len = MAX_PACKET_BUF - MOUSE_HOOK(vis)->packet_len - 1;

	/* ASSERT(read_len >= 1) */

	read_len = read(MOUSE_FD(vis), MOUSE_HOOK(vis)->packet_buf
			+ MOUSE_HOOK(vis)->packet_len, read_len);

	if (read_len < 1) {
		DPRINT("L/mouse: Error reading mouse.\n");
		return;
	}

	MOUSE_HOOK(vis)->packet_len += read_len;
	

	/* parse any packets */

	while (MOUSE_HOOK(vis)->packet_len >= 
		MOUSE_HOOK(vis)->type->minimum_packet_len) {

		int used;

		used = do_parse_packet(vis);

		if (used <= 0) {
			break;	 /* not enough data yet */
		}

		MOUSE_HOOK(vis)->packet_len -= used;

		if (MOUSE_HOOK(vis)->packet_len > 0) {
		
			memmove(MOUSE_HOOK(vis)->packet_buf,
				MOUSE_HOOK(vis)->packet_buf + used,
				MOUSE_HOOK(vis)->packet_len);
		} else {
			MOUSE_HOOK(vis)->packet_len = 0;
		}
	}

	DPRINT3("mouse_handle_data() done.\n");
}
