/* 
 *   pageflip.c
 *
 *   by Steve Cheng <steve@ggi-project.org>
 *
 * This is a demonstration of LibGGI's functions and can be used as a
 * reference programming example.
 *
 *   This program is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 *
 *   Usage: ./pageflip [mode-spec]
 *
 *   Example: ./pageflip F2
 *   Example: ./pageflip 320x200F8[8]
 *
 *   Test new multiple buffering functions of libGGI.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ggi/ggi.h>

ggi_color black  = { 0x0000, 0x0000, 0x0000 };
ggi_color white  = { 0xffff, 0xffff, 0xffff };
ggi_color blue   = { 0x0000, 0x0000, 0xffff };
ggi_color yellow = { 0xffff, 0xffff, 0x0000 };

/* Palette helper function
 */
static void setup_palette(ggi_visual_t vis)
{
	ggi_mode mode;

	int depth, numcols;

	ggi_color palette[256];

	ggiGetMode(vis, &mode);

	depth = GT_DEPTH(mode.graphtype);
	numcols = 1 << depth;

	if (depth < 3) {

		palette[0] = black;
		palette[1] = white;
		palette[2] = blue;
		palette[3] = yellow;

	} else {
		int i;

		int rnum = (depth+1) / 3;
		int gnum = (depth+2) / 3;
		int bnum = (depth)   / 3;

		int rmask = (1 << rnum) - 1;
		int gmask = (1 << gnum) - 1;
		int bmask = (1 << bnum) - 1;

		for (i=0; i < numcols; i++) {

			int j=i, r, g, b;
			
			b = j & bmask;  j >>= bnum;
			g = j & gmask;  j >>= gnum;
			r = j & rmask;

			palette[i].r = r * 0xffff / rmask;
			palette[i].g = g * 0xffff / gmask;
			palette[i].b = b * 0xffff / bmask;
		}
        }

        ggiSetPalette(vis, 0, numcols, palette);
}

int main(int argc, char *argv[])
{
	ggi_visual_t	vis;
	ggi_mode	mode;
	int		i, j, cx, cy, c;
	char		buf[80];

	/* Set up the random number generator */
	srandom(time(NULL));

	/* Initialize libGGI */
	if(ggiInit()) {
		fprintf(stderr, "Cannot initialize libGGI!\n");
		return EXIT_FAILURE;
	}
	
	/* Open default visual */
	vis = ggiOpen(NULL);
	if(!vis) {
		fprintf(stderr, "Cannot open default visual!\n");
		ggiExit();
		return EXIT_FAILURE;
	}
	
	/* Set visual to async mode (drawing not immediate) */
	ggiSetFlags(vis, GGIFLAG_ASYNC);

	/* Set default mode, but with multiple buffering */
	if (argc>1) {
		ggiParseMode(argv[1], &mode);
	} else {
		ggiParseMode("", &mode);
		mode.frames = 2;
	}

	if (ggiSetMode(vis, &mode)) {
		fprintf(stderr, "Cannot set mode!\n");
		ggiClose(vis);
		ggiExit();
		return EXIT_FAILURE;
	}

	ggiGetCharSize(vis, &cx, &cy);
	
	/* Setup palette */
	if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
		setup_palette(vis);
	}

	/* Write something into each frame */
	for(i = 0; i < mode.frames; i++) {

		if(ggiSetWriteFrame(vis, i)) {
			fprintf(stderr, "Cannot set write frame!\n");
			ggiClose(vis);
			ggiExit();
			return EXIT_FAILURE;
		}
		
		ggiSetGCBackground(vis, ggiMapColor(vis, &black));
		ggiSetGCForeground(vis, ggiMapColor(vis, &black));
		ggiFillscreen(vis);

		sprintf(buf, "Hello World #%d!", i);

		for (j=0; j < mode.visible.y; j += cy) {

			ggi_color col;

			int x = random() % mode.visible.x;

			int h = (random() & 0x7fff) + 0x8000;
			int l = (random() & 0x7fff);

			/* Use different colors for different frames */
			col.r = ((i+1) & 1) ? h : l;
			col.g = ((i+1) & 2) ? h : l;
			col.b = ((i+1) & 4) ? h : l;

			ggiSetGCForeground(vis, ggiMapColor(vis, &col));
			ggiPuts(vis, x, j, buf);
		}
	}	

	/* Cycle through frames */
	i = 0;
	do {
		if(ggiSetDisplayFrame(vis, i)) {
			fprintf(stderr, "Cannot set display frame!\n");
			ggiClose(vis);
			ggiExit();
			return EXIT_FAILURE;
		}

		/* Wait */
		do {
			c = ggiGetc(vis);
		} while (c==GIIK_VOID || GII_KTYP(c)==GII_KT_SHIFT);

		i = (i+1) % mode.frames;

	} while (c != 'q' && c != 'Q' && c != 'x' && c != 'X' && 
		 c != GIIUC_Escape);
	
	ggiClose(vis);
	ggiExit();
	return EXIT_SUCCESS;
}
