/* $Id: db.c,v 1.17 1998/09/20 21:21:52 marcus Exp $
***************************************************************************

   DirectBuffer handling.

   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]
  
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <string.h>
#include <stdlib.h>
#include <ggi/internal/internal.h>

/* Internal functions */

/* _ggi_db_get_new
   Returns a pointer to a newly allocated ggi_directbuffer structure.
*/
ggi_directbuffer *_ggi_db_get_new(void)
{
	ggi_directbuffer *db = _ggi_malloc(sizeof(ggi_directbuffer));
	
	db->frame = 0;
	db->type = GGI_DB_NORMAL;
	db->read = db->write = NULL;
	db->page_size = 0;
	
	return db;
}


/* _ggi_db_free
   Frees a ggi_directbuffer obtained from _ggi_db_get_new.
*/
void _ggi_db_free(ggi_directbuffer *db)
{
	LIBGGI_ASSERT(db != NULL, "_ggi_db_free: db is NULL");
	
	free(db);
}


/* _ggi_db_add_buffer
   Adds a buffer at the end of the list.
   Returns the position of the buffer (starting with 0).
*/
int _ggi_db_add_buffer(ggi_db_list *dbl, ggi_directbuffer *buf)
{
	LIBGGI_ASSERT(dbl != NULL, "_ggi_db_add_buffer: list is NULL");
	LIBGGI_ASSERT(dbl != NULL, "_ggi_db_add_buffer: buffer is NULL");

	dbl->num++;
	dbl->bufs = _ggi_realloc(dbl->bufs, sizeof(ggi_directbuffer *) * dbl->num);
	dbl->bufs[dbl->num-1] = buf;

	return dbl->num-1;
}


/* _ggi_db_del_buffer
   Deletes the buffer at position idx from the list.
   Returns the number of buffers left in the list.
*/
int _ggi_db_del_buffer(ggi_db_list *dbl, int idx)
{
	LIBGGI_ASSERT(dbl != NULL, "_ggi_db_del_buffer: list is NULL");
	LIBGGI_ASSERT(dbl->num > 0, "_ggi_db_del_buffer: called for empty list");

	dbl->num--;
	memmove(dbl->bufs+idx, dbl->bufs+idx+1, (dbl->num-idx)*sizeof(ggi_directbuffer));
	if (dbl->num == 0) {
		free(dbl->bufs);
		dbl->bufs = NULL;
	} else {
		dbl->bufs = _ggi_realloc(dbl->bufs, sizeof(ggi_directbuffer *) * dbl->num);
	}
	
	return dbl->num;
}

/* _ggi_db_move_buffer
   Moves the buffer at position idx in src to the end of dst.
   Returns the position of the buffer in dst.
*/
int _ggi_db_move_buffer(ggi_db_list *dst, ggi_db_list *src, int idx)
{
	int pos = _ggi_db_add_buffer(dst, src->bufs[idx]);
	_ggi_db_del_buffer(src, idx);

	return pos;
}


ggi_directbuffer *_ggi_db_find_frame(ggi_visual *vis, int frameno)
{
	int i;

	for (i=0; i < LIBGGI_APPLIST(vis)->num; i++) {
		ggi_directbuffer *cur = LIBGGI_APPBUFS(vis)[i];

		if ((cur->type & GGI_DB_NORMAL) && (cur->frame == frameno)) {
			return cur;
		}
	}

	for (i=0; i < LIBGGI_PRIVLIST(vis)->num; i++) {
		ggi_directbuffer *cur = LIBGGI_PRIVBUFS(vis)[i];

		if ((cur->type & GGI_DB_NORMAL) && (cur->frame == frameno)) {
			return cur;
		}
	}

	return NULL;  /* not found */
}


/************** DirectBuffer calls ****************/

int  ggiDBGetNumBuffers(ggi_visual *vis)
{
	return LIBGGI_APPLIST(vis)->num;
}

const ggi_directbuffer *ggiDBGetBuffer(ggi_visual *vis, int bufnum)
{
	if (bufnum >= 0 && bufnum < LIBGGI_APPLIST(vis)->num)
		return LIBGGI_APPLIST(vis)->bufs[bufnum];
	return NULL;
}
