/*
 * Thermostat driver for Dallas Semiconductor DS1620 Digital Thermometer
 * running on NetWinder by Corel Computer Corp
 * Woody 1998 woody@corelcomputer.com
 */
/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/config.h>

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include <linux/types.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <linux/proc_fs.h>
#include <asm/segment.h>
//#include <unistd.h>
#include <linux/sched.h>
#include <linux/miscdevice.h>

#define VERSION$ "0.08"
/*****************************************************************************/
#include "therm.h"

#define	THERM_USE_PROC

#define	WRITE_TH	0x01
#define	READ_TH		0xA1
#define	WRITE_TL	0x02
#define	READ_TL		0xA2
#define	WRITE_CONFIG	0x0C
#define	READ_CONFIG	0xAC
#define	START_CONVERT	0xEE
#define	READ_TEMPERATURE 0xAA
#define	SW_RESET	0xAF	//undocumented command - magic reset

extern int get_CPLD(void);
extern int set_CPLD(int bitset, int bitmask);

static int	open_therm(struct inode* inodep, struct file* filep);
static void	release_therm(struct inode* inodep, struct file* filep);
static int	therm_ioctl(struct inode* inodep, struct file* filep, unsigned int cmd, unsigned long arg);
static int      therm_read(struct inode *inode, struct file *file, char *buf, int count);
static void	therm_out(int cmd, int data, int bits);
static int	therm_in(int cmd, int bits);
static void	delay10ms(void);
static void	delay1us(void);
static void	set_reset(int therm_state);
static void	set_io(int io);
static void	start_therm(void);

static const char therm_name[] = "temperature";

static struct file_operations therm_fops = {
	NULL,			/* lseek              */
	therm_read,		/* read               */
	NULL,			/* write              */
	NULL,			/* no special readdir            */
	NULL,			/* no special select             */
	therm_ioctl,
	NULL,			/* no special mmap               */
	open_therm,
	release_therm,
	NULL,			/* no special fsync              */
	NULL,			/* no special fasync             */
	NULL,			/* no special check_media_change */
	NULL			/* no special revaldate          */
};


//pull down reset line, while waiting...
void    delay_rst(int ticks)
{
    outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
    set_reset(0);	//toggle reset line
    
    //wait ticks*10 ms
    current->state = TASK_INTERRUPTIBLE;
    current->timeout = jiffies + ticks;
    schedule();
    
    set_reset(1);
}

void	delay10ms(void)
{
    //wait good 10 ms
    delay_rst(2);
}

void	delay1us(void)
{
    //wait 1 us
    inb(0x80);
}

#ifdef THERM_USE_PROC /* don't waste space if unused */
/*
 * The proc filesystem: function to read and entry
 */

int therm_read_procmem(char *buf, char **start, off_t offset,
                   int len, int unused)
{
    int i,j,k;
    
    outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
    set_reset(0);	//toggle reset line
    set_reset(1);
    delay1us();

    k = therm_in(READ_TH,9);
    if (k & 0x100)
    {
        k+=0xFFFFFE00;	//if negative - convert to int
    }

    outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
    set_reset(0);	//toggle reset line
    set_reset(1);
    delay1us();

    i = therm_in(READ_TL,9);
    if (i & 0x100)
    {
        i+=0xFFFFFE00;	//if negative - convert to int
    }

    outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
    set_reset(0);		//toggle reset line
    set_reset(1);
    delay1us();

    j = therm_in(READ_TEMPERATURE,9);
    if (j & 0x100)
    {
        j+=0xFFFFFE00;	//if negative - convert to int
    }
    
    len=0;
    len += sprintf(buf+len,"\nThermostat: HI %i.%i, LO %i.%i; temperature: %i.%i C, fan %s.\n",
	k>>1, (k&1)?5:0, i>>1, (i&1)?5:0, j>>1, (j&1)?5:0 ,
	(inb(0x338) & 0x04) ? "ON":"OFF");
    
    return len;
}

struct proc_dir_entry therm_proc_entry = {
        0,			/* low_ino: the inode -- dynamic */
        5, "therm",		/* len of name and name */
        S_IFREG | S_IRUGO,	/* mode */
        1, 0, 0,		/* nlinks, owner, group */
        0, NULL,		/* size - unused; operations -- use default */
        &therm_read_procmem,	/* function used to read data */
        /* nothing more */
    };

#endif /* THERM_USE_PROC */


static int therm_read(struct inode *inode, struct file *file, char *buf, int count)
{
	int i;
	unsigned char cp;
	int err;
	

	err=verify_area(VERIFY_WRITE, buf, 1);
	if(err)
		return err;

        outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
	set_reset(0);		//toggle reset line
	set_reset(1);
	delay1us();

	i = therm_in(READ_TEMPERATURE,9);

	i>>=1;

        if (i & 0x80)
	{
	    i+=0xFFFFFF00;	//if negative - convert to int
	}

        i*=11;
	i/=15;
	cp=i+7;
	memcpy_tofs(buf,&cp,1);
	return 1;
}


void	therm_out(int cmd, int data, int bits)
{
int j;
unsigned char current338;
unsigned char current33A;


// we assume that the RESET bit is already set HI by the setup_sa.c module
// same way the GP22 and GP15 bits are programmed as outputs
// the clock is GP22 == BIT2 on port 0x33A, data is GP15 == Bit5 on 0x338

//there should be an access lock to read the current setting of port 0x338, normaly
// set to 0x06 = GreenLed+FAN, and during write to yellow = 0x80+0x02+0x04

//clock cycle is assumed by a falling clock followed by a rising edge
//data can consist of either 8 or 9 bits - specified in bits variable

//	printk(KERN_DEBUG "therm: cmd=0x%02X, data=0x%02X, bits=%d.\n",cmd, data, bits);
	

// do the 8-bit cmd sequence
	current338 = inb(0x338) & 0xDF;	//clear 0x20
        current33A = inb(0x33A) & 0xFB;	//clear 0x04
			
	for(j=0; j<8;j++)
	{
		if (cmd & 0x01)
			outb(0x20 | current338,0x338);//GP15 bit set
		else
			outb(0x00 | current338,0x338);//GP15 bit clear

		outb(current33A,0x33A);	//clear clock
		
		inb(0x80);		//do a 2 us delay...
		inb(0x80);
		inb(0x80);
		
		outb(current33A | 0x04,0x33A);	//set clock

		cmd>>=1;		//shift right by one
	}


//now do the 8-bit or 9-bit data write sequence, if specified

	if(bits)
	{			
            for(j=0; j<bits;j++)
            {
                    if (data & 0x01)
                            outb(0x20 | current338,0x338);//GP15 bit set
                    else
                            outb(0x00 | current338,0x338);//GP15 bit clear
    
                    outb(current33A,0x33A);	//clear clock
                    
                    inb(0x80);
                    inb(0x80);
                    
                    outb(current33A | 0x04,0x33A);	//set clock
    
                    data>>=1;		//shift right by one
            }
	}
	inb(0x80);			//wait a miocrosecond
	set_reset(0);			//toggle the reset to write command
	set_reset(1);
		
	outb(current338, 0x338);	//restore the 338 port and exit
	delay10ms();			//give the device time to write to E2E
}


int	therm_in(int cmd, int bits)
{
int i,j,iTemp,iMask;
unsigned char current338;
unsigned char current33A;

// we assume that the RESET bit is aleready set HI by the setup_sa.c module
// same way the GP22 and GP15 bits are programmed as outputs
// the clock is GP22 == BIT2 on port 0x33A, data is GP15 == Bit5 on 0x338

//there should be an access to read the current setting of port 0x338, normaly
// set to 0x02 = GreenLed, and during write to yellow = 0x80+0x02

//clock cycle is assumed by a falling clock followed by a rising edge
//data can consist of either 8 or 9 bits - specified in bits variable

//	printk(KERN_DEBUG "therm: cmd=0x%02X, data=0x%02X, bits=%d.\n",cmd, data, bits);
	

// do the 8-bit cmd sequence
	current338 = inb(0x338) & 0xDF;	//clear 0x20
        current33A = inb(0x33A) & 0xFB;	//clear 0x04
				
	for(j=0; j<8;j++)
	{
		if (cmd & 0x01)
			outb(0x20+current338,0x338);//GP15 bit set
		else
			outb(0x00+current338,0x338);//GP15 bit clear

		outb(current33A,0x33A);	//clear clock
		
		inb(0x80);		//do a 2 us delay...
		inb(0x80);
		inb(0x80);
		
		outb(current33A | 0x04,0x33A);	//set clock

		cmd>>=1;		//shift right by one
	}


//now do the 8-bit or 9-bit data read sequence

	set_io(1);		//set GP15 pin to become an input

	iTemp = 0;
	iMask = 1;
	
	for(j = 0; j < bits; j++)
	{
		outb(current33A,0x33A);	//clock the next bit out

		inb(0x80);		//give it a moment to settle down
				
		i = inb(0x338);		//read bit
		
		if (i & 0x20)
		{
		    iTemp += iMask;	//the bit was HI
//		    printk(KERN_DEBUG "Bit %d HI.\n",j);
		}
//		else
//		    printk(KERN_DEBUG "Bit %d LO.\n",j);
		    
		outb(current33A | 0x04,0x33A);	//set clock
		iMask <<= 1;
		inb(0x80);
	}
	
	set_io(0);		//set GP15 pin to become an output (default)
	
//	inb(0x80);			//wait a miocrosecond
	set_reset(0);			//toggle the reset to write command
	set_reset(1);
		
	outb(current338, 0x338);	//restore the 338 port and exit
//	outb(current33A, 0x33A);	//leave green LED on and exit
	return(iTemp);
}


	
static int open_therm(struct inode* inodep, struct file* filep)
{
	MOD_INC_USE_COUNT;

//	therm_out(WRITE_CONFIG, 0x02, 8);//program for continuous conversion
	
	return 0;
}


static void release_therm(struct inode* inodep, struct file* filep)
{
	MOD_DEC_USE_COUNT;
}


static int therm_ioctl(struct inode* inodep, struct file* filep, unsigned int cmd, unsigned long arg)
{
int	i;
struct	THERM_SETTING	sTherm;
int	bits,mask,oldCPLD;


//DS1620 internally operates in half-degree numbers, so all readings/settings
//need to be divided/multiplied by 2.
//For negative numbers, hi bit has to be copied to all upper bits, to convert
// signed char into signed 32-bit int

//	printk(KERN_DEBUG "therm_ioctl: cmd = 0x%X, arg = 0x%X.\n",cmd, *(int*)arg);

	switch (cmd)
	{

//these 4 IOCTL calls have nothing to do with therm device
//it is just convenient to stick it in, as therm is supposed to be always on...	
		case CMD_WRITE_CPLD:
			//expect 1 integer - bits, we need to create mask=0x0E
			//to prevent touching flash_writeable bit (bit 0)
			memcpy_fromfs(&i, (void*)arg, sizeof(i));
			oldCPLD = get_CPLD();	//get current setting
			mask = 0x7FFFFFFE;	//w/o top bit to be affected
			bits = i & 0x7FFFFFFE;	//7 bits to be set
			set_CPLD(bits, mask);	
//			printk(KERN_DEBUG "Kernel: Old CPLD 0x%X, new CPLD 0x%X.\n",
//			    oldCPLD, get_CPLD());
			break;
		    

		case CMD_READ_CPLD:
			//return 1 integer
			i=get_CPLD();	//the HIGH bit is already stripped
//			printk(KERN_DEBUG "Kernel: Current CPLD 0x%X.\n",i);
			memcpy_tofs((void*)arg, &i, sizeof(i));
			break;
		    

		case CMD_GET_LED:
			//return 1 integer
			i=getLED();
			memcpy_tofs((void*)arg, &i, sizeof(i));
			break;
		    

		case CMD_SET_LED:
			//expect 1 integer - LED state
			// (0=off, 1=green, 2=red, 3=yellow)
			memcpy_fromfs(&i, (void*)arg, sizeof(i));
			setLED(i);	
			break;
		    

		case CMD_SET_THERMOSTATE:
		case CMD_SET_THERMOSTATE2:
			if (cmd==CMD_SET_THERMOSTATE2)
			{
			//expect 2 integers - Therm.HI and Therm.LO
			memcpy_fromfs(&sTherm, (void*)arg,
				 sizeof(sTherm));
			}
			else
			{
			//expect 1 integer - Therm.HI. Build Therm.LO
			memcpy_fromfs(&sTherm.hi, (void*)arg,
				 sizeof(sTherm.hi));
			sTherm.lo = sTherm.hi - 3;	//make it 3 degree lower
	                }
			
			outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
			set_reset(0);		//toggle reset line
			set_reset(1);
			delay1us();
			
			therm_out(WRITE_CONFIG, 0x02, 8);//program for continuous conversion
			delay10ms();
			
			therm_out(WRITE_TH, (sTherm.hi<<1),9);
			delay10ms();
			
			therm_out(WRITE_TL, (sTherm.lo<<1),9);
			delay10ms();
			
			therm_out(START_CONVERT, 0, 0); 
			delay10ms();
			
// 			printk(KERN_DEBUG "therm: Setting thermostat HI to %d C.\n",sTherm.hi);
			break;

		case CMD_GET_TEMPERATURE:
		case CMD_GET_TEMPERATURE2:

			outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
			set_reset(0);		//toggle reset line
			set_reset(1);
			delay1us();

			i = therm_in(READ_TEMPERATURE,9);

			if (cmd==CMD_GET_TEMPERATURE2)	//report in HALF degrees
			{
				if (i & 0x100)
				{
				    i+=0xFFFFFE00;	//if negative - convert to int
				}
			}
			else
			{
				i>>=1;
				if (i & 0x80)
				{
				    i+=0xFFFFFF00;	//if negative - convert to int
				}
			}			
//			printk(KERN_DEBUG "therm: Reporting temperature %d C.\n",i);
			memcpy_tofs((void*)arg, &i, sizeof(i));
			break;

		case CMD_GET_THERMOSTATE:
		case CMD_GET_THERMOSTATE2:

			outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
			set_reset(0);		//toggle reset line
			set_reset(1);
			delay1us();

			i = therm_in(READ_TH,9);
			i>>=1;					
			if (i & 0x80)
			{
			    i+=0xFFFFFF00;	//if negative - convert to int
			}
			sTherm.hi=i;
			
			i = therm_in(READ_TL,9);
			i>>=1;					
			if (i & 0x80)
			{
			    i+=0xFFFFFF00;	//if negative - convert to int
			}
			sTherm.lo=i;
			
//			printk(KERN_DEBUG "therm: Reporting thermostat HI %d, LO %d.\n",
//			    sTherm.hi, sTherm.lo);

			if (cmd==CMD_GET_THERMOSTATE2)
				memcpy_tofs((void*)arg, &sTherm, sizeof(sTherm));
			else
				memcpy_tofs((void*)arg, &sTherm.hi, sizeof(sTherm.hi));
			break;

		case CMD_GET_STATUS:
// status:
// bit 7: DONE - conversion ready
// bit 6: THF - temperature high flag(cleared by power down or SET_THERMOSTATE)
// bit 5: TLF - temperature down flag
// bit 4: NVB - device busy updating the non-volatile memory
// bit 3: 1
// bit 2: 0
// bit 1: CPU - our operation mode, OK for continuous temp measurements
// bit 0: 1SHOT - one-shot mode (we do not use it)

			outb(inb(0x33A) | 0x04, 0x33A);	//start from setting clock HI
			set_reset(0);		//toggle reset line
			set_reset(1);
			delay1us();

			i = therm_in(READ_CONFIG,8);
//			i &= 0xE3;		//mask out undefined bits
								
//			printk(KERN_DEBUG "therm: Reporting status %d.\n",i);
			memcpy_tofs((void*)arg, &i, sizeof(i));
			break;

		case CMD_GET_FAN:
			i = (inb(0x338) & 0x04)?1:0;
//			printk(KERN_DEBUG "therm: Reporting fan status %d.\n",i);
			memcpy_tofs((void*)arg, &i, sizeof(i));
			break;

		case CMD_SET_FAN:
		
			memcpy_fromfs(&i, (void*)arg, sizeof(i));
//			printk(KERN_DEBUG "SET_FAN: cmd=%X.\n",i);
			if (i)		
			    outb(inb(0x338) | 0x04,0x338);
			else
			    outb(inb(0x338) & ~0x04, 0x338);
		
			break;

		default:
			return -EINVAL;
	}

	return 0;
}



// do not save 338 and 33A state variables, as always Therm_in or Therm_out
// will do it...

void	set_reset(int therm_state)
{
	//we want to write a bit pattern X110 to Xilinx, where X - therm_state
	set_CPLD(therm_state<<3, 0x08);	
}


//set GP15 pin to default OUTPUT (io=0) or input (io=1) to read config/temperature
void	set_io(int io)
{
 	// open up the SuperIO chip
 	outb(0x87,0x370);
 	outb(0x87,0x370);

 	outb(0x07,0x370);
 	outb(0x07,0x371);	// aux function group 1 (Logical Device 7)

 	outb(0xE5,0x370);
 	outb((io & 0x01),0x371);// set GP15 for output = 0 or input = 1

 	// and close up the EFER gate
 	outb(0xAA,0x370);
}


void	start_therm(void)
{
	delay10ms();
	therm_out(SW_RESET, 0, 0); 	//always reset device first

        delay_rst(6);
		
	therm_out(WRITE_CONFIG, 0x02, 8);//program for continuous conversion
	delay10ms();
			
	therm_out(START_CONVERT, 0, 0); 
        delay_rst(10);
}

static struct miscdevice therm_miscdev=
{
	TEMPERATURE_MINOR,
	"temperature",
	&therm_fops
};

void therm_init(void)
{
    int i,j;
    
    start_therm();

    misc_register(&therm_miscdev);
#ifdef THERM_USE_PROC
    /* add proc entry if required */
    /* give more time for the device to initialize */
    proc_register_dynamic(&proc_root, &therm_proc_entry);
#endif

    //wait 250ms till before first reading. DONE bit does not work in this mode
    delay_rst(25);		//give the device time to read temperature
    i = therm_in(READ_TEMPERATURE,9);

    //if the first reading (usually after power-up) is 0.0, retry it after 150ms
    if (!i)
    {
        delay_rst(15);
        i = therm_in(READ_TEMPERATURE,9);
    }

    if (i & 0x100)
    {
        i+=0xFFFFFE00;	//if negative - convert to int
    }

    printk(KERN_NOTICE "Therm driver version %s, current temperature %d.%d C, fan %s.\n",
           VERSION$, i>>1, (i&1)?5:0,
	   (inb(0x338) & 0x04) ? "ON":"OFF");


}

#ifdef MODULE
int init_module(void)
{
	therm_init();
	return 0;
}

void cleanup_module(void)
{
#ifdef THERM_USE_PROC
        proc_unregister(&proc_root, therm_proc_entry.low_ino);
#endif
	misc_deregister(&therm_miscdev);
}
#endif
