/*C* -*-c++-*-
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#ifndef util_Collection_h
#define util_Collection_h

#include "debug.h"
#include "File.h"
#include "Object.h"
#include <assert.h>
#include <stdlib.h>

//--------------------------------------------------------------------------------------------------------------------------------

#define foreach(c,i) for(int i=0; i < ((c).length()); i++)
#define every(c,op) for(int i=0; i < ((c).length()); i++) (c).at(i)->op;

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> class Collection : public Object
{
private:
 T** data;
 int space, delta;

 size_t dataSize() { return sizeof(T*) * space; }
 void grow();

protected:
 int pLength;

 int length(int _length) { return pLength = _length; }

public:
 Collection(int _space = 64, int _delta = 64);
 Collection(Collection& object);
 virtual ~Collection();

 int length() { return pLength; }

 inline T* at(int n)
  {
   assert(n >= 0 && n <= length());
   return data[n];
  }
 inline T* operator [] (int n) { return at(n); }

 T* add(class T* object);

 inline T* remove(int index) { T* ret = data[index]; data[index] = 0; return ret; }
 inline void removeAll() { pLength = 0; }
 void pack();

 void del(int index) { delete at(index); remove(index); }
 void delAll();
};

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> void Collection<T>::grow()
{
 space += delta;
 data = (T**) realloc(data, dataSize());
 if(!data) fatal("cannot realloc(%d) in Collection::Grow()\n", dataSize());
}

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> Collection<T>::Collection(int _space, int _delta)
{
 assert(_space > 0); assert(_delta > 0);
 space = _space;
 delta = _delta;
 pLength = 0;
 VPRINTF("<collection> allocating collection space=%d delta=%d\n", space, delta);
 data = (T**) malloc(dataSize());
};

template <class T> Collection<T>::Collection(Collection& object)
 : Object(object)
{
 space = object.space;
 delta = object.delta;
 pLength = 0;
 VPRINTF("<collection> duplicating collection space=%d delta=%d\n", space, delta);
 data = (T**) malloc(dataSize());
}

template <class T> Collection<T>::~Collection()
{
 delAll();
 free(data);
};

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> T* Collection<T>::add(T* object)
{
 if(length() == space) grow();
 return data[pLength++] = object;
}

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> void Collection<T>::pack()
{
 int dc = 0;
 for(int sc=0; sc<length(); sc++)
  if(data[sc]) data[dc++] = data[sc];
 length(dc);
}

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> void Collection<T>::delAll()
{
 for(int i=0; i<length(); i++)
  del(i);
 pack();
}

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> class SCollection : public Collection<T>
{
public:
 SCollection(int _space = 64, int _delta = 64) : Collection<T>(_space, _delta) { };
 bool read(File& f);
 bool write(File& f);
 static SCollection* create(File& f);
};

//--------------------------------------------------------------------------------------------------------------------------------

template <class T> bool SCollection<T>::read(File& f)
{
 delAll();
 int l;
 if(!f.read(l)) return false;
 VPRINTF("<collection> reading collection of length %d\n", l);
 pLength = 0;
 for(int n=0; n<l; n++)
  {
   T* s = T::create(f);
   if(!s) { setError("%d out of %d records read: %s", n, l, errStr); return false; }
   add(s);
  }
 VPRINTF("<collection> all %d objects read succesfully\n", pLength);
 return true;
}

template <class T> bool SCollection<T>::write(File& f)
{
 if(!f.write(pLength)) return false;
 for(int i=0; i<pLength; i++)
  if(!at(i)->write(f)) return false;
 return true;
}

template <class T> static SCollection<T>* create(class File& f)
{
 SCollection<T>* sc = new SCollection<T>;
 if(!sc->read(f)) { delete sc; return 0; }
 return sc;
}

//--------------------------------------------------------------------------------------------------------------------------------

#endif
