/*C*
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "Console.h"
#include "Rect.h"
#include "Sprite.h"
#include "VgaContext.h"
#include "alphaTable.h"
#include "palette.h"
#include "../util/Collection.h"
#include "../util/debug.h"
#include "../util/error.h"
#include "../util/File.h"
#include "../util/String.h"
#include "../util/types.h"
#include "../util/util.h"
#include <assert.h>
#include <string.h>

extern "C"
{
#include <jpeglib.h>
}

//--------------------------------------------------------------------------------------------------------------------------------

template class Collection<Sprite>;
template class SCollection<Sprite>;

//--------------------------------------------------------------------------------------------------------------------------------

Sprite::Sprite()
{
 initAlphaTable();
 data32 = NULL;
 data8 = NULL;
}

Sprite::Sprite(Sprite& object)
{
 data32 = NULL;
 data8 = NULL;
 if(object.data32)
  {
   z = object.z;
   alloc();
   memcpy(data32[0], object.data32[0], z.x * z.y * 4);
  }
}

Sprite::~Sprite()
{
 free();
}

//--------------------------------------------------------------------------------------------------------------------------------

void Sprite::alloc()
{
 free();
 data32 = new uint8* [z.y];
 unsigned char* p = new uint8 [z.x * z.y * 4];
 for(int i = 0; i<z.y; i++, p += z.x * 4) data32[i] = p;
}

void Sprite::free()
{
 if(data32)
  {
   delete data32[0];
   delete data32;
   data32 = NULL;
  }
 free8();
}

//--------------------------------------------------------------------------------------------------------------------------------

void Sprite::alloc8()
{
 assert(z.y > 0);
 free8();
 data8 = new uint8* [z.y];
 unsigned char* p = new uint8 [z.x * z.y];
 for(int i = 0; i<z.y; i++, p += z.x) data8[i] = p;
}

void Sprite::free8()
{
 if(data8)
  {
   delete data8[0];
   delete data8;
   data8 = NULL;
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

Point Sprite::size(Point newSize)
{
 if(!data32)
  {
   z = newSize;
   alloc();
   return z;
  }
 else
  {
   // keep the old data (oldData32, oldSize) and make the new one
   uint8** oldData32 = data32;
   Point oldSize = z;
   data32 = NULL;
   z = newSize;
   alloc();

   // resize it
   // FIXME this probably needs tweaking to be technically perfect
   for(int y=0; y<z.y; y++)
    {
     if(y < oldSize.y)
      {
       if(z.x <= oldSize.x)
	{
	 memcpy(data32[y], oldData32[y], z.x * 3);
	 memcpy(data32[y] + z.x * 3, oldData32[y] + oldSize.x * 3, z.x);
	}
       else
	{
	 memset(data32[y], 0, z.x * 4);
	 memcpy(data32[y], oldData32[y], oldSize.x * 3);
	 memcpy(data32[y] + z.x * 3, oldData32[y] + oldSize.x * 3, oldSize.x);
	}
      }
     else
      memset(data32[y], 0, z.x * 4);
    }
   delete oldData32[0];
   delete oldData32;
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

Color Sprite::setPixel(Point p, Color c)
{
 assert(z.contains(p));
 uint8* dest = data32[p.y] + p.x * 3; 
 *dest++ = c.b;
 *dest++ = c.g;
 *dest++ = c.r;
 return c;
}

int Sprite::setPixel(Point p, int mask)
{
 return *(data32[p.y] + z.x * 3 + p.x) = mask;
}

void Sprite::setPixel(Point p, Color c, int mask)
{
 uint8* dest = data32[p.y] + p.x * 3; 
 *dest++ = c.b;
 *dest++ = c.g;
 *dest++ = c.r;
 *(data32[p.y] + z.x * 3 + p.x) = mask;
}

//--------------------------------------------------------------------------------------------------------------------------------

bool Sprite::read(File& f)
{
 if(!z.read(f)) return false;
 VPRINTF("<spr> reading sprite with dimensions: %d x %d\n", z.x, z.y);
 alloc();
 if(!f.read(data32[0], z.x * z.y * 4)) { free(); return false; }
 return true;
}

bool Sprite::write(File& f)
{
 if(!data32) { setError("no data"); return false; }
 if(!z.write(f)) return false;
 if(!f.write(data32[0], z.x * z.y * 4)) return false;
 return true;
}

//--------------------------------------------------------------------------------------------------------------------------------

typedef unsigned char uchar;
typedef unsigned short ushort;
typedef unsigned long ulong;
typedef signed char schar;
typedef signed short sshort;
typedef signed long slong;

#define PK __attribute__ ((packed));
struct TgaHeader
{
 uchar idSize PK;
 uchar mapType PK;
 uchar imageType PK;
 ushort cmapOrigin PK;
 ushort cmapLength PK;
 uchar cmapEntrySize PK;
 sshort xOrigin PK;
 sshort yOrigin PK;
 ushort width PK;
 ushort height PK;
 uchar pixelSize PK;
 uchar descriptor PK;
};
#undef PK

bool Sprite::readTga(File& f)
{
 free();

 TgaHeader header;
 if(f.read(&header, 18) < 1) return false;

 VPRINTF("<spr> tga: size=%dx%d imageType=%d pixelSize=%d descriptor=0x%2X\n", header.width, header.height, header.imageType,
	 header.pixelSize, header.descriptor);

 // figure out what type of tga this is
 int tgaFormat = 0;
 if(header.imageType == 2 && header.pixelSize == 24 && (header.descriptor & 0x0F) == 0x00) tgaFormat = 1; // no alpha channel
 if(header.imageType == 2 && header.pixelSize == 32 && (header.descriptor & 0x0F) == 0x08) tgaFormat = 2; // 8bit alpha channel
 if(tgaFormat == 0) { setError("unsupported tga format: %d", header.imageType); return false; }

 // set this object
 z = Point(header.width, header.height);
 alloc();

 // seek past any id data, read it to a buffer in case the file is a pipe
 if(header.idSize > 0)
  {
   VPRINTF("<spr> skipping %db of header\n", header.idSize);
   uchar idBuffer[header.idSize];
   if(f.read(idBuffer, header.idSize) < 1) { free(); return false; }
  }

 // read the data
 for(int y=0; y<z.y; y++)
  switch(tgaFormat)
   {
   case 1:
    if(f.read(data32[y], z.x * 3) < 1) { free(); return false; }
    memset(data32[y] + z.x * 3, 255, z.x);
    break;

   case 2:
    {
     uint8 lineBuffer[z.x * 4];
     if(f.read(lineBuffer, z.x * 4) < 1) { free(); return false; }
     uint8 *s = lineBuffer, *d = data32[y], *m = &data32[y][z.x * 3];
     for(int x=0; x<z.x; x++)
      {
       *d++ = *s++;
       *d++ = *s++;
       *d++ = *s++;
       *m++ = *s++;
      }
    }
    break;
   }

 // flip the sprite if it is stored bottom to top
 if((header.descriptor & 0x20) == 0x00) flipV();

 return true;
}

bool Sprite::writeTga(File& f)
{
 if(!data32) { setError("no data"); return false; }

 // Create and write header
 TgaHeader header;
 header.idSize = 0;
 header.mapType = 0;
 header.imageType = 2;
 header.cmapOrigin = 0;
 header.cmapLength = 0;
 header.cmapEntrySize = 0;
 header.xOrigin = 0;
 header.yOrigin = 0;
 header.width = z.x;
 header.height = z.y;
 header.pixelSize = 24;
 header.descriptor = 0x20; // indicates top to bottom storage
 if(f.write(&header, 18) != 1) return false;

 // Write image data
 for(int y=0; y<z.y; y++)
  if(!f.write(data32[y], z.x * 3)) return false;

 return true;
}

//--------------------------------------------------------------------------------------------------------------------------------

bool Sprite::readJpeg(File& f)
{
 jpeg_decompress_struct cinfo;
 jpeg_error_mgr jerr;

 // initialise jpeg library
 cinfo.err = jpeg_std_error(&jerr);
 jpeg_create_decompress(&cinfo);
 jpeg_stdio_src(&cinfo, (FILE*) f);

 // read jpeg header
 jpeg_read_header(&cinfo, TRUE);

 // set up decompression
 cinfo.out_color_space = JCS_RGB;
 cinfo.output_components = 3;
 jpeg_calc_output_dimensions(&cinfo);

 // allocate image data
 z = Point(cinfo.output_width, cinfo.output_height);
 VPRINTF("<spr> reading jpeg with dimensions: %d x %d\n", z.x, z.y);
 alloc();

 // read image data
 jpeg_start_decompress(&cinfo);
 for(int y=0; y<z.y;)
  y += jpeg_read_scanlines(&cinfo, &data32[y], z.y - y);
 swapRB();

 // set an opaque mask
 for(int y=0; y<z.y; y++)
  memset(data32[y] + z.x * 3, 0xFF, z.x);

 jpeg_finish_decompress(&cinfo);
 jpeg_destroy_decompress(&cinfo);
 return true;
}

bool Sprite::writeJpeg(File& f)
{
 setError("unimplemented");
 return false;
}

//--------------------------------------------------------------------------------------------------------------------------------

void Sprite::flipH()
{
 fatal("injured function\n");
 /*
 if(!data) return;
 freeCompiled();
 for(int x = 0; x < z.x/2; x++) for(int y = 0; y < z.y; y++)
  swap(data[y][x], data[y][z.x - x - 1]);
 */
}

void Sprite::flipV()
{
 if(!data32) return;
 for(int y = 0; y < z.y/2; y++) for(int x = 0; x < z.x * 4; x++)
  swap(data32[y][x], data32[z.y - y - 1][x]);
}

void Sprite::rotate()
{
 fatal("injured function\n");
 /*
 if(!ok || !data) return;
 freeCompiled();
 unsigned char** temp;
 alloc(temp);
 for(int x = 0; x < z.x; x++) for(int y = 0; y < z.y; y++)
  temp[y][x] = data[z.x - x - 1][y];
 free();
 data = temp;
 */
}

void Sprite::dim(int factor)
{
 assert(factor >= 0 && factor <= 255);
 if(!data32) return;
 for(int y=0; y<z.y; y++)
  {
   unsigned char* p = data32[y];
   for(int x=0; x<z.x * 3; x++)
    *p++ = *p * factor / 255;
  }
}

void Sprite::swapRB()
{
 if(!data32) return;
 for(int y=0; y<z.y; y++)
  {
   unsigned char* p = data32[y];
   for(int x=0; x<z.x; x++)
    {
     swap(p[0], p[2]);
     p += 3;
    }
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

void Sprite::dither()
{
 // FIXME this could be far more efficient
 alloc8();
 int8 vErrorR[z.x], vErrorG[z.x], vErrorB[z.x];
 for(int x=0; x<z.x; x++) vErrorR[x] = vErrorG[x] = vErrorB[x] = 0;
 for(int y=0; y<z.y; y++)
  {
   const uint8* s = data32[y];
   const uint8* m = data32[y] + z.x * 3;
   uint8* d = data8[y];
   int8 hErrorR = 0, hErrorG = 0, hErrorB = 0;
   for(int x=0; x<z.x; x++)
    {
     // set 0 for completely transparent bits
     if(*m < 2) { s += 3; m++; *d++ = 0; continue; }

     int b = (int)hErrorB + (int)vErrorB[x] + (int)*s++;
     int g = (int)hErrorG + (int)vErrorG[x] + (int)*s++;
     int r = (int)hErrorR + (int)vErrorR[x] + (int)*s++;

     Color wanted;
     if(b < 0) wanted.b = 0; else if(b > 0xFF) wanted.b = 0xFF; else wanted.b = b;
     if(g < 0) wanted.g = 0; else if(g > 0xFF) wanted.g = 0xFF; else wanted.g = g;
     if(r < 0) wanted.r = 0; else if(r > 0xFF) wanted.r = 0xFF; else wanted.r = r;

     Color got = pal2rgb(*d++ = rgb2pal(wanted));

     hErrorB = vErrorB[x] = (b - (int)got.b) >> 1;
     hErrorG = vErrorG[x] = (g - (int)got.g) >> 1;
     hErrorR = vErrorR[x] = (r - (int)got.r) >> 1;

     m++;
    }
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

Rect Sprite::draw(VgaContext* vc, Point p)
{
 if(!data32) return emptyRect;
 Rect r = Rect(p, z);

 if(vc->clipping)
  {
   if(!r.meets(vc->clippingRect)) return emptyRect;
   r.clipTo(vc->clippingRect);
  }

 switch(bpp)
  {

  case 8:
   if(!data8) dither();
   for(int y = r.top(); y <= r.bottom(); y++)
    {
     unsigned char* s = &data8[y - p.y][r.x - p.x];
     unsigned char* d = &vc->fb()[y][r.x];
     for(int x = 0; x < r.w; x++)
      {
       if(*s) *d = *s;
       d++; s++;
      }
    }
   break;

  case 16:
   for(int cy = 0; cy < r.h; cy++)
    {
     unsigned char* s = data32[r.y - p.y + cy] + 3 * (r.x - p.x);
     unsigned char* m = data32[r.y - p.y + cy] + z.x * 3 + (r.x - p.x);
     unsigned short* d = (unsigned short*) &vc->fb()[r.y + cy][2 * r.x];
     for(int cx = 0; cx < r.w; cx++)
      {
       unsigned char b = alphaTable[255 - *m][(*d & 0x001F) << 3] + alphaTable[*m][*s++];
       unsigned char g = alphaTable[255 - *m][(*d & 0x07E0) >> 3] + alphaTable[*m][*s++];
       unsigned char r = alphaTable[255 - *m][(*d & 0xF800) >> 8] + alphaTable[*m][*s++];
       *d++ = b >> 3 | (g & 0xFC) << 3 | (r & 0xF8) << 8;
       m++;
      }
    }
   break;

  case 24:
   for(int y = r.top(); y <= r.bottom(); y++)
    {
     unsigned char* s = &data32[y - p.y][(r.x - p.x) * 3];
     unsigned char* m = &data32[y - p.y][(z.x * 3) + (r.x - p.x)];
     unsigned char* d = &vc->fb()[y][r.x * 3];
     for(int cx = 0; cx < r.w; cx++)
      {
       *d++ = alphaTable[255 - *m][*d] + alphaTable[*m][*s++];
       *d++ = alphaTable[255 - *m][*d] + alphaTable[*m][*s++];
       *d++ = alphaTable[255 - *m][*d] + alphaTable[*m][*s++];
       m++;
      }
    }
   break;

  case 32:
   for(int cy = 0; cy < r.h; cy++)
    {
     unsigned char* s = data32[r.y - p.y + cy] + 3 * (r.x - p.x);
     unsigned char* m = data32[r.y - p.y + cy] + z.x * 3 + (r.x - p.x);
     unsigned char* d = &vc->fb()[r.y + cy][4 * r.x];
     for(int cx = 0; cx < r.w; cx++)
      {
       *d++ = alphaTable[255 - *m][*d] + alphaTable[*m][*s++];
       *d++ = alphaTable[255 - *m][*d] + alphaTable[*m][*s++];
       *d++ = alphaTable[255 - *m][*d] + alphaTable[*m][*s++];
       m++; d++;
      }
    }
   break;
  }
 return r;
}

//--------------------------------------------------------------------------------------------------------------------------------

void Sprite::setMask(Sprite& s)
{
 if(z != s.z) fatal("Sprite::setMask() - passed sprite size mismatch");
 unsigned char* dst = data32[0] + z.x * 3;
 unsigned char* src = s.data32[0];
 for(int y = 0; y < z.y; y++)
  {
   for(int x = 0; x < z.x; x++)
    *dst++ = (*src++ + *src++ + *src++) / 3;
   dst += z.x * 3;
   src += z.x;
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

objectCreateFunc Sprite::getCreateFunc(const char* filename)
{
 String s = String(filename);
 if(s.endsin(".jpg") || s.endsin(".jpeg")) return (objectCreateFunc) Sprite::createJpeg;
 if(s.endsin(".tga") || s.endsin(".targa")) return (objectCreateFunc) Sprite::createTga;
 return (objectCreateFunc) Sprite::create;
}

Sprite* Sprite::create(File& f)
{
 Sprite* s = new Sprite;
 if(!s->read(f)) { delete s; return NULL; }
 return s;
}

Sprite* Sprite::createTga(File& f)
{
 Sprite* s = new Sprite;
 if(!s->readTga(f)) { delete s; return NULL; }
 return s;
}

Sprite* Sprite::createJpeg(File& f)
{
 Sprite* s = new Sprite;
 if(!s->readJpeg(f)) { delete s; return NULL; }
 return s;
}

//--------------------------------------------------------------------------------------------------------------------------------

Sprite emptySprite;

//--------------------------------------------------------------------------------------------------------------------------------
