// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ececs.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Swaminathan Subramanian ssubrama@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_ProcessStatement.cc,v 1.5 1999/07/23 21:08:09 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_ProcessStatement.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_FileDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_IndexedName.hh"
#include "IIR_Label.hh"
#include "IIR_ProcedureCallStatement.hh"
#include "IIR_SelectedName.hh"
#include "IIR_SelectedNameByAll.hh"
#include "IIR_SimpleName.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_SliceName.hh"
#include "IIR_VariableDeclaration.hh"
#include "IIR_WaitStatement.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "StandardPackage.hh"

extern symbol_table *cgen_sym_tab_ptr;


IIRScram_ProcessStatement::IIRScram_ProcessStatement() { }


IIRScram_ProcessStatement::~IIRScram_ProcessStatement() { }


void 
IIRScram_ProcessStatement::_type_check(){
  process_declarative_part._type_check_attribute_specifications( process_statement_part ); 
}


void 
IIRScram_ProcessStatement::_publish_vhdl(ostream &_vhdl_out) {
  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ": ";
  }
    
  if (get_postponed() == true) {
    _vhdl_out << "postponed ";
  }
    
  _vhdl_out << "process\n";
  process_declarative_part._publish_vhdl_decl(_vhdl_out);
  _vhdl_out << "\nbegin\n";
  process_statement_part._publish_vhdl(_vhdl_out);
  _vhdl_out << "end ";
  if (get_postponed() == true) {
    _vhdl_out << "postponed ";
  }
  _vhdl_out << "process";
  if (get_label() != NULL) {
    _vhdl_out << " ";
    get_label()->_publish_vhdl(_vhdl_out);
  }
  _vhdl_out << ";\n\n";
}


void 
IIRScram_ProcessStatement::_publish_cc_file_open() {
  IIR_Declaration *decl;
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    ASSERT(decl->_is_read_file() || decl->_is_write_file());

    if (decl->_is_read_file()) {
      _cc_out << "((";
      _get_declarator()->_publish_cc();
      _cc_out << "_state *) state->current)->";
      
      ((IIR_FileDeclaration*)decl)->_get_declarator()->_publish_cc();
      _cc_out << ".openFile(this, ";
      ((IIR_FileDeclaration*)decl)->get_file_logical_name()->_publish_cc();
      _cc_out << ", READ_MODE);\n"; 
    }
    else if (decl->_is_write_file()) {
      _cc_out << "((";
      _get_declarator()->_publish_cc();
      _cc_out << "_state *) state->current)->";
      ((IIR_FileDeclaration*)decl)->_get_declarator()->_publish_cc();
      _cc_out << ".openFile(this, ";
      ((IIR_FileDeclaration*)decl)->get_file_logical_name()->_publish_cc();
      _cc_out << ", WRITE_MODE";

      if (decl->_get_declarative_region()->get_kind() == IIR_ARCHITECTURE_DECLARATION) {
	_cc_out << ", ios::app";
      }
      
      _cc_out << ");\n";
    }
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
}


void 
IIRScram_ProcessStatement::_publish_cc() {
  IIR_Declaration *decl = NULL;
  IIR_WaitStatement *wait = NULL;
  int numInFiles = 0, numOutFiles = 0;
  char *procname = NULL;
  ostrstream ent_arch;
  IIR_Char* design_unit_name = NULL;
  symbol_table cgen_sym_tab( 4093, false );
  IIR_Boolean first = TRUE;
  
  cgen_sym_tab_ptr = &cgen_sym_tab;
  
  ent_arch << _current_entity_name;
  _set_current_process_statement((IIR_ProcessStatement*)this);
  // This check is needed as process stmnts in entity decls have no arch
  if (_current_architecture_name != NULL) {
    ent_arch << "_" << _current_architecture_name;
    _publish_cc_enclosing_stmt_to_architecture_path(ent_arch);
  }
  
  ent_arch << ends;
  design_unit_name = ent_arch.str();
  _cc_out.set_file(TRUE, design_unit_name, "_decls.hh");

  strstream labelstr;
  ASSERT(_get_label() != NULL);	// Since we are mangling it before this point.
  _get_label()->_print(labelstr);
  labelstr << ends;
  procname = labelstr.str();

  // Publish the required stuff in the declarative part of the process.
  _cc_out.set_file(TRUE, procname, "_decls.hh");
  _cc_out << "#ifndef " << procname << "_DECLS_HH\n";
  _cc_out << "#define " << procname << "_DECLS_HH\n";
  // Include the declarative region to which this process belongs
  _cc_out << "#include \"" << design_unit_name << "_decls.hh\"\n";

  // Include all the stuff in the declarative part of process in this file
  process_declarative_part._publish_cc();

  _cc_out.set_file(TRUE, procname, "_decls.hh");
  _cc_out << "#endif\n";

  delete [] design_unit_name;

  _cc_out.set_file(TRUE, procname, "_decls.cc");
  _cc_out << "#include \"" << _current_entity_name << "_decls.hh\"" 
	  << endl << endl;
  _cc_out << "#include \"";
  _cc_out << *_get_label();
  _cc_out << ".hh\"\n";
  process_declarative_part._publish_cc_decl();
  
  _build_wait_list();

  // generate code for the <process>.hh file
  _cc_out.set_file(TRUE, procname, ".hh");
  _cc_out << "#ifndef " << procname << "_HH\n";
  _cc_out << "#define " << procname << "_HH\n";

  _cc_out << "#include \"VHDLKernel.hh\"\n";
  _cc_out << "#include \"" << procname << "_state.hh\"\n\n";

  //redefine savantnow so that any initializations through savant now is 
  //performed
  _cc_out << "#ifdef savantnow" << endl;
  _cc_out << "#undef savantnow" << endl;
  _cc_out << "#define savantnow (PhysicalType(ObjectBase::VARIABLE, "
	  << "UniversalLongLongInteger((getTimeNow().time)), "
	  << "SavanttimeType_info)) " << endl;
  _cc_out << "#endif" << endl;
  
  // This has been commented out as we need to use SB or SEA
  // depending on the situation. This info is now available in
  // _current_publish_name which is set in the architecture or block
  // statements.
  
  // _cc_out << "class " << "SEA"<< _current_entity_name << "_";
  _cc_out << "class " << _current_publish_name << _current_entity_name;

  // This if check is needed as process stmts in entity decls will not have
  // arch set for them.
  if (_current_architecture_name != NULL) {
    _cc_out << "_" << _current_architecture_name;
    _publish_cc_enclosing_stmt_to_architecture_path();
  }
  
  _cc_out << "_elab;\n";
  
  _publish_cc_extern_declarations();
  _cc_out << "class " << procname << " : public VHDLKernel {\n";
  _cc_out << "public:\n";
  _cc_out << "  " << procname << "( " << _current_publish_name;
  _cc_out << _current_entity_name;

  if (_current_architecture_name != NULL) {
    _cc_out << "_"<< _current_architecture_name;
    _publish_cc_enclosing_stmt_to_architecture_path();
  }
  
  _cc_out << "_elab* proc );\n";
  _cc_out << "  ~" << procname << "();\n\n";
  _cc_out << "  void executeVHDL();\n";
  _cc_out << "  BasicState* allocateState();\n";
  _cc_out << "  void initialize();\n";
  // End the .hh file later after searching for guard signal

  // generate code for the <process>.cc file
  _cc_out.set_file(TRUE, procname, ".cc");

  _cc_out << "#include \"" << procname << ".hh\"\n\n";
  _cc_out << "\n\n";
  _publish_cc_extern_type_info();
  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(PROCESS);

  // executeVHDL
  _cc_out << "void " << procname << "::executeVHDL() {\n";
  _cc_out << "#ifdef VHDLDBG\n";
  _cc_out << "  cout << \"" << procname 
	  << " executing VHDL at \" << getLVT() << endl;\n";
  _cc_out << "#endif\n\n";

  for (wait = _wait_stmt_list.first(); 
       wait != NULL; 
       wait = _wait_stmt_list.successor(wait))  {
    _cc_out << "  if (((VHDLKernel_state *) state->current)->waitLabel == " << wait->wait_id 
	    << ") goto ";
    wait->_publish_cc_wait_label();
    _cc_out << ";\n";
  }

  IIR_ProcedureCallStatement *proc;
  dl_list<IIR_ProcedureCallStatement> _proc_stmt_list;
  process_statement_part._build_procedure_call_stmt_list(&_proc_stmt_list);
  // "goto"s for procedure call statements in the process.
  _cc_out << "if(((VHDLKernel_state *) state->current)->waitLabel == WAITING_IN_PROC) {\n"
	  << "  ((VHDLKernel_state *) state->current)->stack.setCurrentToTop();\n";
  for (proc = _proc_stmt_list.first(); proc != NULL; 
       proc = _proc_stmt_list.successor(proc))  {
    _cc_out << "if(((VHDLKernel_state *) state->current)->stack.getCurrentTop()->waitLabel == "
	    << proc << ") goto ";
    proc->_publish_cc_proc_label();
    _cc_out << ";\n";
  } // for
  _cc_out << "}\n\n";


  _cc_out << "\nwhile (true) {\n";

  ostrstream prefixStream;
  char *prefixStr;
  prefixStream << "((";
  _get_declarator()->_print(prefixStream);
  prefixStream << "_state *) state->current)->" << ends;
  prefixStr = prefixStream.str();

  _set_publish_prefix_string(prefixStr);
  process_statement_part._publish_cc();
  _set_publish_prefix_string(NULL);
  
  _cc_out << "} // end primary while loop\n";
  _cc_out << "} // end execute_VHDL\n\n\n";
  
  // This piece of code is very tricky it has to come only after 
  // the publish_cc is called on the process statement part
  IIR_Declaration *guardSignal = NULL;
  if (IIRScram_BlockStatement::guard_expr != NULL) {
    // This means an implicit guard signal exists in this scope
    // If the signal is used in this process then get it
    guardSignal = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
    while (guardSignal != NULL) {
      if (!IIRScram_TextLiteral::_cmp(guardSignal->get_declarator(), "guard")) {
	break;
      }
      guardSignal = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(guardSignal);
    }
  }

  //  -----** This is done for guard signals
  _cc_out.set_file(TRUE, procname, ".hh");
  if (guardSignal != NULL) {
    // This means the process is using the implicit guard signal
    _cc_out << "  VHDLType *getGuard();\n";
    _cc_out << "  VHDLType *getGuardExpression();\n";
  }

  // Publish the constants declared and used in this process here.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PROCESS_STATEMENT:
      decl->_publish_cc_decl();
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  _cc_out << "};\n\n";

  _cc_out << "#endif\n";
  //  ------**

  _set_currently_publishing_unit(_saved_publishing_unit);

  // Back to publishing code in the cc file
  _cc_out.set_file(TRUE, procname, ".cc");
  // <process> destructor
  _cc_out << procname << "::~" << procname << "() {}\n\n";

  // find out the number of things declared & used in this scope.
  // WARNING!  Code gets ugly here!

  // <process> constructor
  
  // This has been commented out as we need to use SB or SEA
  // depending on the situation. This info is now available in
  // _current_publish_name which is set in the architecture or block
  // statements.
 
  // _cc_out << procname << "::" << procname << "( SEA";
  _cc_out << procname << "::" << procname << "( " << _current_publish_name;
  _cc_out << _current_entity_name;

  // This check is needed as process stmnts in entity decl have no arch
  if (_current_architecture_name != NULL)  {
    _cc_out << "_" << _current_architecture_name;
    _publish_cc_enclosing_stmt_to_architecture_path();
  }
  
  _cc_out << "_elab* ptr) ";

  // See if some constant intializations have to be publised. If so, do them.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  first = TRUE;
  while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PROCESS_STATEMENT:
      if (first == TRUE) {
	_cc_out << " : \n";
	first = FALSE;
      }
      else {
	_cc_out << ",\n";
      }
      
      decl->_publish_cc_constant_object_init();
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  _cc_out << "{\n  id = globalobjectid++;\n";
  _cc_out << "  proc = ptr;\n";
  _cc_out << "  name = \"" << procname << "\";\n";
  _cc_out << "// ((VHDLKernel_state *) state->current)->proc = ptr;\n";

  process_declarative_part._publish_cc_global_constants_assignments();
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    ASSERT(decl->_is_read_file() || decl->_is_write_file());

    if (decl->_is_read_file()) {
      numInFiles++;
    }
    else if (decl->_is_write_file()) {
      numOutFiles++;
    }

    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }

  if (numInFiles >= 0) {
    _cc_out << "  numInFiles = 0;\n";
    _cc_out << "  inFileQ = new InFileQueue[" << (numInFiles) << " + maxExtraInFilesInProcedure ];\n";
  }
  if (numOutFiles >= 0) {
    _cc_out << "  numOutFiles = 0;\n";
    _cc_out << "  outFileQ = new FileQueue[" << (numOutFiles + 1) << " + maxExtraOutFilesInProcedure ];\n";
  }

  // All the files are actually opened in initialize() function.

  _cc_out << "}\n\n";

  _cc_out << "BasicState*"
	  << procname << "::allocateState()  {\n"
	  << "  proc->copyGenericsToGlobals();\n"
	  << "  return new " << procname << "_state;\n"
	  << "}\n\n";

  _cc_out << "void\n" << procname << "::initialize()  {\n";
  _publish_cc_file_open();
  _cc_out << "  VHDLKernel::initialize();\n";
  _cc_out << "}\n\n";
  
  _set_publish_prefix_string(prefixStr);
  if (guardSignal != NULL) {
    // This means the implicit guard signal is used in this process
    _cc_out << "VHDLType*\n" << procname << "::getGuard() {\n" << "  return &(";
    guardSignal->_publish_cc();
    _cc_out <<");\n}\n";
      
    _cc_out << "VHDLType*\n"  << procname << "::getGuardExpression() {\n"
	    <<"  return new EnumerationType(ObjectBase::SIGNAL, ";
    IIRScram_BlockStatement::guard_expr->_publish_cc();
    _cc_out << ", SavantbooleanType_info);\n}";
  }
  _set_publish_prefix_string(NULL);
  
  _publish_cc_state();


  cgen_sym_tab_ptr = NULL;
  _set_current_process_statement(NULL);
  delete [] procname;
  delete [] prefixStr;
}


void
IIRScram_ProcessStatement::_publish_cc_state() {
  char *procname;

  //Reset the code-generation attributes set during publishing any of the
  //previously encountered process statements
  _clear();

  //   cgen_sym_tab_ptr = &cgen_sym_tab;

  //All objects to be defined in _sate.hh file should not
  //be printed with prefix "state.current->"
  //So setting the flag so all object and Interface declarations
  //print as per the value of that flag
  IIR_Boolean tmp_value = _get_publish_object_without_prefix();
  _set_publish_object_without_prefix(TRUE);

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  _cc_out.set_file(TRUE, procname, "_state.hh");

  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(PROCESS_STATE);

  //The following function adds the declarations in initializations of
  //visible declarations
  // This routine is not ok -- Probably we can remove it altogether
  _add_declarations_in_initializations();

  _publish_cc_headers();
  _publish_cc_state_class();

  _cc_out.set_file(TRUE, procname, "_state.cc");

  _cc_out << "#include \"" << procname << "_state.hh\"\n";

  //redefine savantnow so that any initializations through savant now is 
  //performed
  _cc_out << "#ifdef savantnow" << endl;
  _cc_out << "#undef savantnow" << endl;
  _cc_out << "#define savantnow (PhysicalType(ObjectBase::VARIABLE, "
	  << "UiversalLongLongInteger((this->lVT.time)), "
	  << "SavanttimeType_info)) " << endl;
  _cc_out << "#endif" << endl;

  _publish_cc_constructor();
  _publish_cc_initstate();
  _publish_cc_cleanstate();
  _publish_cc_locatesignal();
  // These methods are no longer necessary in the state.
  //   _publish_cc_ostream_operator();
  //   _publish_cc_print();
  _publish_cc_operator_equalto();

  //publish Serialize and deserialize that are needed for OFC
  _publish_cc_serialize_state(procname);
  _publish_cc_deserialize_state(procname);
  
  //Resetting the flag value to the original value
  _set_publish_object_without_prefix(tmp_value);
  _set_currently_publishing_unit(_saved_publishing_unit);
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_headers() {
  char *procname;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  _cc_out << "#ifndef " << procname << "_STATE_HH\n"
	  << "#define " << procname << "_STATE_HH\n\n"
	  << "#include \"VHDLKernel_state.hh\"\n"
	  << "#include \"SignalNetinfo.hh\"\n"
	  << "#include \"Wait.hh\"\n"
	  << "#include \"AccessVariable.hh\"\n"

    // This has been commented out as we need to use SB or
    // SEA depending on the situation. This info is now available in
    // _current_publish_name which is set in the architecture or block
    // statements.
    
    // << "#include \"SEA"
	  << "#include \"" << _current_publish_name
	  << _current_entity_name;

  // This check is needed as processes in entity decl have no arch.
  if (_current_architecture_name != NULL)  {
    _cc_out << "_" << _current_architecture_name;
    _publish_cc_enclosing_stmt_to_architecture_path();
  }
  
  _cc_out << "_elab.hh\"\n"
	  << "#include \"" << procname << "_decls.hh\"\n";
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_file_declarations() {
  IIR_Declaration *decl;
  int numifiles = 0;
  int numofiles = 0;

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    if (((IIR_FileDeclaration*)decl)->_is_read_file()) {
      numifiles++;
      _cc_out << "const int ";
      ((IIR_Identifier *) ((IIR_FileDeclaration *) decl)->_get_declarator())->_publish_cc();
      _cc_out << "_" << decl;
      _cc_out << " = " << numifiles-1 << ";\n";
    }
    else if (((IIR_FileDeclaration*)decl)->_is_write_file()) {
      numofiles++;
      _cc_out << "const int ";
      ((IIR_Identifier*)((IIR_FileDeclaration*)decl)->_get_declarator())->_publish_cc();
      _cc_out << "_" << decl;
      _cc_out << " = " << numofiles-1 << ";\n";
    }
    else {
      cerr << "_publish_process_stmt::_publish_cc ERROR!" 
	   << "  File is neither read nor write!\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
}

void 
IIRScram_ProcessStatement::_publish_cc_binding_name( ostream &_cc_out ){
  if( _get_label() != NULL ) {
    _get_label()->_publish_cc_elaborate();
  }
  else {
    _cc_out << "ANON_PROCESS" << this;
  }
}



void
IIRScram_ProcessStatement::_publish_cc_extern_declarations() {
  IIR_Declaration *decl;

  _cc_out << "extern int globalobjectid;\n";
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    if (((IIR_FileDeclaration*)decl)->_is_read_file()) {
      _cc_out << "extern const int ";
      ((IIR_Identifier *) ((IIR_FileDeclaration *) decl)->_get_declarator())->_publish_cc();
      _cc_out << "_" << decl;
      _cc_out << ";\n";
    }
    else if (((IIR_FileDeclaration*)decl)->_is_write_file()) {
      _cc_out << "extern const int ";
      ((IIR_Identifier*)((IIR_FileDeclaration*)decl)->_get_declarator())->_publish_cc();
      _cc_out << "_" << decl;
      _cc_out << ";\n";
    }
    else {
      cerr << "_publish_process_stmt::_publish_cc ERROR!" 
	   << "  File is neither read nor write!\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  _cc_out << "extern const int maxExtraInFilesInProcedure;\n";
  _cc_out << "extern const int maxExtraOutFilesInProcedure;\n";
}


void
IIRScram_ProcessStatement::_publish_cc_read_or_write(const char *functionName,
						     const char *streamName) {
  IIR_Declaration *decl;
  // Publish copy operator for constants in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
    case IIR_PROCESS_STATEMENT:
      break;
    default:
      decl->_publish_cc();
      _cc_out << "." << functionName << "(" << streamName;
      _cc_out << ");\n";
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  // Publish copy operator for signals in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if (decl->_is_implicit_declaration() == FALSE) {
      decl->_get_declarator()->_publish_cc();
      _cc_out << "." << functionName << "(" << streamName;
      _cc_out << ");\n";
      decl->_publish_cc_implicit_signal_attributes_read_or_write(functionName, streamName);
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_get_declarator()->_publish_cc();
    _cc_out << "." << functionName << "(" << streamName;
    _cc_out << ");\n";
    decl->_publish_cc_implicit_signal_attributes_read_or_write(functionName, streamName);
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  
  // Publish copy operator for variables in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    if (decl->_get_subtype()->_is_access_type() == FALSE) {
      decl->_get_declarator()->_publish_cc();
      _cc_out << "." << functionName << "(" << streamName;
      _cc_out << ");\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  
  // Publish copy operator for interface variables in this state - removed.
  
  //   decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  //   while (decl != NULL) {
  //     _cc_out << functionName << "(NULL, " << streamName << ", ";
  //     decl->_get_declarator()->_publish_cc();
  //     _cc_out << ");\n";
  //     decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  //   }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    decl->_publish_cc();
    _cc_out << "." << functionName << "(" << streamName << ");\n";
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }
  
  _cc_out << "\n";
}
						     
void
IIRScram_ProcessStatement::_publish_cc_serialize_state(const char *procName) {
  _cc_out << "void\n";
  _cc_out << procName << "_state::serialize(ofstream *ckFile, int)  {\n";
  _cc_out << "  AccessVariable <char *> outputLine;\n";
  _publish_cc_read_or_write("savantwrite", "outputLine");
  _cc_out << "  *ckFile << outputLine.dataLength << CHECKPOINT_DELIMITER;\n";
  _cc_out << "  ckFile->write(outputLine.val, outputLine.dataLength);\n";
  _cc_out << "  *ckFile << CHECKPOINT_DELIMITER;\n";
  _cc_out << "  delete [] outputLine.val;\n";
  _cc_out << "}\n";
}

void
IIRScram_ProcessStatement::_publish_cc_deserialize_state(const char *procName){
  _cc_out << "void\n";
  _cc_out << procName << "_state::deserialize(ifstream* inFile)  {\n";
  _cc_out << "  AccessVariable <char *> inputLine;\n"
	  << "  int linesize;\n"
	  << "  char dummyChar;\n" << endl;

  _cc_out << "  *inFile >> linesize;\n";
  _cc_out << "  *inFile >> dummyChar;\n";

  _cc_out << "  ASSERT ( dummyChar == CHECKPOINT_DELIMITER );\n";
  
  _cc_out << "  inputLine.val        = new char[linesize];\n"
	  << "  inFile->read(inputLine.val, linesize);\n"
	  << "  *inFile >> dummyChar;\n"
	  << "  ASSERT ( dummyChar == CHECKPOINT_DELIMITER );\n"
	  << "  inputLine.dataLength = linesize;\n"
	  << "  inputLine.curPos     = 0;\n";
  
  _publish_cc_read_or_write("savantread", "inputLine");

  _cc_out << "  delete [] inputLine.val;\n";
  
  _cc_out << "}\n";
}

void
IIRScram_ProcessStatement::_clear() {

  IIR_Declaration *decl;

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    decl->_clear();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_clear();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
}


void
IIRScram_ProcessStatement::_publish_cc_state_class() {
  IIR_Declaration *decl;
  char *procname;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  ///##End line add
  _cc_out << "\n";
  _publish_cc_extern_type_info();
   

  _cc_out << "class " << procname << "_state : public VHDLKernel_state {\n";
//   _cc_out << "  friend ostream& operator<<(ostream&, const "
// 	  << procname << "_state&);\n\n";
  _cc_out << "public:\n"
	  << "  " << procname << "_state();\n"
	  << "  ~" << procname << "_state() {}\n"
	  << "  void initState(_savant_entity_elab *, VHDLKernelBase*);\n"
	  << "  void cleanState();\n"
	  << "  " << procname << "_state& operator=(" << procname 
	  << "_state&);\n";
  
  _cc_out << "  void copyState(BasicState *rhs)  {" << endl
	  << "    *this = *((" << procname << "_state *) rhs);" << endl
	  << "  }" << endl << endl;

  _cc_out << "  void serialize(ofstream* ckFile, int);\n"
	  << "  void deserialize(ifstream *inFile);\n" << endl;
  
  //   _cc_out << "  void print(ostream&);\n";
  _cc_out << "  SignalBase* locateSig(int);\n\n";

  //Constants are made to publish first because,
  //other types of objects can be initialized using these constants
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
   while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
    case IIR_PROCESS_STATEMENT:
      break;
    default:
      decl->_publish_cc_decl();
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_implicit_declaration()) {
      IIR_Declaration* decl_prefix;
      decl_prefix = decl->_get_signal_prefix();
      if(!cgen_sym_tab_ptr->in_scope(decl_prefix)) {
	cgen_sym_tab_ptr->add_declaration(decl_prefix);
      }
    }
    else {
      _cc_out << "  ";
      decl->_publish_cc_decl();
      //This function publishes all the declarations that are in scope.
      decl->_publish_cc_implicit_signal_attributes();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl();
    //This function publishes all the declarations that are in scope.
     decl->_publish_cc_implicit_signal_attributes();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }

  // interface variables are no longer handled here - removed code

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  
  _cc_out << "\n";

  _cc_out << "  const int numWaits;\n";

  //      << "  Wait *wait;\n";
  // _cc_out << _current_publish_name << _current_entity_name;

  // We need to check if _current_architecture_name is NULL or not since
  // process statements in entity decl will not have an arch. In other
  // places i.e. in arch decls, blocks etc, this will be set.
  
  //   if (_current_architecture_name != NULL)  {
  //     _cc_out << "_"  << _current_architecture_name;
  //     _publish_cc_enclosing_stmt_to_architecture_path();
  //   }
  
  //   _cc_out << "_elab* proc;\n"

  _cc_out << "\n  int getSize() const  {\n"
	  << "    return sizeof(*this);\n"
	  << "}\n\n";
  _cc_out << "};\n\n" << "#endif\n";
  delete [] procname;
}

void
IIRScram_ProcessStatement::_publish_cc_constructor() {
  IIR_Declaration *decl;
  char *procname;
  int index;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();


  // generate <process>_state constructor
  _cc_out << procname << "_state::" << procname << "_state():\n";
  set<IIR_Declaration> busSignals;
  _publish_cc_state_objects_init(&busSignals);
  // signal names could go here for human readability--not required.
  _cc_out << "numWaits(" << _wait_stmt_list.num_elements() << ") {\n";
  
  // If the signal is of kind bus then set it 
  decl = busSignals.get_element();
  for (index = 0; index < busSignals.num_elements(); index++) {
    _cc_out << "  ";
    decl->_publish_cc();
    _cc_out <<".setBusKind();\n";
    decl = busSignals.get_next_element();
  }

  _cc_out << "  wait = new Wait[" << _wait_stmt_list.num_elements()
	  << "];\n";
  
  //For a composite Resolved signals, certain data members namely
  // compositeResolvedSignal, and parentCompositeType of the kernel Signal
  //has to be initialized. The following function publishes them.
  _publish_cc_composite_resolved_signal_init();

  _cc_out << "}\n\n\n";
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_composite_resolved_signal_init() {
  IIR_Declaration *decl = NULL;
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_composite_resolved_signal() == TRUE) {
      decl->_publish_cc();
      _cc_out << ".setParentCompositeType(&";
      decl->_publish_cc();
      _cc_out << ");\n";
      decl->_publish_cc();
      _cc_out << ".setCompositeResolvedSignal(true);\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_composite_resolved_signal() == TRUE) {
      decl->_publish_cc();
      _cc_out << ".setParentCompositeType(&";
      decl->_publish_cc();
      _cc_out << ");\n";
      decl->_publish_cc();
      _cc_out << ".setCompositeResolvedSignal(true);\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
}


void
IIRScram_ProcessStatement::_publish_cc_initstate() {
  IIR_Declaration *decl;
  char *procname;
  char *old_current_publish_name = _current_publish_name;
  IIR* oldNode = _current_publish_node;
  
  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  // generate_state::initState

  _cc_out << "void\n" << procname << "_state::initState(_savant_entity_elab *ptr, VHDLKernelBase* processPointer) {\n";
  // We need to check if _current_architecture_name is NULL or not since
  // process statements in entity decl will not have an arch. In other
  // places i.e. in arch decls, blocks etc, this will be set.

  ostrstream elabPtrStream;
  elabPtrStream << _current_publish_name << _current_entity_name;
  if (_current_architecture_name != NULL)  {
    elabPtrStream << "_"  << _current_architecture_name;
    _publish_cc_enclosing_stmt_to_architecture_path(elabPtrStream);
  }
  elabPtrStream << "_elab" << ends;
  
  char *elabString = elabPtrStream.str();
  _cc_out << elabString << "* proc = (" << elabString << " *) ptr;\n";
  delete [] elabString;
  
  _cc_out << "  VHDLKernel_state::initState(ptr, processPointer);\n";

  _current_publish_name = procname;

  // Set the file handles for the files in this process state
  int inputFileHandleCount = 0, outputFileHandleCount = 0;
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT ( decl->get_kind() == IIR_FILE_DECLARATION );
    ASSERT ((decl->_is_read_file() == TRUE) ||
	    (decl->_is_write_file() == TRUE));
    decl->_get_declarator()->_publish_cc();
    _cc_out << ".set_fileHandle(";
    if (decl->_is_read_file() == TRUE) {
      _cc_out << inputFileHandleCount;
      inputFileHandleCount++;
    }
    else {
      ASSERT ( decl->_is_write_file() == TRUE );
      _cc_out << outputFileHandleCount;
      outputFileHandleCount++;
    }
    _cc_out << ");\n";

    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  // Publish initialization for Constants and Constant Interfaces
  _cc_out << "\n";
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    if ( decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_DECLARATION && decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_BODY_DECLARATION ){
      
      decl->_publish_cc_init();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  // Publish intialization for Signals and Signal Interfaces
  // Need this stuff for scoping of signals.
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    _current_publish_node = this->_get_enclosing_scope();
    if(decl->_is_implicit_declaration() == FALSE){
      decl->_publish_cc_init();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    _current_publish_node = this->_get_enclosing_scope();
    if(decl->_is_implicit_declaration() == FALSE){
      decl->_publish_cc_init();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  _current_publish_node = oldNode;
  
  // Publish initialization for Variables and Variable Interfaces
  _cc_out << "\n";
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    decl->_publish_cc_init();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  // interface variables no longer handled here - code removed 

  _cc_out << "\n";
  _publish_cc_wait_init();
  _cc_out << "\n}\n\n\n";

  _current_publish_name = old_current_publish_name;
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_cleanstate() {
  char *procname;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  // generate <process>_state::cleanState
  _cc_out << "void\n";
  _cc_out << procname << "_state::cleanState() {\n";
  _cc_out << "/" << "/" << " Hey!!! cleanup is done in the signal's destructor\n";
  for(int i=0; i < _wait_stmt_list.num_elements(); i++){ 
    _cc_out << "delete [] (char *)wait[" << i <<"].sensList;\n";
  }

  _cc_out << "\n}\n\n\n";
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_locatesignal() {
  IIR_Declaration *decl;
  char *procname;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  // generate <process>_state::locateSig
  _cc_out << "SignalBase*\n";
  _cc_out << procname << "_state::locateSig(int sigId) {\n";
  _cc_out << "SignalBase* ptr = NULL;" << endl;
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    decl->_publish_cc_locatesignal();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_publish_cc_locatesignal();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }

  _cc_out << "\n";

  _cc_out << "    cerr << \"" << procname 
	  << "_state::locateSig ERROR: signal \" << sigId\n";
  _cc_out << "         << \" not in state!\";\n";
  _cc_out << "  abort();\n \n  return ptr; \n}\n\n";
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_ostream_operator() {
  IIR_Declaration *decl;
  char *procname;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  // generate <process>_state operator<<
  _cc_out << "ostream&\n";
  _cc_out << "operator<<(ostream& os, " << procname << "_state& s) {\n";

  // Publish print operator for Constants
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    if ( decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_DECLARATION && decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_BODY_DECLARATION ){
      
      ASSERT(decl->_get_declarator()->get_kind() == IIR_IDENTIFIER);
      _cc_out << "  s.";
      decl->_publish_cc();
      _cc_out << ".print(os);\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }


  // Publish print operator for Signals
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    _cc_out << "  s.";
    decl->_get_declarator()->_publish_cc();
    _cc_out << ".print(os);\n";
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  // Publish print operator for Interface Signals
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    _cc_out << "  s.";
    decl->_get_declarator()->_publish_cc();
    //->get_text();
    _cc_out << ".print(os);\n";
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }

  // Publish print operator for Variables
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    ASSERT(decl->_get_declarator()->get_kind() == IIR_IDENTIFIER);

    _cc_out << "  s.";
    decl->_get_declarator()->_publish_cc();
    _cc_out << ".print(os);\n";
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }

  // Publish print operator for Interface Variables - removed
  _cc_out << "\n";

  // Publish print operator for Interface Constants.
  // This is not necessary since we directly replace the constant with the
  // value of the constant.

  _cc_out << "  for (register int i = 0; i < s.numWaits; i++) {\n";
  _cc_out << "    os << \"Wait \" << i << \": \\n  sensList: \";\n";
  _cc_out << "    s.wait[i].print(os);\n";
  _cc_out << "    os << \"\\ntimeout is: \" << ((s.wait[i].timeout)(&s)).getVTime() << \" fS\\n\";\n";
  _cc_out << "  }\n  return os;\n}\n\n\n";
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_print() {
  char *procname;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  // generate <process>_state::print
  _cc_out << "void\n";
  _cc_out << procname << "_state::print(ostream& os) {\n";
  _cc_out << "  os << *this;\n}\n\n\n";
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_cc_operator_equalto() {
  IIR_Declaration *decl;
  char *procname;

  ostrstream labelstr;
  if (_get_label() != NULL) {
    _get_label()->_print(labelstr);
  }
  else {
    labelstr << "ANON_PROCESS" << this;
  }
  labelstr << ends;
  procname = labelstr.str();

  // generate <process>_state::operator=
  _cc_out << procname << "_state&\n";
  _cc_out << procname << "_state::operator=(" << procname << "_state& s) {\n";
  _cc_out << "  (VHDLKernel_state&)*this = (VHDLKernel_state&)s;\n";

  // Publish copy operator for constants in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
    case IIR_PROCESS_STATEMENT:
      break;
    default:
      _cc_out << "  this->";
      decl->_publish_cc();
      _cc_out << " = s.";
      decl->_publish_cc();
      _cc_out << ";\n";
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  _cc_out << "\n";
  
  // Publish copy operator for signals in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if (decl->_is_implicit_declaration() == FALSE) {
      _cc_out << "  this->";
      decl->_get_declarator()->_publish_cc();
      _cc_out << " = s.";
      decl->_get_declarator()->_publish_cc();
      _cc_out << ";\n";
      decl->_publish_cc_implicit_signal_attributes_copying();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    _cc_out << "  this->";
    decl->_get_declarator()->_publish_cc();
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc();
    _cc_out << ";\n";
    decl->_publish_cc_implicit_signal_attributes_copying();
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  
  // Publish copy operator for variables in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    _cc_out << "  this->";
    decl->_get_declarator()->_publish_cc();
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc();
    _cc_out << ";\n";
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }

  // Publish copy operator for interface variables in this state - removed.

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    _cc_out << "  this->";
    decl->_get_declarator()->_publish_cc();
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc();
    _cc_out << ";\n";
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    _cc_out << "  this->";
    decl->_publish_cc();
    _cc_out << " = s.";
    decl->_publish_cc();
    _cc_out << ";\n";
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    _cc_out << "  this->";
    decl->_get_declarator()->_publish_cc();
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc();
    _cc_out << ";\n";
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  
  // Publish copy operator for interface constants in this state.
  // This is not necessary since we directly replace the constant with the
  // value of the constant.

  _cc_out << "  for (register int i = 0; i < numWaits; i++) wait[i] = s.wait[i];\n";
  _cc_out << "  return *this;\n"
	  << "}\n\n";
  delete [] procname;
}


void
IIRScram_ProcessStatement::_publish_createNetInfo() {
  //### This thing not complete, work needed here
  //  IIR_DeclarationList signallist;
  //  IIR_DesignatorList signallist;
  set<IIR_Declaration> signallist;
  ostrstream objectname;
  _get_list_of_input_signals(&signallist);
  
  objectname << ", ";

  if(_get_label() != NULL) {
    _get_label()->_print(objectname);
  }
  else {
    objectname << "ANON_PROCESS" << this;
  }

  if (_is_currently_publishing_generate_for()) {
    objectname << "_elab_obj[i - generateLeft]->id);\n" << ends;
  }
  else {
    objectname << "_elab_obj->id);\n" << ends;
  }
  
  IIR_Char* ptr = objectname.str();
  IIR* signals = signallist.get_element();
  int numelements = signallist.num_elements();
  
  ASSERT ( _get_enclosing_scope() != NULL );
  int i;
  for( i = 0; i < numelements; i++) { //signals!= NULL) {
    _cc_out << "Add(";
    switch(signals->get_kind()) {
    case IIR_SIGNAL_INTERFACE_DECLARATION:
      ASSERT ( signals->_is_iir_declaration() == TRUE );
      ASSERT( ((IIR_Declaration *) signals)->_get_declarative_region() != NULL );
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *) signals)->_get_declarative_region(),this->_get_enclosing_scope());
      ((IIR_Declaration*)signals)->_publish_cc_elaborate();
      break;
    case IIR_DESIGNATOR_EXPLICIT:
      ASSERT ( signals->_is_iir_declaration() == TRUE );
      ASSERT ( ((IIR_Declaration *) signals)->_get_declarative_region() != NULL );
      _get_enclosing_scope()->_publish_cc_scoping_prefix( ((IIR_Declaration *) signals)->_get_declarative_region(), this->_get_enclosing_scope() );
      ((IIR_Declaration*)((IIR_DesignatorExplicit*)signals)->get_name())->_publish_cc_elaborate();
      break;
    case IIR_INDEXED_NAME:
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_IndexedName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_IndexedName *)signals)->_publish_cc_elaborate();
      break;
    case IIR_SLICE_NAME:
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SliceName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SliceName *)signals)->_publish_cc_elaborate();
      break;
    case IIR_SELECTED_NAME:
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SelectedName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SelectedName *)signals)->_publish_cc_elaborate();
      break;
    case IIR_SIMPLE_NAME:
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SimpleName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SimpleName *)signals)->_publish_cc_elaborate();
      break;
    case IIR_SELECTED_NAME_BY_ALL:
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SelectedNameByAll*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SelectedNameByAll *)signals)->_publish_cc_elaborate();
      break;
    default:
      ASSERT ( signals->_is_iir_declaration() == TRUE );

      if ( ((IIR_Declaration *) signals)->_is_implicit_declaration() != TRUE ) {
	ASSERT ( ((IIR_Declaration *) signals)->_get_declarative_region() != NULL );
	_get_enclosing_scope()->_publish_cc_scoping_prefix( ((IIR_Declaration *) signals)->_get_declarative_region(), this->_get_enclosing_scope() );
      }
      else {
	IIR_Declaration* tempDeclaration = (IIR_Declaration *) ((IIR_SignalDeclaration *) signals)->_get_signal_decl();
	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	ASSERT ( tempDeclaration->_get_declarative_region() != NULL );
	
	_publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
      }
      
      signals->_publish_cc_elaborate();
      break;
    }
    _cc_out << ptr;
    signals = signallist.get_next_element();
  }
  
  set<IIR_Declaration> signalset;
  _get_signal_source_info(&signalset);
  IIR_Char *_saved_current_publish_name = _current_publish_name;
  _current_publish_name = ptr;
  //  IIR_Declaration* sig_decl = signalset.get_element();
  IIR* sig_decl = signalset.get_element();
  while(sig_decl != NULL) {
    if(sig_decl->_is_iir_declaration() == TRUE) {
      sig_decl->_publish_cc_addChild();
    }
    sig_decl = signalset.get_next_element();
  }
  sig_decl = signalset.get_element();
  while(sig_decl != NULL) {
    if(sig_decl->_is_name() == TRUE) {
      sig_decl->_publish_cc_addChild();
    }
    sig_decl = signalset.get_next_element();
  }
  _current_publish_name = _saved_current_publish_name;
  delete [] ptr;
}

void
IIRScram_ProcessStatement::_publish_cc_type_info(){
  process_declarative_part._publish_cc_type_info();
}

void
IIRScram_ProcessStatement::_publish_cc_extern_type_info(){
  process_declarative_part._publish_cc_extern_type_info();
}

#ifdef PROCESS_GRAPH
void
IIRScram_ProcessStatement::_publish_cc_driver_info() {
  set<IIR_Declaration> signallist;
  _get_list_of_input_signals(&signallist);
  IIR* signals = signallist.get_element();
  int numelements = signallist.num_elements();
  int i;
  
  ASSERT ( _get_enclosing_scope() != NULL );
  for( i = 0; i < numelements; i++) { //signals!= NULL) {
    switch(signals->get_kind()) {
    case IIR_SIGNAL_INTERFACE_DECLARATION:
      ASSERT ( signals->_is_iir_declaration() == TRUE );
      ASSERT( ((IIR_Declaration *) signals)->_get_declarative_region() != NULL );
      _cc_out << "fp << \"Signal:";
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *) signals)->_get_declarative_region(),this->_get_enclosing_scope());
      ((IIR_Declaration*)signals)->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *) signals)->_get_declarative_region(),this->_get_enclosing_scope());
      ((IIR_Declaration*)signals)->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
      break;
    case IIR_DESIGNATOR_EXPLICIT:
      ASSERT ( signals->_is_iir_declaration() == TRUE );
      ASSERT ( ((IIR_Declaration *) signals)->_get_declarative_region() != NULL );
      _cc_out << "fp << \"Signal:";
      _get_enclosing_scope()->_publish_cc_scoping_prefix( ((IIR_Declaration *) signals)->_get_declarative_region(), this->_get_enclosing_scope() );
      ((IIR_Declaration*)((IIR_DesignatorExplicit*)signals)->get_name())->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      _get_enclosing_scope()->_publish_cc_scoping_prefix( ((IIR_Declaration *) signals)->_get_declarative_region(), this->_get_enclosing_scope() );
      ((IIR_Declaration*)((IIR_DesignatorExplicit*)signals)->get_name())->_publish_cc_elaborate();
      _cc_out << ".dump_connectitvity_info(fp);" << endl;
      break;
    case IIR_INDEXED_NAME:
      _cc_out << "fp << \"Signal";
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_IndexedName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_IndexedName *)signals)->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_IndexedName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_IndexedName *)signals)->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
      break;
    case IIR_SLICE_NAME:
      _cc_out << "fp << \"Signal";
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SliceName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SliceName *)signals)->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SliceName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SliceName *)signals)->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
      break;
    case IIR_SELECTED_NAME:
      _cc_out << "fp << \"Signal:";
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SelectedName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SelectedName *)signals)->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SelectedName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SelectedName *)signals)->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
      break;
    case IIR_SIMPLE_NAME:
      _cc_out << "fp << \"Signal:";
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SimpleName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SimpleName *)signals)->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SimpleName*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SimpleName *)signals)->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
      break;
    case IIR_SELECTED_NAME_BY_ALL:
      _cc_out << "fp << \"Signal:";
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SelectedNameByAll*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SelectedNameByAll *)signals)->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      _get_enclosing_scope()->_publish_cc_scoping_prefix(((IIR_Declaration *)((IIR_SelectedNameByAll*)signals)->_get_prefix_declaration())->_get_declarative_region(), this->_get_enclosing_scope());
      ((IIR_SelectedNameByAll *)signals)->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
      break;
    default:
      ASSERT ( signals->_is_iir_declaration() == TRUE );
      
      if ( ((IIR_Declaration *) signals)->_is_implicit_declaration() != TRUE ) {
	ASSERT ( ((IIR_Declaration *) signals)->_get_declarative_region() != NULL );
	_cc_out << "fp << \"Signal:";
	_get_enclosing_scope()->_publish_cc_scoping_prefix( ((IIR_Declaration *) signals)->_get_declarative_region(), this->_get_enclosing_scope() );
	signals->_publish_cc_elaborate();
	_cc_out << "\" << endl;" << endl;
	_get_enclosing_scope()->_publish_cc_scoping_prefix( ((IIR_Declaration *) signals)->_get_declarative_region(), this->_get_enclosing_scope() );
	signals->_publish_cc_elaborate();
	_cc_out << ".dump_connectivity_info(fp);" << endl;
      }
      else {
	IIR_Declaration* tempDeclaration = (IIR_Declaration *) ((IIR_SignalDeclaration *) signals)->_get_signal_decl();
	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	ASSERT ( tempDeclaration->_get_declarative_region() != NULL );
	
	_cc_out << "fp << \"Signal:";
	_publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
	signals->_publish_cc_elaborate();
       _cc_out << "\" << endl;" << endl;
	_publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
	signals->_publish_cc_elaborate();
	_cc_out << ".dump_connectivity_info(fp);" << endl;
      }
      break;
    }
    signals = signallist.get_next_element();
  }
  
  set<IIR_Declaration> signalset;
  _get_signal_source_info(&signalset);
  //  IIR_Declaration* sig_decl = signalset.get_element();
  IIR* sig_decl = signalset.get_element();
  while(sig_decl != NULL) {
    if (sig_decl->_is_name() == TRUE ||
	sig_decl->_is_iir_declaration() == TRUE) {
      _cc_out << "fp << \"Signal:";
      sig_decl->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      sig_decl->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
    }
      sig_decl = signalset.get_next_element();
  }
}
#endif

void
IIRScram_ProcessStatement::_get_signal_source_info(set<IIR_Declaration>* siginfo) {
  process_statement_part._get_signal_source_info(siginfo);
}


void
IIRScram_ProcessStatement::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  process_statement_part._get_list_of_input_signals(list);
}


void
IIRScram_ProcessStatement::_publish_cc_implicit_declarations(IIR_Declaration* decl) {
  ASSERT(decl->_is_implicit_declaration() == FALSE);
  if( decl->_get_implicit_declarations() != NULL ){
    IIR_Declaration* implied_declarations = decl->_get_implicit_declarations()->get_element();
    while(implied_declarations != NULL) {
      if(cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	implied_declarations->_publish_cc_decl();
	implied_declarations->_publish_cc_implicit_signal_attributes();
      }
      implied_declarations = decl->_get_implicit_declarations()->get_next_element();
    }
  }
}


void
IIRScram_ProcessStatement::_publish_cc_state_objects_init(set<IIR_Declaration> *busSigs) {
  IIR_Declaration* decl;

  // Publish inititalization for constant
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch(decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
      break;
    default:
      if ((decl->_get_declarative_region()->get_kind() != IIR_PROCESS_STATEMENT) &&
	  (decl->_get_declarative_region()->get_kind() != IIR_ARCHITECTURE_DECLARATION)) {
	decl->_publish_cc_state_object_init();
	_cc_out << "\n";
      }
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  // Publish inititalization for signals
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_implicit_declaration() == FALSE) {
      decl->_publish_cc_state_object_init();
      _cc_out << "\n";
      decl->_publish_cc_implicit_state_objects_init();
      if (decl->get_kind() == IIR_SIGNAL_DECLARATION) {
	if (((IIR_SignalDeclaration*)decl)->get_signal_kind() == IIR_BUS_KIND) {
	  busSigs->add(decl);
	}
      }
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  // Publish inititalization for interface signals
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_implicit_declaration() == FALSE) {
      decl->_publish_cc_state_object_init();
      _cc_out << "\n";
      decl->_publish_cc_implicit_state_objects_init();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }

  // Publish inititalization for variable
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    decl->_publish_cc_state_object_init();
    _cc_out << "\n";      
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }

  // Publish inititalization for alias declarations
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    decl->_publish_cc_state_object_init();
    _cc_out << "\n";      
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    decl->_publish_cc_state_object_init();
    _cc_out << "\n";      
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }
}


void
IIRScram_ProcessStatement::_publish_cc_wait_init() {

  IIR_WaitStatement *wait;
  IIR_Boolean sens_list_newed = FALSE; // Flag used to indicate that the
  // new'ed memory should be deleted.
  for (wait = _wait_stmt_list.first();
       wait != NULL;
       wait = _wait_stmt_list.successor(wait)) {
    if (wait->sensitivity_list.num_elements() == 0) { 
      if(wait->get_condition_clause() != NULL) {
	wait->get_condition_clause()->_build_sensitivity_list(&wait->sensitivity_list);
	sens_list_newed = TRUE;
      }
    }

    if(wait->sensitivity_list.num_elements() != 0) {
      _cc_out << "  wait[" << wait->wait_id << "].sensSize = " 
	      << wait->sensitivity_list.num_elements() << ";\n";
      _cc_out << "  wait[" << wait->wait_id << "].sensList = ";
      _cc_out << "(VHDLType**) new char[sizeof(VHDLType*)*";
      _cc_out << wait->sensitivity_list.num_elements() << "];\n";
      int sig = 0;
      IIR_Declaration *sens_sig;
      IIR_Designator *current_signal_designator;
      current_signal_designator = wait->sensitivity_list.first();
      while (current_signal_designator != NULL) {
	ASSERT(current_signal_designator->get_kind()==IIR_DESIGNATOR_EXPLICIT);
	sens_sig = (IIR_Declaration *)((IIR_DesignatorExplicit *)current_signal_designator)->get_name();

	_cc_out <<"  wait[" << wait->wait_id << "].sensList[" << sig << "] = "
		<< "(VHDLType*)&(";
	sens_sig->_publish_cc_declarator();
	_cc_out << ");\n";
	current_signal_designator = wait->sensitivity_list.successor( current_signal_designator );
	sig++;
      }	// while
      if(sens_list_newed == TRUE) {
	current_signal_designator = wait->sensitivity_list.first();
	while(current_signal_designator != NULL) {
	  IIR_Designator *next =  wait->sensitivity_list.successor(current_signal_designator);
	  wait->sensitivity_list.remove( current_signal_designator );
	  delete (IIR_DesignatorExplicit *)current_signal_designator;
	  current_signal_designator = next;
	} // while
      }	// if(sens_list_newed == TRUE)
    }
  } // for
  _cc_out << "\n";
}


set<IIR_Declaration> *
IIRScram_ProcessStatement::_find_declarations( IIR_Name *name ){
  return process_declarative_part._find_declarations( name );
}


IIR *
IIRScram_ProcessStatement::_clone() {
  IIR_Declaration *decl;
  IIR_SequentialStatement *clone, *stmt = process_statement_part.first();
  IIR_ProcessStatement *newproc = new IIR_ProcessStatement;
  IIRScram::_clone( newproc );
  IIR_Label *label = get_label();

  if (label != NULL) {
    label = (IIR_Label*)label->_clone();
  }
  newproc->set_label(label);

#ifdef DEBUG_ELAB
  cout << "cloning process |";
  if (label == NULL || label->get_declarator() == NULL) {
    cout << "NONAME|\n";
  }
  else {
    cout << *label->get_declarator() << "|\n";;
  }
#endif

  decl = process_declarative_part.first();
  while ( decl != NULL ) {
      // This forces a distinct clone of these node types to be created
      // for this process.
    switch (decl->get_kind()) {
    case IIR_SIGNAL_DECLARATION:
      ((IIRScram_SignalDeclaration*)decl)->_my_clone = NULL;
      break;
    case IIR_FILE_DECLARATION:
      ((IIRScram_FileDeclaration*)decl)->_my_clone = NULL;
      break;
    case IIR_VARIABLE_DECLARATION:
      ((IIR_VariableDeclaration*)decl)->_my_clone = NULL;
      break;
    case IIR_ALIAS_DECLARATION:
      ((IIR_AliasDeclaration*)decl)->_my_clone = NULL;
      break;
    default:
      break;
    }
    newproc->process_declarative_part.append((IIR_Declaration*)decl->_clone());
    decl = process_declarative_part.successor( decl );
  }

  newproc->set_postponed(get_postponed());
  while (stmt != NULL) {
    clone = (IIR_SequentialStatement*)stmt->_clone();
    newproc->process_statement_part.append(clone);
    stmt = process_statement_part.successor(stmt);
  }
  return newproc;
}


#ifdef PROCESS_COMBINATION
void 
IIRScram_ProcessStatement::_static_elaborate(IIR_ArchitectureDeclaration *arch,
					     IIR_DeclarationList *cfglist,
					     char *hier_location) {
  ostrstream newname;
  IIR_Label *label;
  IIR_Char *text = NULL;
  IIR_Declaration *decl, *declclone, *next;
  char *hier_ptr;
  int hier_length = strlen( hier_location );

  label = get_label();
  if (label != NULL) {
    newname << hier_location << *label->get_declarator() << "_" << ends;
  }
  else {
    label = new IIR_Label;
    copy_location( this, label );
    newname << hier_location << this << ends;
  }
  text = newname.str();

  // remove the trailing underscore from the process name
  *strrchr(text, '_') = '\0';
#ifdef DEBUG_ELAB
  cout << "elaborated process |" << text << "|\n";
#endif
  label->set_declarator(IIR_Identifier::get(text, strlen(text)));

  set_label(label);
  
  // replace the trailing underscore for furhter hierarchy building
  text[ strlen(text) ] = '_';

  strcpy( hier_location, text );

  decl = process_declarative_part.first();
  while ( decl != NULL ) {
    next = process_declarative_part.successor(decl);
    decl->_static_elaborate_decl(hier_location);
    decl = next;
  }

  hier_location[hier_length] = '\0';
  delete [] text;
}


bool
IIRScram_ProcessStatement::_convert_to_TWF() {
  // A process might have waits inside procedure calls it contains.
  // Therefore, I consider processes with procedure calls un-combinable.
  // This prevents processes with e.g. writes or reads inside them from
  // being combined.  This is fine, because the generated code will
  // contain duplicate labels otherwise, as each statement gets generated
  // twice: once in the initialization part, and once in the execution
  // part.
  dl_list<IIR_ProcedureCallStatement> proc_stmt_list;
  process_statement_part._build_procedure_call_stmt_list(&proc_stmt_list);
  if ( 0 < proc_stmt_list.num_elements() ) {
    return false;
  }

  if ( IIR_WAIT_STATEMENT != process_statement_part.last()->get_kind() ) {
    cout << "Process ";
    if ( NULL != get_label() ) {
      cout << *get_label();
    }
    else {
      cout << "UNNAMED";
    }
    cout << " needs to be TWF-converted.\n";
    return false;
  }
  return true;
}
#endif


void
IIRScram_ProcessStatement::_add_declarations_in_initializations() {
  IIR_Declaration *decl;
  //Constants are made to publish first because,
  //other types of objects can be initialized using these constants
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    if ( decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_DECLARATION && decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_BODY_DECLARATION ){
      
      decl->_add_declarations_in_initializations();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_implicit_declaration() == FALSE) {
      decl->_add_declarations_in_initializations();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_add_declarations_in_initializations();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    decl->_add_declarations_in_initializations();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }

  //   decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_VARIABLE].first();
  //   while (decl != NULL) {
  //     decl->_add_declarations_in_initializations();
  //     decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_VARIABLE].successor(decl);
  //   }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    decl->_add_declarations_in_initializations();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
}


void
IIRScram_ProcessStatement::_build_wait_list() {
  // This core of this method will only execute if the wait statement list
  // has no elements.  It will have no elements if 1) the method hasn't
  // been called yet, or 2) it has been called, but the process has no
  // wait statements (which is bad VHDL, but should cause no problems
  // here).  This check is to make the method idempotent.
  if ( 0 == _wait_stmt_list.num_elements() ) {
    IIR_SequentialStatement *stmt;
 
    stmt = process_statement_part.first();
    while ( stmt != NULL ) {
      stmt->_build_wait_list((dl_list<IIRScram_WaitStatement>*)&_wait_stmt_list);
      stmt = process_statement_part.successor(stmt);
    }
  }
}

void
IIRScram_ProcessStatement::_publish_cc_concurrent_stmt_init(IIR_DeclarationList* decl_list) {
  if (this->_get_label() != NULL) {
    _cc_out << *(this->_get_label());
  }
  else {
    _cc_out << "ANON_PROCESS" << this;
  }
  _cc_out << "_elab_obj = NULL;\n";
}
