// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_ObjectDeclaration.cc,v 1.6 1999/10/21 15:13:35 dmadhava Exp $
// 
//---------------------------------------------------------------------------

#include "IIR_ObjectDeclaration.hh"
#include "IIR_RecordTypeDefinition.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "error_func.hh"
#include "IIR_TextLiteral.hh"
#include <strstream.h>
#include "IIR_Identifier.hh"
#include "IIR_ArrayTypeDefinition.hh"

#include "symbol_table.hh"
extern symbol_table *cgen_sym_tab_ptr;

IIRScram_ObjectDeclaration::~IIRScram_ObjectDeclaration() {}

IIR_TypeDefinition *
IIRScram_ObjectDeclaration::_get_rval_pointed_at(){
  IIR_TypeDefinition *my_subtype = get_subtype();
  if( my_subtype->_is_access_type() == FALSE ){
    return NULL;
  }
  else{
    IIR_AccessTypeDefinition *as_access_type = (IIR_AccessTypeDefinition *)my_subtype;
    return as_access_type->get_designated_type();
  }
}


IIR_TypeDefinition *
IIRScram_ObjectDeclaration::_get_subtype(){
  return get_subtype();
}

IIR_Boolean
IIRScram_ObjectDeclaration::_is_object(){
  return TRUE;
}

IIR_Int32 
IIRScram_ObjectDeclaration::_get_num_indexes(){
  return get_subtype()->_get_num_indexes();
}

IIR_TypeDefinition *
IIRScram_ObjectDeclaration::_get_type_of_element( int index ){
  IIR_TypeDefinition *retval;
  retval = get_subtype()->_get_type_of_element(index);

  return retval;
}

set<IIR_Declaration> *
IIRScram_ObjectDeclaration::_find_declarations( IIR_Name *to_find ){
  set<IIR_Declaration> *retval = NULL;

  ASSERT( get_subtype() != NULL );

  if( get_subtype()->_is_record_type() == TRUE ){
    IIR_RecordTypeDefinition *my_subtype;
    my_subtype = (IIR_RecordTypeDefinition *)get_subtype();
    retval = my_subtype->_find_declarations( to_find );
  }
  else if( get_subtype()->_is_access_type() == TRUE ){
    IIR_AccessTypeDefinition *my_subtype;
    my_subtype = (IIR_AccessTypeDefinition *)get_subtype();
    if( my_subtype == NULL ){
      ostrstream err;
      err << "|" << *this << "| cannot refer to element |" << *(IIR *)to_find << "|, as " 
	  << "subtype |" << *get_subtype() << "| has not yet been defined." << ends;
      report_error( (IIR *)to_find, err );
      return NULL;
    }
    else{
      retval = my_subtype->_find_declarations( to_find );
    }
  }

  return retval;
}

void 
IIRScram_ObjectDeclaration::_type_check(){
  if( get_value() != NULL ){
    set<IIR_TypeDefinition> *initializer_types = get_value()->_get_rval_set();
    if( initializer_types == NULL ){
      report_undefined_symbol( get_value() );
      return;
    }

    IIR_Boolean exact_match = FALSE;
    IIR_TypeDefinition *current_type = initializer_types->get_element();
    while( current_type != NULL ){
      if( current_type == get_subtype() ){
	exact_match = TRUE;
	break;
      }
      current_type = initializer_types->get_next_element();
    }
    
    IIR_Boolean one_matched = FALSE;
    if( exact_match == FALSE ){
      current_type = initializer_types->get_element();
      while( current_type != NULL ){
	if( current_type->_is_compatible( get_subtype() ) != NULL ){
	  one_matched = TRUE;
	  break;
	}
	current_type = initializer_types->get_next_element();
      }
    }

    if( exact_match == TRUE ){
      set_value( get_value()->_semantic_transform( current_type ) );
      get_value()->_type_check( current_type );
      set_value( get_value()->_rval_to_decl( current_type ) );      
    }
    else if( one_matched == TRUE ){
      set_value( get_value()->_semantic_transform( get_subtype() ) );
      get_value()->_type_check( get_subtype() );
      set_value( get_value()->_rval_to_decl( get_subtype() ) );
    }
    else{
      ostrstream err;
      err << "Initializer |" << *get_value() << "| is incompatible with type |" << 
	*get_subtype()->_get_declarator() << "|." << ends;
      report_error( this, err );
    }

    delete initializer_types;
  }
  // There's no initializer.
  if( (_is_interface() == FALSE && (_is_variable() == TRUE || _is_signal() == TRUE))
      || _is_element() == TRUE ){
    if( get_subtype()->_is_access_type() == FALSE && 
	get_subtype()->_is_unconstrained_array_type() == TRUE ){
      ostrstream err;
      err << "A signal or variable object with an array type must be constrained." << ends;
      report_error( this, err ); 
    }
  }
  
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_scalar_type(){
  return get_subtype()->_is_scalar_type();
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_array_type(){
  return get_subtype()->_is_array_type();
}

IIR_Boolean
IIRScram_ObjectDeclaration::_is_record_type() {
  return get_subtype()->_is_record_type();
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_access_type(){
  return get_subtype()->_is_access_type();
}

void 
IIRScram_ObjectDeclaration::_publish_cc_constructor_args() {
  _cc_out << "(";

  if (get_subtype()->_is_access_type() == FALSE) {
    _cc_out << "ObjectBase::";
    _publish_cc_object_type();
  }  
  
  if((get_subtype()->_is_array_type() == TRUE) && 
     (get_subtype()->_is_anonymous() == TRUE)) {
    _cc_out << ", ";
    get_subtype()->_publish_cc_range();
  } else if(get_subtype()->_is_unconstrained_array_type()){
    // ASSERT(get_value() != NULL);
    if((get_value() != NULL) &&
       (get_value()->get_kind() != IIR_CONCATENATION_OPERATOR)) {
      _cc_out << ", ";
      get_value()->_publish_cc_range();
    }
  }
  
  if (get_subtype()->_is_access_type() == TRUE)  {
    IIR_TypeDefinition *accessType = ((IIR_AccessTypeDefinition *) get_subtype())->get_designated_type();
    if (accessType->_is_array_type() == TRUE) {
      if ((accessType->_is_anonymous()) || (accessType->_is_unconstrained_array_type())) {
	if (get_value() != NULL) {
	  get_value()->_publish_cc();
	  _cc_out << ")";
	  return;
	}
      }
    }
  }

  // Well publishing the value of guard signals .ie. the guard expression
  // also drops here. The guard expression could use signal attributes that
  // are not defined at the time of construction. The actual value will be
  // set in initialize() routine in the Kernel. So we don't need to publish
  // the guard expression. Just get out if the signal is a guard.

  if ((_is_signal() == TRUE) && (_is_guard_signal() == TRUE)) {
    _cc_out << ")";
    return;
  }

  // If the value is a parameter in a function/procedure then don't publish it
  if ((get_value() != NULL) && (get_value()->get_kind() == IIR_CONSTANT_INTERFACE_DECLARATION) && (((IIR_Declaration *) get_value())->_get_declarative_region()->_is_subprogram() == TRUE))  {
    if ((_get_currently_publishing_unit() != IIRScram::PROCEDURE) && 
         (_get_currently_publishing_unit() != IIRScram::FUNCTION))   {
      _cc_out << ")";
      return;
    }
  }
  if(get_value() != NULL) {
    if (get_subtype()->_is_access_type() == FALSE) {
      _cc_out << ", ";
    }
    if ((this->_get_subtype()->_is_scalar_type() == TRUE) && 
	(get_value()->_is_literal() == FALSE) && 
	(get_value()->_is_enumeration_literal() == FALSE) && 
	(get_value()->get_kind() != IIR_PHYSICAL_LITERAL)) {
      _cc_out << "(";
      get_value()->_get_subtype()->_publish_cc_kernel_type();
      _cc_out << " &) ";
    }
    get_value()->_publish_cc_initialization_value();
  }
  else {
    if ((_get_subtype()->_is_iir_scalar_type_definition() == TRUE) &&
	(_get_subtype()->_is_anonymous() == TRUE) &&
	(_is_constant() == FALSE)) {
      _cc_out << ", ";
      if (_get_subtype()->_get_attribute() != NULL) {
	_cc_out << "(";
	_get_subtype()->_publish_cc_universal_type();
	_cc_out << " &) ";
      }
      _get_subtype()->_publish_cc_universal_left();
    }
  }

  // Check and publish the necessary type_info for scalar types.
  
  if (get_subtype()->_is_scalar_type() == TRUE &&
      get_subtype()->_is_access_type() == FALSE) {
    get_subtype()->_publish_cc_object_type_info();
  }
  
  _cc_out << ")";
}

void
  IIRScram_ObjectDeclaration::_publish_cc_universal_type() {
  get_subtype()->_publish_cc_universal_type();
}

void 
IIRScram_ObjectDeclaration::_publish_cc_object_type() {
  _report_undefined_scram_fn("_publish_cc_object_type()");
}

void
IIRScram_ObjectDeclaration::_publish_vhdl(ostream &_vhdl_out){
  get_declarator()->_publish_vhdl(_vhdl_out);
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_declarator_with_colon(ostream &_vhdl_out){
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " : ";
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_subtype_indication(ostream &_vhdl_out){
  if (get_subtype()->_is_anonymous() == TRUE) {
    get_subtype()->_publish_vhdl_decl(_vhdl_out);
  }
  else {
    get_subtype()->_publish_vhdl(_vhdl_out);
  }
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_expression(ostream &_vhdl_out){
  if (get_value() != NULL) {
    _vhdl_out << " := ";
    get_value()->_publish_vhdl(_vhdl_out);
  }
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_subtype_indication_with_expression(ostream &_vhdl_out){
  _publish_vhdl_subtype_indication(_vhdl_out);
  _publish_vhdl_expression(_vhdl_out);
  _vhdl_out << ";\n";
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_signal_kind(ostream &_vhdl_out){
  switch(_get_signal_kind()) {
  case IIR_REGISTER_KIND:
    _vhdl_out << " register ";
    break;
  case IIR_BUS_KIND:
    _vhdl_out << " bus ";
    break;
  case IIR_NO_SIGNAL_KIND:
    break;
  default:
    cerr << " ERROR!!! undefined Signal Kind in node " << get_kind_text() << ends;
    abort();
   }
}

void
IIRScram_ObjectDeclaration::_publish_cc_decl_with_args(){
  get_subtype()->_publish_cc();
  _cc_out << " ";
  _get_declarator()->_publish_cc();
  _cc_out << "(";
  _publish_cc_constructor_args();
  _cc_out << ");" << endl;
}

void
IIRScram_ObjectDeclaration::_publish_cc_type_name() {
  ASSERT(_get_subtype() != NULL);
  ASSERT(_get_subtype()->_is_TypeDefinition() == TRUE);
  if (_get_subtype()->_is_scalar_type() == TRUE)  {
    _get_subtype()->_publish_cc_kernel_type();
  }
  else {
    _get_subtype()->_publish_cc_type_name();
  }
}

void
IIRScram_ObjectDeclaration::_publish_cc_range() {
  ASSERT(get_subtype() != NULL);
  ASSERT(get_subtype()->_is_TypeDefinition() == TRUE);
  get_subtype()->_publish_cc_range();
}

void
IIRScram_ObjectDeclaration::_publish_cc_bounds() {
  ASSERT(get_subtype() != NULL);
  ASSERT(get_subtype()->_is_TypeDefinition() == TRUE);
  get_subtype()->_publish_cc_bounds();
}

const IIR_Char*
IIRScram_ObjectDeclaration::_get_mangling_prefix()
{
  return ((const IIR_Char *) "O");
}

IIR_SignalKind
IIRScram_ObjectDeclaration::_get_signal_kind(){
  _report_undefined_scram_fn("_get_signal_kind()");

  return IIR_NO_SIGNAL_KIND;
}
  
void
IIRScram_ObjectDeclaration::_add_declarations_in_initializations() {
  if(get_value() != NULL) {
    get_value()->_add_decl_into_cgen_symbol_table();
  }
  if(_get_subtype() != NULL) {
    _get_subtype()->_add_decl_into_cgen_symbol_table();
  }
}


void 
IIRScram_ObjectDeclaration::_add_decl_into_cgen_symbol_table(){
  if(get_value() != NULL) {
    get_value()->_add_decl_into_cgen_symbol_table();
  }
  
  if ((cgen_sym_tab_ptr != NULL) && (!cgen_sym_tab_ptr->in_scope(this))) {
    cgen_sym_tab_ptr->add_declaration(this);
  }
}


void
IIRScram_ObjectDeclaration::_clone(IIR_ObjectDeclaration *clone) {
  IIRScram_Declaration::_clone(clone);
  clone->attributes = attributes;
  clone->set_subtype(get_subtype());
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_locally_static(){
  IIR_Boolean retval = TRUE;

  ASSERT( get_subtype() != NULL );
  if( get_subtype()->_is_access_type() == TRUE ){
    retval = FALSE;
  }

  return retval;
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_globally_static(){
  IIR_Boolean retval = TRUE;

  ASSERT( get_subtype() != NULL );
  if( get_subtype()->_is_access_type() == TRUE ){
    retval = FALSE;
  }

  return retval;
}
