// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ececs.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_ConcurrentConditionalSignalAssignment.cc,v 1.3 1999/03/25 02:57:21 tmcbraye Exp $
// 
//---------------------------------------------------------------------------
#include "IIRScram_ConcurrentConditionalSignalAssignment.hh"
#include "IIR_ConditionalWaveform.hh"
#include "IIR_DesignatorList.hh"
#include "IIR_Elsif.hh"
#include "IIR_Identifier.hh"
#include "IIR_IfStatement.hh"
#include "IIR_Label.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_SignalAssignmentStatement.hh"
#include "IIR_WaitStatement.hh"
#include "IIR_WaveformElement.hh"
#include "error_func.hh"
#include "IIR_StringLiteral.hh"
#include "IIR_SignalDeclaration.hh"

extern bool parse_error;

IIRScram_ConcurrentConditionalSignalAssignment::~IIRScram_ConcurrentConditionalSignalAssignment() {}


void 
IIRScram_ConcurrentConditionalSignalAssignment::_publish_vhdl(ostream &_vhdl_out) {
  if(get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ": ";
  }

  if (get_postponed() == true) {
    _vhdl_out << "postponed ";
  }
  
  get_target()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " <= ";
  switch(get_delay_mechanism()) {
  case IIR_TRANSPORT_DELAY:
    _vhdl_out << " transport ";
    break;
  case IIR_INERTIAL_DELAY:
    if(get_reject_time_expression() != NULL) {
      _vhdl_out << " reject ";
      get_reject_time_expression()->_publish_vhdl(_vhdl_out);
    }
    _vhdl_out << " inertial ";
    break;
  }
  conditional_waveforms._publish_vhdl(_vhdl_out);
  _vhdl_out << ";" << endl;
}


IIR * 
IIRScram_ConcurrentConditionalSignalAssignment::_transmute() {
  //### yet to take care of guarded signal assignment
  IIR_ProcessStatement* pstmt = new IIR_ProcessStatement;
  IIR_WaitStatement* wstmt = new IIR_WaitStatement;

  copy_location( this, pstmt );
  copy_location( this, wstmt );

  IIR_SignalAssignmentStatement* sastmt = NULL;
  IIR_IfStatement* guardIfStmt = NULL;
  IIR_IfStatement* ifstmt;
  IIR_Elsif* elsestmt;
  int ifflag = 0;
  IIR* prevstmt;
  IIR* ifelsifstmt;
  IIR* cndtion;
  pstmt->set_label(get_label());
  pstmt->set_postponed(get_postponed());

  if(conditional_waveforms.num_elements() == 0) {
    cerr << "IIRScram_ConcurrentConditionalSignalAssignmentStatement: Encountered Conditional Signal Assignment Stmt with an empty waveform list." << endl;
    abort();
    return NULL;
  }

  // If it is a guarded sign assignemnt then the transform is done as per
  // LRM chapter 9.5  
  if (get_guarded() == TRUE) {
    ASSERT(_get_guard_signal() != NULL);
    cndtion = _get_guard_signal();
    guardIfStmt = new IIR_IfStatement();
    copy_location( this, guardIfStmt );
    guardIfStmt->set_condition(cndtion);
      
    // else clause 
    // The disconnection specification applies only if the target is a
    // guarded target - section 9.5 of LRM
    if (get_target()->_get_signal_kind() == IIR_BUS_KIND || get_target()->_get_signal_kind() == IIR_REGISTER_KIND) {
      sastmt = new IIR_SignalAssignmentStatement;
      copy_location( this, sastmt );
      sastmt->set_target(get_target());
      sastmt->set_delay_mechanism(get_delay_mechanism());
      sastmt->set_reject_time_expression(get_reject_time_expression());
      IIR_WaveformElement *wave = new IIR_WaveformElement();
      copy_location( this, wave );
      // Post a null transaction 
      IIR_StringLiteral *nullLit = IIRBase_StringLiteral::get("null", 4);
      nullLit->_set_subtype(get_target()->_get_subtype());
      wave->set_value(nullLit);
      // Set the disconnection specification
      wave->set_time(NULL);
      sastmt->waveform.append(wave);
      guardIfStmt->else_sequence.append(sastmt);
    }

    if((sastmt != NULL) &&
       (sastmt->get_delay_mechanism() == IIR_INERTIAL_DELAY)) {
      // Either a reject time was specified, or not...
      if( sastmt->get_reject_time_expression() == NULL ){
	IIR_WaveformElement *first_waveform = sastmt->waveform.first();
	ASSERT( first_waveform != NULL );
	IIR *time_expression = first_waveform->get_time();
	if( time_expression == NULL ){
	  // Then the statement is something like x <= inertial y;
	  // We should checnge it to transport...
	  sastmt->set_delay_mechanism( IIR_TRANSPORT_DELAY );
	}
      }
    }

    ASSERT (guardIfStmt != NULL);
    
    pstmt->process_statement_part.append(guardIfStmt);
  }

  int numelements = conditional_waveforms.num_elements();
  IIR_ConditionalWaveform* waveform= conditional_waveforms.first();

  if(numelements == 1) {

    sastmt = new IIR_SignalAssignmentStatement;
    copy_location( this, sastmt );
    sastmt->set_target(get_target());
    sastmt->set_delay_mechanism(get_delay_mechanism());
    sastmt->set_reject_time_expression(get_reject_time_expression());
    sastmt->waveform = waveform->waveform;

    if( sastmt->get_delay_mechanism() == IIR_INERTIAL_DELAY ){
      // Either a reject time was specified, or not...
      if( sastmt->get_reject_time_expression() == NULL ){
	IIR_WaveformElement *first_waveform = sastmt->waveform.first();
	ASSERT( first_waveform != NULL );
	IIR *time_expression = first_waveform->get_time();
	if( time_expression == NULL ){
	  // Then the statement is something like x <= inertial y;
	  // We should checnge it to transport...
	  sastmt->set_delay_mechanism( IIR_TRANSPORT_DELAY );
	}
      }
    }

    if (waveform->get_condition() != NULL) {
      ifstmt = new IIR_IfStatement();
      copy_location( this, ifstmt );
      ifstmt->then_sequence.append(sastmt);
      ifstmt->set_condition(waveform->get_condition());
      
      if (guardIfStmt != NULL) {
	guardIfStmt->then_sequence.append(ifstmt);
      } else {
	pstmt->process_statement_part.append(ifstmt);
      }
    } else {
      if (guardIfStmt != NULL) {
	guardIfStmt->then_sequence.append(sastmt);
      } else {
	pstmt->process_statement_part.append(sastmt);
      }
    }
  }
  else {
    for(; waveform!= NULL; ) {
      sastmt =  new IIR_SignalAssignmentStatement;
      copy_location( this, sastmt );
      sastmt->set_target(get_target());
      sastmt->set_delay_mechanism(get_delay_mechanism());
      sastmt->set_reject_time_expression(get_reject_time_expression());
      sastmt->waveform = waveform->waveform;
      cndtion = waveform->get_condition();
      
      if( sastmt->get_delay_mechanism() == IIR_INERTIAL_DELAY ){
	// Either a reject time was specified, or not...
	if( sastmt->get_reject_time_expression() == NULL ){
	  IIR_WaveformElement *first_waveform = sastmt->waveform.first();
	  ASSERT( first_waveform != NULL );
	  IIR *time_expression = first_waveform->get_time();
	  if( time_expression == NULL ){
	    // Then the statement is something like x <= inertial y;
	    // We should checnge it to transport...
	    sastmt->set_delay_mechanism( IIR_TRANSPORT_DELAY );
	  }
	}
      }
      
      if(!ifflag) {
	ifstmt = new IIR_IfStatement();
	copy_location( this, ifstmt );
	prevstmt = ifstmt;
	ifelsifstmt = ifstmt;
	ifstmt->then_sequence.append(sastmt);
	ifstmt->set_condition(cndtion);
	ifflag =1;
      }
      else {
	if(cndtion != NULL) {
	  elsestmt= new IIR_Elsif();
	  copy_location( this, elsestmt );
	  ifelsifstmt = elsestmt;
	  elsestmt->set_condition(cndtion);
	  elsestmt->then_sequence_of_statements.append(sastmt);
	  if(ifflag == 1) {
	    ((IIR_IfStatement*)prevstmt)->set_elsif((IIR_Elsif*)ifelsifstmt);
	    ifflag = 2;
	  }
	  else {
	    ((IIR_Elsif*)prevstmt)->set_else_clause((IIR_Elsif*)ifelsifstmt);
	    ifflag = 2;
	  }
	}
	else {
	  ifstmt->else_sequence.append(sastmt);
	}
      }
      prevstmt = ifelsifstmt;
      waveform = conditional_waveforms.successor(waveform);
    }
    if (guardIfStmt != NULL) {
      guardIfStmt->then_sequence.append(ifstmt);
    } else {
      pstmt->process_statement_part.append(ifstmt);
    }
  }
  IIR_DesignatorList sensitivity_list;
  _build_sensitivity_list(&sensitivity_list);
  // Adding the guard signal to the sensitivity list for GSA's
  if (get_guarded() == TRUE) {
    _get_guard_signal()->_build_sensitivity_list(&sensitivity_list);
  }
  wstmt->sensitivity_list._add_signals_to_sensitivity_list(&sensitivity_list);
  pstmt->process_statement_part.append(wstmt);  
  //  pstmt->process_declarative_part = ((IIR_ArchitectureDeclaration*)_current_publish_node)->architecture_declarative_part;
  return pstmt;
}


void 
IIRScram_ConcurrentConditionalSignalAssignment::_type_check(){
  _type_check_target_and_waveforms();
  ASSERT( _is_resolved() == TRUE || parse_error == TRUE );
  if( get_guarded() == TRUE ){
    _resolve_guard_signal( _get_symbol_table() );
  }
}


void 
IIRScram_ConcurrentConditionalSignalAssignment::_type_check_target_and_waveforms(){
  IIR_ConditionalWaveform *current_waveform = conditional_waveforms.first();
  while( current_waveform != NULL ){
    if( current_waveform->get_condition() != NULL ){
      IIR *resolved_condition;
      resolved_condition =
	_type_check_and_resolve_boolean_condition( current_waveform->get_condition() );
      current_waveform->set_condition( resolved_condition );
    }
    
    _type_check_target_and_waveform( &current_waveform->waveform );
    _type_check_mechanism_and_time( &current_waveform->waveform );    
    current_waveform = conditional_waveforms.successor( current_waveform );
  }
}


IIR_Boolean 
IIRScram_ConcurrentConditionalSignalAssignment::_is_resolved(){
  IIR_Boolean retval = TRUE;
  if( get_target() != NULL && get_target()->_is_resolved() == FALSE ){
    retval = FALSE;
  }
  else if( get_reject_time_expression() != NULL 
	   && get_reject_time_expression()->_is_resolved() == FALSE ){
    retval = FALSE;
  }
  else if( conditional_waveforms._is_resolved() == FALSE ){
    retval = FALSE;
  }
  return retval;
}


void 
IIRScram_ConcurrentConditionalSignalAssignment::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list) {
  conditional_waveforms._build_sensitivity_list(sensitivity_list);
}


IIR *
IIRScram_ConcurrentConditionalSignalAssignment::_clone() {
  IIR *cstmt;
  cstmt = _transmute();
  cstmt = cstmt->_clone();
  return cstmt;
}


#ifdef PROCESS_COMBINATION
void
IIRScram_ConcurrentConditionalSignalAssignment::
_static_elaborate(IIR_ArchitectureDeclaration *arch,
		  IIR_DeclarationList *cfglist,
		  char *hier_location) {
  ostrstream newname;
  IIR_Label *label;
  IIR_Char *text;
  IIR_Int32 i;

  newname << hier_location;
  label = get_label();
  if (label != NULL) {
    newname << *label->get_declarator();
  }
  else {
    newname << "UNKNOWNCCA";
  }
  newname << ends;
  text = newname.str();
#ifdef DEBUG_ELAB
  cout << "elaborated ConcCondSigAssign |" << text << "|\n";
#endif

  ASSERT(label->get_declarator()->get_kind() == IIR_IDENTIFIER);
  ((IIR_Identifier *)label->get_declarator())->release();
  label->set_declarator(IIR_Identifier::get(text, strlen(text)));
}
#endif


IIR *
IIRScram_ConcurrentConditionalSignalAssignment::_get_target(){
  return get_target();
}


void 
IIRScram_ConcurrentConditionalSignalAssignment::_set_target( IIR *new_target ){
  set_target( new_target );
}

  
IIR_DelayMechanism 
IIRScram_ConcurrentConditionalSignalAssignment::_get_delay_mechanism(){
  return get_delay_mechanism();
}


void 
IIRScram_ConcurrentConditionalSignalAssignment::_set_delay_mechanism(IIR_DelayMechanism new_mech){
  set_delay_mechanism( new_mech );
}


IIR *
IIRScram_ConcurrentConditionalSignalAssignment::_get_reject_time_expression(){
  return get_reject_time_expression();
}


void 
IIRScram_ConcurrentConditionalSignalAssignment::_set_reject_time_expression(IIR *new_reject_time){
  set_reject_time_expression( new_reject_time );
}
