/*
 * Derived from QEMU sources.
 * Modified for FAUmachine by Volkmar Sieh.
 *
 *  Copyright (c) 2007-2009 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */

#ifdef STATE

// struct {
	/* emulator internal variables */
	float80 ft0;
#define FT0(cpssp)    (cpssp->ft0)

	union MMXReg {
		uint8_t _b[8];
		uint16_t _w[2];
		uint32_t _l[1];
		uint64_t q;
	} mmx_t0;
#ifdef WORDS_BIGENDIAN
#define MMX_B(n) _b[7 - (n)]
#define MMX_W(n) _w[3 - (n)]
#define MMX_L(n) _l[1 - (n)]
#else
#define MMX_B(n) _b[n]
#define MMX_W(n) _w[n]
#define MMX_L(n) _l[n]
#endif
#define MMX_Q(n) q
	union XMMReg {
		uint8_t _b[16];
		uint16_t _w[8];
		uint32_t _l[4];
		uint64_t _q[2];
		float32 _s[4];
		float64 _d[2];
	} xmm_t0;
#ifdef WORDS_BIGENDIAN
#define XMM_B(n) _b[15 - (n)]
#define XMM_W(n) _w[7 - (n)]
#define XMM_L(n) _l[3 - (n)]
#define XMM_S(n) _s[3 - (n)]
#define XMM_Q(n) _q[1 - (n)]
#define XMM_D(n) _d[1 - (n)]
#else
#define XMM_B(n) _b[n]
#define XMM_W(n) _w[n]
#define XMM_L(n) _l[n]
#define XMM_S(n) _s[n]
#define XMM_Q(n) _q[n]
#define XMM_D(n) _d[n]
#endif
	float_status fp_status;
	float_status sse_status;

	/* FPU state */
	unsigned int fpstt; /* top of stack index */

	unsigned int fpus;

#define FPUS_IE (1 << 0)
#define FPUS_DE (1 << 1)
#define FPUS_ZE (1 << 2)
#define FPUS_OE (1 << 3)
#define FPUS_UE (1 << 4)
#define FPUS_PE (1 << 5)
#define FPUS_SF (1 << 6)
#define FPUS_SE (1 << 7)
#define FPUS_B  (1 << 15)

#define FPUC_EM 0x3f

#define RC_MASK         0xc00
#define RC_NEAR         0x000
#define RC_DOWN         0x400
#define RC_UP           0x800
#define RC_CHOP         0xc00
	unsigned int fpuc;

	uint8_t fptags[8];   /* 0 = valid, 1 = empty */
	union {
		float80 d __attribute__((aligned(16)));
		union MMXReg mmx;
	} fpregs[8];
	uint32_t mxcsr;
	union XMMReg xmm_regs[CPU_NB_REGS];
#define ST0(cpssp)    (cpssp->fpregs[cpssp->fpstt].d)
#define ST(cpssp, n)  (cpssp->fpregs[(cpssp->fpstt + (n)) & 7].d)
#define ST1(cpssp)    ST(cpssp, 1)
// } NAME;

#endif /* STATE */
#ifdef EXPORT
#endif /* EXPORT */
#ifdef BEHAVIOR

#define MAXTAN 9223372036854775808.0

/* the following deal with x86 long double-precision numbers */
#define MAXEXPD 0x7fff
#define EXPBIAS 16383
#define EXPD(fp)        (fp.l.upper & 0x7fff)
#define SIGND(fp)       ((fp.l.upper) & 0x8000)
#define MANTD(fp)       (fp.l.lower)
#define BIASEXPONENT(fp) fp.l.upper = (fp.l.upper & ~(0x7fff)) | EXPBIAS

/* only for x86 */
typedef union {
	long double d;
	struct {
		unsigned long long lower;
		unsigned short upper;
	} l;
} CPU86_LDoubleU;

static float80
NAME_(helper_fldt)(struct cpssp *cpssp, target_ulong ptr)
{
	CPU86_LDoubleU temp;

	temp.l.lower = ldq(cpssp, ptr);
	temp.l.upper = lduw(cpssp, ptr + 8);
	return temp.d;
}

static void
NAME_(helper_fstt)(struct cpssp *cpssp, float80 f, target_ulong ptr)
{
	CPU86_LDoubleU temp;

	temp.d = f;
	stq(cpssp, ptr, temp.l.lower);
	stw(cpssp, ptr + 8, temp.l.upper);
}

static void
fpu_set_exception(struct cpssp *cpssp, int mask)
{
	cpssp->fpus |= mask;
	if (cpssp->fpus & (~cpssp->fpuc & FPUC_EM)) {
		cpssp->fpus |= FPUS_SE | FPUS_B;
		if (unlikely(! (cpssp->cr[0] & CPU_CR0_NE_MASK))) {
			NAME_(fpu_check_ferr)(cpssp);
		}
	}
}

void
NAME_(helper_fldt_ST0_A0)(struct cpssp *cpssp, target_ulong a0)
{
	int new_fpstt;

	new_fpstt = (cpssp->fpstt - 1) & 7;
	cpssp->fpregs[new_fpstt].d = NAME_(helper_fldt)(cpssp, a0);
	cpssp->fpstt = new_fpstt;
	cpssp->fptags[new_fpstt] = 0; /* validate stack entry */
}

void
NAME_(helper_fstt_ST0_A0)(struct cpssp *cpssp, target_ulong a0)
{
	NAME_(helper_fstt)(cpssp, ST0(cpssp), a0);
}

void
NAME_(fpu_check_ferr)(struct cpssp *cpssp)
{
	if (cpssp->fpus & (~cpssp->fpuc & FPUC_EM)) {
		cpssp->fpus |= FPUS_SE | FPUS_B;
		if (! cpssp->state_n_ignne) {
			sig_std_logic_or_set(cpssp->sig_n_ferr, cpssp, 1);
		}
	} else {
		cpssp->fpus &= ~(FPUS_SE | FPUS_B);
		sig_std_logic_or_set(cpssp->sig_n_ferr, cpssp, 0);
	}
}

float80
NAME_(helper_fdiv)(struct cpssp *cpssp, float80 a, float80 b)
{
	if (b == 0.0)
		fpu_set_exception(cpssp, FPUS_ZE);
	return a / b;
}

void
NAME_(helper_fbld_ST0_A0)(struct cpssp *cpssp, target_ulong a0)
{
	float80 tmp;
	uint64_t val;
	unsigned int v;
	int i;

	val = 0;
	for (i = 8; i >= 0; i--) {
		v = ldub(cpssp, a0 + i);
		val = (val * 100) + ((v >> 4) * 10) + (v & 0xf);
	}
	tmp = val;
	if (ldub(cpssp, a0 + 9) & 0x80)
		tmp = -tmp;
	fpush(cpssp);
	ST0(cpssp) = tmp;
}

void
NAME_(helper_fbst_ST0_A0)(struct cpssp *cpssp, target_ulong a0)
{
	float80 tmp;
	int v;
	target_ulong mem_ref, mem_end;
	int64_t val;

	tmp = rint(ST0(cpssp));
	val = (int64_t)tmp;
	mem_ref = a0;
	mem_end = mem_ref + 9;
	if (val < 0) {
		stb(cpssp, mem_end, 0x80);
		val = -val;
	} else {
		stb(cpssp, mem_end, 0x00);
	}
	while (mem_ref < mem_end) {
		if (val == 0)
			break;
		v = val % 100;
		val = val / 100;
		v = ((v / 10) << 4) | (v % 10);
		stb(cpssp, mem_ref++, v);
	}
	while (mem_ref < mem_end) {
		stb(cpssp, mem_ref++, 0);
	}
}

void
NAME_(helper_f2xm1)(struct cpssp *cpssp)
{
	ST0(cpssp) = powl(2.0, ST0(cpssp)) - 1.0;
}

void
NAME_(helper_fyl2x)(struct cpssp *cpssp)
{
	float80 fptemp;

	fptemp = ST0(cpssp);
	if (fptemp>0.0){
		fptemp = log2l(fptemp);
		ST1(cpssp) *= fptemp;
		fpop(cpssp);
	} else {
		cpssp->fpus &= (~0x4700);
		cpssp->fpus |= 0x400;
	}
}

void
NAME_(helper_fptan)(struct cpssp *cpssp)
{
	float80 fptemp;

	fptemp = ST0(cpssp);
	if ((fptemp > MAXTAN)||(fptemp < -MAXTAN)) {
		cpssp->fpus |= 0x400;
	} else {
		ST0(cpssp) = tanl(fptemp);
		fpush(cpssp);
		ST0(cpssp) = 1.0;
		cpssp->fpus &= (~0x400);  /* C2 <-- 0 */
		/* the above code is for  |arg| < 2**52 only */
	}
}

void
NAME_(helper_fpatan)(struct cpssp *cpssp)
{
	float80 fptemp, fpsrcop;

	fpsrcop = ST1(cpssp);
	fptemp = ST0(cpssp);
	ST1(cpssp) = atan2l(fpsrcop,fptemp);
	fpop(cpssp);
}

void
NAME_(helper_fxtract)(struct cpssp *cpssp)
{
	CPU86_LDoubleU temp;
	unsigned int expdif;

	temp.d = ST0(cpssp);
	expdif = EXPD(temp) - EXPBIAS;
	/*DP exponent bias*/
	ST0(cpssp) = expdif;
	fpush(cpssp);
	BIASEXPONENT(temp);
	ST0(cpssp) = temp.d;
}

void
NAME_(helper_fprem1)(struct cpssp *cpssp)
{
	float80 dblq, fpsrcop, fptemp;
	CPU86_LDoubleU fpsrcop1, fptemp1;
	int expdif;
	int q;

	fpsrcop = ST0(cpssp);
	fptemp = ST1(cpssp);
	fpsrcop1.d = fpsrcop;
	fptemp1.d = fptemp;
	expdif = EXPD(fpsrcop1) - EXPD(fptemp1);
	if (expdif < 53) {
		dblq = fpsrcop / fptemp;
		dblq = (dblq < 0.0) ? ceill(dblq) : floorl(dblq);
		ST0(cpssp) = fpsrcop - fptemp*dblq;
		q = (int)dblq; /* cutting off top bits is assumed here */
		cpssp->fpus &= (~0x4700); /* (C3,C2,C1,C0) <-- 0000 */
					/* (C0,C1,C3) <-- (q2,q1,q0) */
		cpssp->fpus |= (q&0x4) << 6; /* (C0) <-- q2 */
		cpssp->fpus |= (q&0x2) << 8; /* (C1) <-- q1 */
		cpssp->fpus |= (q&0x1) << 14; /* (C3) <-- q0 */
	} else {
		cpssp->fpus |= 0x400;  /* C2 <-- 1 */
		fptemp = powl(2.0, expdif - 50);
		fpsrcop = (ST0(cpssp) / ST1(cpssp)) / fptemp;
		/* fpsrcop = integer obtained by rounding to the nearest */
		fpsrcop = (fpsrcop - floor (fpsrcop) < ceill(fpsrcop)-fpsrcop) ?  floorl(fpsrcop) : ceill(fpsrcop);
		ST0(cpssp) -= (ST1(cpssp) * fpsrcop * fptemp);
	}
}

void
NAME_(helper_fprem)(struct cpssp *cpssp)
{
	float80 dblq, fpsrcop, fptemp;
	CPU86_LDoubleU fpsrcop1, fptemp1;
	int expdif;
	int q;

	fpsrcop = ST0(cpssp);
	fptemp = ST1(cpssp);
	fpsrcop1.d = fpsrcop;
	fptemp1.d = fptemp;
	expdif = EXPD(fpsrcop1) - EXPD(fptemp1);
	if ( expdif < 53 ) {
		dblq = fpsrcop / fptemp;
		dblq = (dblq < 0.0) ? ceill(dblq) : floorl(dblq);
		ST0(cpssp) = fpsrcop - fptemp*dblq;
		q = (int)dblq; /* cutting off top bits is assumed here */
		cpssp->fpus &= (~0x4700); /* (C3,C2,C1,C0) <-- 0000 */
					/* (C0,C1,C3) <-- (q2,q1,q0) */
		cpssp->fpus |= (q&0x4) << 6; /* (C0) <-- q2 */
		cpssp->fpus |= (q&0x2) << 8; /* (C1) <-- q1 */
		cpssp->fpus |= (q&0x1) << 14; /* (C3) <-- q0 */
	} else {
		cpssp->fpus |= 0x400;  /* C2 <-- 1 */
		fptemp = powl(2.0, expdif - 50);
		fpsrcop = (ST0(cpssp) / ST1(cpssp)) / fptemp;
		/* fpsrcop = integer obtained by chopping */
		fpsrcop = (fpsrcop < 0.0) ? -(floorl(fabsl(fpsrcop))) : floorl(fpsrcop);
		ST0(cpssp) -= (ST1(cpssp) * fpsrcop * fptemp);
	}
}

void
NAME_(helper_fyl2xp1)(struct cpssp *cpssp)
{
	float80 fptemp;

	fptemp = ST0(cpssp);
	if ((fptemp+1.0)>0.0) {
		fptemp = log2l(fptemp + 1.0);
		ST1(cpssp) *= fptemp;
		fpop(cpssp);
	} else {
		cpssp->fpus &= (~0x4700);
		cpssp->fpus |= 0x400;
	}
}

void
NAME_(helper_fsqrt)(struct cpssp *cpssp)
{
	float80 fptemp;

	fptemp = ST0(cpssp);
	if (fptemp<0.0) {
		cpssp->fpus &= (~0x4700);  /* (C3,C2,C1,C0) <-- 0000 */
		cpssp->fpus |= 0x400;
	}
	ST0(cpssp) = sqrtl(fptemp);
}

void
NAME_(helper_fsincos)(struct cpssp *cpssp)
{
	float80 fptemp;

	fptemp = ST0(cpssp);
	if ((fptemp > MAXTAN)||(fptemp < -MAXTAN)) {
		cpssp->fpus |= 0x400;
	} else {
		ST0(cpssp) = sinl(fptemp);
		fpush(cpssp);
		ST0(cpssp) = cosl(fptemp);
		cpssp->fpus &= (~0x400);  /* C2 <-- 0 */
		/* the above code is for  |arg| < 2**63 only */
	}
}

void
NAME_(helper_frndint)(struct cpssp *cpssp)
{
	float80 a;

	a = ST0(cpssp);
	a = rintl(a);
	ST0(cpssp) = a;
}

void
NAME_(helper_fscale)(struct cpssp *cpssp)
{
	float80 fpsrcop, fptemp;

	fpsrcop = 2.0;
	fptemp = powl(fpsrcop, ST1(cpssp));
	ST0(cpssp) *= fptemp;
}

void
NAME_(helper_fsin)(struct cpssp *cpssp)
{
	float80 fptemp;

	fptemp = ST0(cpssp);
	if ((fptemp > MAXTAN)||(fptemp < -MAXTAN)) {
		cpssp->fpus |= 0x400;
	} else {
		ST0(cpssp) = sinl(fptemp);
		cpssp->fpus &= (~0x400);  /* C2 <-- 0 */
		/* the above code is for  |arg| < 2**53 only */
	}
}

void
NAME_(helper_fcos)(struct cpssp *cpssp)
{
	float80 fptemp;

	fptemp = ST0(cpssp);
	if((fptemp > MAXTAN)||(fptemp < -MAXTAN)) {
		cpssp->fpus |= 0x400;
	} else {
		ST0(cpssp) = cosl(fptemp);
		cpssp->fpus &= (~0x400);  /* C2 <-- 0 */
		/* the above code is for  |arg5 < 2**63 only */
	}
}

void
NAME_(helper_fxam_ST0)(struct cpssp *cpssp)
{
	CPU86_LDoubleU temp;
	int expdif;
	temp.d = ST0(cpssp);

	cpssp->fpus &= (~0x4700);  /* (C3,C2,C1,C0) <-- 0000 */
	if (SIGND(temp))
		cpssp->fpus |= 0x200; /* C1 <-- 1 */

	if (cpssp->fptags[cpssp->fpstt]) {
		/* empty */
		cpssp->fpus |= 0x4100;

	} else {
		expdif = EXPD(temp);
		if (expdif == MAXEXPD) {
			if (MANTD(temp) == 0x8000000000000000ULL)
				cpssp->fpus |=  0x500 /*Infinity*/;
			else
				cpssp->fpus |=  0x100 /*NaN*/;

		} else if (expdif == 0) {
			if (MANTD(temp) == 0)
				cpssp->fpus |=  0x4000 /*Zero*/;
			else
				cpssp->fpus |= 0x4400 /*Denormal*/;
		} else {
			cpssp->fpus |= 0x400;
		}
	}
}

void
NAME_(helper_fstenv)(struct cpssp *cpssp, target_ulong ptr, int data32)
{
	int fpus, fptag, exp_, i;
	uint64_t mant;
	CPU86_LDoubleU tmp;

	fpus = (cpssp->fpus & ~0x3800) | (cpssp->fpstt & 0x7) << 11;
	fptag = 0;
	for (i=7; i>=0; i--) {
		fptag <<= 2;
		if (cpssp->fptags[i]) {
			fptag |= 3;
		} else {
			tmp.d = cpssp->fpregs[i].d;
			exp_ = EXPD(tmp);
			mant = MANTD(tmp);
			if (exp_ == 0 && mant == 0) {
				/* zero */
				fptag |= 1;
			} else if (exp_ == 0 || exp_ == MAXEXPD || (mant & (1LL << 63)) == 0) {
				/* NaNs, infinity, denormal */
				fptag |= 2;
			}
		}
	}
	if (data32) {
		/* 32 bit */
		stl(cpssp, ptr, 0xffff0000 | cpssp->fpuc);
		stl(cpssp, ptr + 4, 0xffff0000 | fpus);
		stl(cpssp, ptr + 8, 0xffff0000 | fptag);
		stl(cpssp, ptr + 12, 0); /* fpip */
		stl(cpssp, ptr + 16, 0); /* fpcs */
		stl(cpssp, ptr + 20, 0); /* fpoo */
		stl(cpssp, ptr + 24, 0); /* fpos */
	} else {
		/* 16 bit */
		stw(cpssp, ptr, cpssp->fpuc);
		stw(cpssp, ptr + 2, fpus);
		stw(cpssp, ptr + 4, fptag);
		stw(cpssp, ptr + 6, 0);
		stw(cpssp, ptr + 8, 0);
		stw(cpssp, ptr + 10, 0);
		stw(cpssp, ptr + 12, 0);
	}
}

void
NAME_(helper_fldenv)(struct cpssp *cpssp, target_ulong ptr, int data32)
{
	int i, fpus, fptag;

	if (data32) {
		cpssp->fpuc = lduw(cpssp, ptr);
		fpus = lduw(cpssp, ptr + 4);
		fptag = lduw(cpssp, ptr + 8);
	} else {
		cpssp->fpuc = lduw(cpssp, ptr);
		fpus = lduw(cpssp, ptr + 2);
		fptag = lduw(cpssp, ptr + 4);
	}
	cpssp->fpstt = (fpus >> 11) & 7;
	cpssp->fpus = fpus & ~0x3800;
	for(i = 0;i < 8; i++) {
		cpssp->fptags[i] = ((fptag & 3) == 3);
		fptag >>= 2;
	}
}

void
NAME_(helper_fsave)(struct cpssp *cpssp, target_ulong ptr, int data32)
{
	float80 tmp;
	int i;

	NAME_(helper_fstenv)(cpssp, ptr, data32);

	ptr += (14 << data32);
	for(i = 0;i < 8; i++) {
		tmp = ST(cpssp, i);
		NAME_(helper_fstt)(cpssp, tmp, ptr);
		ptr += 10;
	}

	/* fninit */
	cpssp->fpus = 0;
	cpssp->fpstt = 0;
	cpssp->fpuc = 0x37f;
	cpssp->fptags[0] = 1;
	cpssp->fptags[1] = 1;
	cpssp->fptags[2] = 1;
	cpssp->fptags[3] = 1;
	cpssp->fptags[4] = 1;
	cpssp->fptags[5] = 1;
	cpssp->fptags[6] = 1;
	cpssp->fptags[7] = 1;
}

void
NAME_(helper_frstor)(struct cpssp *cpssp, target_ulong ptr, int data32)
{
	float80 tmp;
	int i;

	NAME_(helper_fldenv)(cpssp, ptr, data32);
	ptr += (14 << data32);

	for(i = 0;i < 8; i++) {
		tmp = NAME_(helper_fldt)(cpssp, ptr);
		ST(cpssp, i) = tmp;
		ptr += 10;
	}
}

void
NAME_(helper_fxsave)(struct cpssp *cpssp, target_ulong ptr, int data64)
{
	int fpus, fptag, i, nb_xmm_regs;
	float80 tmp;
	target_ulong addr;

	fpus = (cpssp->fpus & ~0x3800) | (cpssp->fpstt & 0x7) << 11;
	fptag = 0;
	for(i = 0; i < 8; i++) {
		fptag |= (cpssp->fptags[i] << i);
	}
	stw(cpssp, ptr, cpssp->fpuc);
	stw(cpssp, ptr + 2, fpus);
	stw(cpssp, ptr + 4, fptag ^ 0xff);

	if (data64) {
		stq(cpssp, ptr + 0x08, 0); /* rip */
		stq(cpssp, ptr + 0x10, 0); /* rdp */
	} else {
		stl(cpssp, ptr + 0x08, 0); /* eip */
		stl(cpssp, ptr + 0x0c, 0); /* sel */
		stl(cpssp, ptr + 0x10, 0); /* dp */
		stl(cpssp, ptr + 0x14, 0); /* sel */
	}

	addr = ptr + 0x20;
	for(i = 0;i < 8; i++) {
		tmp = ST(cpssp, i);
		NAME_(helper_fstt)(cpssp, tmp, addr);
		addr += 16;
	}

	if (cpssp->cr[4] & CPU_CR4_OSFXSR_MASK) {
		/* XXX: finish it */
		stl(cpssp, ptr + 0x18, cpssp->mxcsr); /* mxcsr */
		stl(cpssp, ptr + 0x1c, 0x0000ffff); /* mxcsr_mask */
		if (cpssp->hflags & HF_CS64_MASK) {
			nb_xmm_regs = 16;
		} else {
			nb_xmm_regs = 8;
		}
		addr = ptr + 0xa0;
		for(i = 0; i < nb_xmm_regs; i++) {
			stq(cpssp, addr, cpssp->xmm_regs[i].XMM_Q(0));
			stq(cpssp, addr + 8, cpssp->xmm_regs[i].XMM_Q(1));
			addr += 16;
		}
	}
}

void
NAME_(helper_fxrstor)(struct cpssp *cpssp, target_ulong ptr, int data64)
{
	int i, fpus, fptag, nb_xmm_regs;
	float80 tmp;
	target_ulong addr;

	cpssp->fpuc = lduw(cpssp, ptr);
	fpus = lduw(cpssp, ptr + 2);
	fptag = lduw(cpssp, ptr + 4);
	cpssp->fpstt = (fpus >> 11) & 7;
	cpssp->fpus = fpus & ~0x3800;
	fptag ^= 0xff;
	for(i = 0;i < 8; i++) {
		cpssp->fptags[i] = ((fptag >> i) & 1);
	}

	addr = ptr + 0x20;
	for(i = 0;i < 8; i++) {
		tmp = NAME_(helper_fldt)(cpssp, addr);
		ST(cpssp, i) = tmp;
		addr += 16;
	}

	if (cpssp->cr[4] & CPU_CR4_OSFXSR_MASK) {
		/* XXX: finish it */
		cpssp->mxcsr = ldl(cpssp, ptr + 0x18);
		//ldl(ptr + 0x1c);
		if (cpssp->hflags & HF_CS64_MASK) {
			nb_xmm_regs = 16;
		} else {
			nb_xmm_regs = 8;
		}
		addr = ptr + 0xa0;
		for(i = 0; i < nb_xmm_regs; i++) {
			cpssp->xmm_regs[i].XMM_Q(0) = ldq(cpssp, addr);
			cpssp->xmm_regs[i].XMM_Q(1) = ldq(cpssp, addr + 8);
			addr += 16;
		}
	}
}

void
NAME_(get_fp80)(uint64_t *pmant, uint16_t *pexp, float80 f)
{
	CPU86_LDoubleU temp;
	temp.d = f;
	*pmant = temp.l.lower;
	*pexp = temp.l.upper;
}

float80
NAME_(set_fp80)(uint64_t mant, uint16_t upper)
{
	CPU86_LDoubleU temp;
	temp.l.upper = upper;
	temp.l.lower = mant;
	return temp.d;
}

float
NAME_(approx_rsqrt)(float a)
{
	return 1.0 / sqrtl((float80) a);
}

float
NAME_(approx_rcp)(float a)
{
	return 1.0 / a;
}

void
NAME_(update_fp_status)(struct cpssp *cpssp)
{
	int rnd_type;

	/* set rounding mode */
	switch(cpssp->fpuc & RC_MASK) {
	default:
	case RC_NEAR:
		rnd_type = FE_TONEAREST;
		break;
	case RC_DOWN:
		rnd_type = FE_DOWNWARD;
		break;
	case RC_UP:
		rnd_type = FE_UPWARD;
		break;
	case RC_CHOP:
		rnd_type = FE_TOWARDZERO;
		break;
	}
	set_float_rounding_mode(rnd_type, &cpssp->fp_status);
#ifdef FLOATX80
	switch((cpssp->fpuc >> 8) & 3) {
	case 0:
		rnd_type = 32;
		break;
	case 2:
		rnd_type = 64;
		break;
	case 3:
	default:
		rnd_type = 80;
		break;
	}
	set_floatx80_rounding_precision(rnd_type, &cpssp->fp_status);
#endif
}

#endif /* BEHAVIOR */
