/* X stream viewer by Justin Schoeman
Based on XJ module hacked from xqcam.c by Paul Chinn <loomer@svpal.org>
*/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <linux/soundcard.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include "XJ.h"

#define SHMKEY 0x00
#ifdef NTSC
#define FPS ((double)(60.0/88100.0)) /* NTSC */
#else
#define FPS ((double)(50.0/88100.0)) /* PAL */
#endif

#define R(x) (unsigned char)(dbuf[(x*2)+1]&0xf8)
#define G(x) (unsigned char)(((dbuf[(x*2)+1]<<5)&0xe0)|((dbuf[x*2]>>3)&0x1c))
#define B(x) (unsigned char)((dbuf[x*2]<<3)&0xf8)

unsigned int lastf=0xffffffff, fpos=0;
int quit=0, *afield, afield_o=0;
pid_t apid=0;
int shmid=0;

/* Set a flag to exit the loop at the end */

void myalarm(int foo)
{
 if(fpos!=0)fpos+=2; /* stored in fields! */
 if(fpos>lastf)
 {
  quit=1;
 }

 if(afield_o!=afield[0]) /* we have a new field - sync to it */
 {
  afield_o=afield[0];
  fpos=afield_o;
 }
}

void quitprogram(int foo)
{
  if(shmid!=0)
  {
   shmctl(shmid, IPC_RMID, NULL);
   shmdt((char *)afield);
  }
  if (apid!=0) kill(apid, SIGINT);
  XJ_exit();
  fprintf(stderr, "\nExiting on signal %d\n", foo);
  exit(0);
}

void audioquit(int foo)
{
  shmdt((char *)afield);
  exit(0);
}

void usage(void)
{
  fprintf(stderr, "Usage:\n");
  fprintf(stderr, " xsv [-double] [-raw] <audio stream> <video stream 1> [<video stream 2> ...]");
  quitprogram(0);
}

int main(int argc, char **argv)
{
  char *sbuf, *dbuf;
  int width, height;
  int i, j, k;
  int astream, *vstream;
  int strmax, strcur=0;
  unsigned int field, tfield, ffield;
  struct itimerval ttime;
  unsigned char abuf[65536];
  int aread, f, dsp;
  int dble=0;
  int araw=0, ablk;
  unsigned long acnt=0;
  struct sigaction qsig={quitprogram, SIGALRM, SA_RESTART|SA_NOMASK, NULL};
  struct sigaction aqsig={audioquit, SIGALRM, SA_RESTART|SA_NOMASK, NULL};
  struct sigaction asig={myalarm, 0, SA_RESTART|SA_NOMASK, NULL};
  struct sigaction isig={SIG_IGN, 0, SA_RESTART|SA_NOMASK, NULL};


  sigaction(SIGHUP, &qsig, NULL);
  sigaction(SIGINT, &qsig, NULL);
  sigaction(SIGQUIT, &qsig, NULL);
  sigaction(SIGTERM, &qsig, NULL);
  sigaction(SIGSEGV, &qsig, NULL);

  /* Start shm */
  if((shmid=shmget(SHMKEY, sizeof(unsigned int), IPC_CREAT|IPC_EXCL|S_IRWXU))<0)
  {
   perror("shmget");
   exit(-1);
  }
  if((afield=(unsigned int *)shmat(shmid, NULL, 0))<0)
  {
   perror("shmat");
   quitprogram(0);
  }
  afield[0]=0;
  
  for(i=1; (i<argc)&&(argv[i][0]=='-'); i++)
  {
    if(strcmp(argv[i],"-double")==0)
    {
     fprintf(stderr, "Displaying double size.\n");
     dble=1;
    }
    else if(strcmp(argv[i],"-raw")==0)
    {
     fprintf(stderr, "Reading raw (processed) audio.\n");
     araw=1;
    } else usage();
  }
  
  if(argc<(i+2))usage();
  fprintf(stderr, "Reading audio from: %s .\n", argv[i]);
  astream=open(argv[i++], O_RDONLY, 0);
  strmax=argc-i;
  vstream=(int *)malloc(strmax*sizeof(int));
  if(!vstream)
  {
   perror("malloc vstream");
   quitprogram(0); 
  }
  fprintf(stderr, "Reading video from: ");
  for(;i<argc;i++)
  {
   fprintf(stderr, "%s ", argv[i]);
   vstream[strcur]=open(argv[i], O_RDONLY, 0);
   if(vstream[strcur++]<=0)
   {
    perror("open video file");
    quitprogram(0);
   }
  }
  strcur=0;  
  fprintf(stderr, "\n");
  read(vstream[0], &width, sizeof(width));
  read(vstream[0], &height, sizeof(height));
  read(vstream[0], &lastf, sizeof(lastf));
  read(vstream[0], &ffield, sizeof(unsigned int));
  lseek(vstream[0], -sizeof(unsigned int), SEEK_CUR);
  fprintf(stderr, "Picture size: %dx%d\n", width, height);
  fpos=ffield;

  dbuf=(unsigned char *)malloc(2*width*height);
  if(!dbuf)
  {
   perror("malloc dbuf");
   quitprogram(0);
  }

/* open audio */
  dsp=open("/dev/dsp", O_RDWR, 0);                   
  if(dsp<=0)
  {
   perror("dspopen");
   quitprogram(0);
  }
  f=16;
  if (ioctl(dsp, SNDCTL_DSP_SAMPLESIZE, &f)<0)
  {
   perror("samplesize");
   quitprogram(0);
  }
  f=0;
  if (ioctl(dsp, SNDCTL_DSP_STEREO, &f)<0)
  {
   perror("stereo");
   quitprogram(0);
  }
  f=44100;
  ioctl(dsp, SNDCTL_DSP_SPEED, &f);
  ioctl(dsp, SNDCTL_DSP_GETBLKSIZE, &ablk);

  /* Start X Display */
  if(dble==1)
  {
   if ((sbuf=XJ_init(2*width, 2*height, "X Stream Viewer", "XSV"))==NULL)
   {
    fprintf(stderr,"XJ_init failed, exiting\n");
    quitprogram(0);
   }
  } else 
  {
   if ((sbuf=XJ_init(width, height, "X Stream Viewer", "XSV"))==NULL)
   {
    fprintf(stderr,"XJ_init failed, exiting\n");
    quitprogram(0);
   }
  }

  ttime.it_interval.tv_sec=0;
#ifdef NTSC
  ttime.it_interval.tv_usec=33333; /* NTSC 30 fps */
#else
  ttime.it_interval.tv_usec=40000; /* PAL 25 fps */
#endif
  ttime.it_value.tv_sec=0;
#ifdef NTSC
  ttime.it_value.tv_usec=33333; /* NTSC */
#else
  ttime.it_value.tv_usec=40000; /* PAL */
#endif
  if(setitimer(ITIMER_REAL, &ttime, NULL)!=0)
  {
   perror("setitimer");
   quit=1;
  }  
  afield[0]=0;
  afield_o=0;

  sigaction(SIGALRM, &asig, NULL);
  
  if ((apid=fork())==0)
  { /* audio player */
   sigaction(SIGALRM, &isig, NULL);
   sigaction(SIGHUP, &aqsig, NULL);
   sigaction(SIGINT, &aqsig, NULL);
   sigaction(SIGQUIT, &aqsig, NULL);
   sigaction(SIGTERM, &aqsig, NULL);
   sigaction(SIGSEGV, &aqsig, NULL);
   
   if(araw==0)
   {
    while(read(astream, &aread, sizeof(aread))==sizeof(aread))
    {
     read(astream, &tfield, sizeof(tfield));
     read(astream, &abuf, aread);
     write(dsp, &abuf, aread);
#ifdef NTSC
     if((tfield-30)>ffield) afield[0]=tfield-30; /* NTSC */
#else
     if((tfield-25)>ffield) afield[0]=tfield-25; /* PAL */
#endif
    }
   } else
   {
    while((tfield=read(astream, &abuf, ablk))>0)
    {
     write(dsp, &abuf, tfield);
     acnt+=tfield;
     tfield=ffield+(unsigned int)(FPS*(double)acnt);
#ifdef NTSC
     if((tfield-30)>ffield) afield[0]=tfield-30; /* NTSC */
#else
     if((tfield-25)>ffield) afield[0]=tfield-25; /* PAL */
#endif
    }
   }
   exit(0);
  }

  
  while(!quit)
  {
   read(vstream[strcur], &field, sizeof(field));
   if((field+2)<fpos) /* falling behind */
   {
    lseek(vstream[strcur++], width*height*2, SEEK_CUR);
    if(strcur==strmax)strcur=0;
     fprintf(stderr, "falling behind - skipping forward %d - %u\n", field, fpos);
   } else
   {
    read(vstream[strcur++], dbuf, width*height*2);
    if(strcur==strmax)strcur=0;
   
    if(dble==1)
    {
     k=0;
     switch(XJ_depth)
     {
      case 8:
       for(i=0; i<(2*height); i+=2)
        for(j=0; j<(2*width); j+=2)
        {
         sbuf[(i*2*width)+j]=sbuf[((i+1)*2*width)+j]=sbuf[((i+1)*2*width)+j+1]=sbuf[(i*2*width)+j+1]=
          XJ_greys[(unsigned char)((int)(R(k)+G(k)+B(k))/3)];
         k++;
        }
       break;
      case 16:
       for(i=0; i<(2*height); i+=2)
        for(j=0; j<(4*width); j+=4)
        {
         sbuf[(i*4*width)+j+1]=sbuf[((i+1)*4*width)+j+1]=sbuf[((i+1)*4*width)+j+3]=sbuf[(i*4*width)+j+3]=dbuf[(k*2)+1];
         sbuf[(i*4*width)+j+0]=sbuf[((i+1)*4*width)+j+0]=sbuf[((i+1)*4*width)+j+2]=sbuf[(i*4*width)+j+2]=dbuf[(k*2)];
         k++;
        }
       break;
      case 24:
#ifndef FIX32
       for(i=0; i<(2*height); i+=2)
        for(j=0; j<(6*width); j+=6)
        {
         sbuf[(i*6*width)+j+2]=sbuf[((i+1)*6*width)+j+2]=sbuf[((i+1)*6*width)+j+5]=sbuf[(i*6*width)+j+5]=R(k);
         sbuf[(i*6*width)+j+1]=sbuf[((i+1)*6*width)+j+1]=sbuf[((i+1)*6*width)+j+4]=sbuf[(i*6*width)+j+4]=G(k);
         sbuf[(i*6*width)+j+0]=sbuf[((i+1)*6*width)+j+0]=sbuf[((i+1)*6*width)+j+3]=sbuf[(i*6*width)+j+3]=B(k);
         k++;
        }
       break;
#endif
      case 32:
       for(i=0; i<(2*height); i+=2)
        for(j=0; j<(8*width); j+=8)
        {
         sbuf[(i*8*width)+j+2]=sbuf[((i+1)*8*width)+j+2]=sbuf[((i+1)*8*width)+j+6]=sbuf[(i*8*width)+j+6]=R(k);
         sbuf[(i*8*width)+j+1]=sbuf[((i+1)*8*width)+j+1]=sbuf[((i+1)*8*width)+j+5]=sbuf[(i*8*width)+j+5]=G(k);
         sbuf[(i*8*width)+j+0]=sbuf[((i+1)*8*width)+j+0]=sbuf[((i+1)*8*width)+j+4]=sbuf[(i*8*width)+j+4]=B(k);
         k++;
        }
       break;
      default: break;
     }
    } else
    {
     switch(XJ_depth)
     {
      case 8:
       for(i=0; i<width*height; i++)
       {
        sbuf[i]=XJ_greys[(unsigned char)((int)(R(i)+G(i)+B(i))/3)];
       }
       break;
      case 16:
       for(i=0; i<(width*height*2)/sizeof(unsigned long); i++)
        ((unsigned long *)sbuf)[i]=((unsigned long *)dbuf)[i];
       break;
      case 24:
#ifndef FIX32
       for(i=0; i<width*height; i++)
       {
        sbuf[(i*3)+2]=R(i);
        sbuf[(i*3)+1]=G(i);
        sbuf[(i*3)+0]=B(i);
       }
       break;
#endif
      case 32:
       for(i=0; i<width*height; i++)
       {
        sbuf[(i*4)+2]=R(i);
        sbuf[(i*4)+1]=G(i);
        sbuf[(i*4)+0]=B(i);
       }
       break;
      default: break;
     }
    }
    while(field>fpos)
    {
     fprintf(stderr, "current: %u desired: %u\r", field, fpos);
     pause();
    }

    XJ_show();
   }
  }
  quitprogram(0);
  return 0;
}
