/*
 *  charset.c
 *  Copyright(C) 2003- Masahito Omote <omote@utyuuzin.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include "charset.h"
#include "intl.h"

gchar *charset_convert(const gchar *in,
		       const gchar *incode,
		       const gchar *outcode)
{
    gsize rbytes, wbytes;
    gchar *out;
    GError *error = NULL;

    g_return_val_if_fail(in, NULL);
    g_return_val_if_fail(incode && *incode, g_strdup(in));
    g_return_val_if_fail(outcode && *outcode, g_strdup(in));

    out = g_convert(in, -1, outcode, incode, &rbytes, &wbytes, &error);
    if(error != NULL) {
	g_printerr("g_convert failed: %s\n", error->message);
	out = NULL;
    }
    return out;
}

gchar *utf8_convert(const gchar *in) {
    gsize rbytes, wbytes;
    gchar *out;
    GError *error = NULL;

    g_return_val_if_fail(in, NULL);

    out = g_locale_to_utf8(in, -1, &rbytes, &wbytes, &error);
    if(out == NULL && error != NULL) {
	if(g_utf8_validate(in, -1, NULL)) {
	    out = g_strdup(in);
	} else {
	    g_printerr("g_locale_to_utf8 failed: %s\n", error->message);
	    out = NULL;
	}
    }
    return out;
}

gchar *utf8_to_eucjp(const gchar *utf8) {
    gchar *eucjp;
    g_return_val_if_fail(utf8, NULL);

    eucjp = charset_convert(utf8, "UTF-8", "EUC-JP");
    return eucjp;
}

gchar *eucjp_to_utf8(const gchar *eucjp) {
    gchar *utf8;
    g_return_val_if_fail(eucjp, NULL);
    
    utf8 = charset_convert(eucjp, "EUC-JP", "UTF-8");
    return utf8;
}
