/*
 *  cannacontainer.c
 *  Copyright(C) 2003- Masahito Omote <omote@utyuuzin.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gtk/gtk.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "cannacontainer.h"
#include "wordlist_view.h"
#include "charset.h"
#include "intl.h"
#include "canna.h"
#include "cannadic.h"
#include "dixchange_dic.h"
#include "posdialog.h"

static char *cclass_code_broad[] = {
    N_("Verb"),
    N_("Substantive"),
    N_("Adjective"),
    N_("Adverb"),
    N_("Etc"),
};

static gchar *dictionary_type[] = {
	"DixChange",
	"cannadic",
/*	"ATOK",
	"MS-IME", */
};

static void cb_canna_viewwordpane_button_load_clicked   (GtkButton *,
						         CannaViewWordPane *);
static void cb_canna_viewwordpane_button_delete_clicked (GtkButton *,
							 CannaViewWordPane *);
static void cb_button_cclass_browse_clicked             (GtkButton *,
							 CannaAddWordPane *);

static void cb_add_button_clicked   (GtkButton *, CannaAddWordPane *);
static void cb_clear_button_clicked (GtkButton *, CannaAddWordPane *);

static gboolean cb_searchword_focusout (GtkWidget *, GdkEventFocus *, CannaViewWordPane *);

static void dixchange_import (const char* filename, int type) { g_print("dixchange_import called\n"); }
static void cb_import_button_clicked(GtkButton *button, CannaImportWordPane *pane);
static void cb_export_button_clicked(GtkButton *button, CannaImportWordPane *pane);
static void cb_browse_button_clicked(GtkButton *button, GtkEntry *entry);

void create_canna_container(CannaContainer *container) {
    /* I changed coding style a little. */
    GtkWidget *notebook;
    CannaViewWordPane *viewwordpane;
    CannaAddWordPane *addwordpane;
    CannaImportWordPane *importwordpane;
    GtkWidget *label_viewword, *label_addword, *label_importword;

    notebook = gtk_notebook_new();
    viewwordpane = canna_viewword_pane_new();
    addwordpane = canna_addword_pane_new();
    importwordpane = canna_importword_pane_new();

    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
    gtk_container_set_border_width(GTK_CONTAINER(notebook), 10);

    gtk_container_add(GTK_CONTAINER(notebook), viewwordpane->pane);
    gtk_widget_show(viewwordpane->pane);
    label_viewword = gtk_label_new(_("View word"));
    gtk_widget_show(label_viewword);
    gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebook),
			       gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 0),
			       label_viewword);

    gtk_container_add(GTK_CONTAINER(notebook), addwordpane->pane);
    gtk_widget_show(addwordpane->pane);
    label_addword = gtk_label_new(_("Add word"));
    gtk_widget_show(label_addword);
    gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebook),
			       gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 1),
			       label_addword);

    gtk_container_add(GTK_CONTAINER(notebook), importwordpane->pane);
    gtk_widget_show(importwordpane->pane);
    label_importword = gtk_label_new(_("Import/Export"));
    gtk_widget_show(label_viewword);
    gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebook),
			       gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 2),
			       label_importword);

    container->container = notebook;
    container->addwordpane = addwordpane;
    container->viewwordpane = viewwordpane;
    container->importwordpane = importwordpane;
}

CannaAddWordPane *canna_addword_pane_new(void) {
    CannaAddWordPane *pane;
    GtkWidget *hbox;
    GtkWidget *vbox1, *vbox2;
    GtkWidget *table1;
    GtkWidget *label_phon, *label_desc;
    GtkWidget *label_cclass_code_narrow, *label_cclass_code_broad;
    GtkWidget *optmenu_cclass_code;
    GtkWidget *alignment_cclass_code;
    GtkWidget *menu_cclass_code, *menuitem_cclass_code;
    GtkWidget *entry_phon, *entry_desc, *entry_cclass_code_narrow;
    GtkWidget *button_cclass_browse;
    GtkWidget *button_add, *button_clear;
    int cclass_code_num;
    int i;

    cclass_code_num = sizeof(cclass_code_broad)/sizeof(cclass_code_broad[0]);

    pane = g_malloc0(sizeof(CannaAddWordPane));

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 15);

    vbox1 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox1);
    gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);

    table1 = gtk_table_new(5, 5, FALSE);
    gtk_widget_show(table1);
    gtk_box_pack_start(GTK_BOX(vbox1), table1, FALSE, FALSE, 0);

    label_phon = gtk_label_new(_("Phonetic:"));
    gtk_misc_set_alignment(GTK_MISC(label_phon), 1.0, 0.5);
    gtk_widget_show(label_phon);
    gtk_table_attach(GTK_TABLE(table1), label_phon,
		     0, 1, 0, 1, GTK_FILL, 0, 5, 5);

    entry_phon = gtk_entry_new();
    gtk_table_attach(GTK_TABLE(table1), entry_phon,
		     1, 2, 0, 1, 0, 0, 5, 5);
    gtk_widget_show(entry_phon);

    label_desc = gtk_label_new(_("Literal:"));
    gtk_misc_set_alignment(GTK_MISC(label_desc), 1.0, 0.5);
    gtk_widget_show(label_desc);
    gtk_table_attach(GTK_TABLE(table1), label_desc,
		     0, 1, 1, 2, GTK_FILL, 0, 5, 5);

    entry_desc = gtk_entry_new();
    gtk_table_attach(GTK_TABLE(table1), entry_desc,
		     1, 2, 1, 2, 0, 0, 5, 5);
    gtk_widget_show(entry_desc);

    label_cclass_code_broad = gtk_label_new(_("Part of Speech(broad):"));
    gtk_misc_set_alignment(GTK_MISC(label_cclass_code_broad), 1.0, 0.5);
    gtk_widget_show(label_cclass_code_broad);
    gtk_table_attach(GTK_TABLE(table1), label_cclass_code_broad,
		     0, 1, 2, 3, GTK_FILL, 0, 5, 5);

    optmenu_cclass_code = gtk_option_menu_new();
    pane->optmenu_cclass_code = optmenu_cclass_code;
    menu_cclass_code = gtk_menu_new();
    for(i = 0; i < cclass_code_num; i++) {
	menuitem_cclass_code = gtk_menu_item_new_with_label(_(cclass_code_broad[i]));
/*
	g_signal_connect(G_OBJECT(menuitem_cclass_code), "activate",
			 G_CALLBACK(cb_toggle_cclass_code), pane);
*/
	gtk_menu_shell_append(GTK_MENU_SHELL(menu_cclass_code),
			      menuitem_cclass_code);
	gtk_widget_show(menuitem_cclass_code);
    }

    gtk_option_menu_set_menu(GTK_OPTION_MENU(optmenu_cclass_code),
			     menu_cclass_code);
    gtk_widget_show(optmenu_cclass_code);

    alignment_cclass_code = gtk_alignment_new(0, 0.5, 0, 0);
    gtk_container_add(GTK_CONTAINER(alignment_cclass_code),
		      optmenu_cclass_code);
    gtk_widget_show(alignment_cclass_code);
    gtk_table_attach(GTK_TABLE(table1), alignment_cclass_code,
		     1, 2, 2, 3, GTK_FILL, GTK_FILL, 5, 5);

    label_cclass_code_narrow = gtk_label_new(_("Part of Speech(narrow):"));
    gtk_misc_set_alignment(GTK_MISC(label_cclass_code_narrow), 1.0, 0.5);
    gtk_widget_show(label_cclass_code_narrow);
    gtk_table_attach(GTK_TABLE(table1), label_cclass_code_narrow,
		     0, 1, 3, 4, GTK_FILL, 0, 5, 5);

    entry_cclass_code_narrow = gtk_entry_new();
    gtk_widget_show(entry_cclass_code_narrow);
    gtk_table_attach(GTK_TABLE(table1), entry_cclass_code_narrow,
		     1, 2, 3, 4, GTK_FILL, 0, 5, 5);

    button_cclass_browse = gtk_button_new_with_label(_("Browse..."));
    gtk_widget_show(button_cclass_browse);
    gtk_table_attach(GTK_TABLE(table1), button_cclass_browse,
		     2, 3, 3, 4, GTK_FILL, 0, 5, 5);

    g_signal_connect(G_OBJECT(button_cclass_browse), "clicked",
		     G_CALLBACK(cb_button_cclass_browse_clicked), pane);

    vbox2 = gtk_vbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(vbox2), GTK_BUTTONBOX_START);
    gtk_box_set_spacing(GTK_BOX(vbox2), 10);
    gtk_widget_show(vbox2);
    gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 50);

    button_add = gtk_button_new_with_label(_("Add"));
    gtk_box_pack_start(GTK_BOX(vbox2), button_add, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(button_add), "clicked",
		     G_CALLBACK(cb_add_button_clicked), pane);
    gtk_widget_show(button_add);

    button_clear = gtk_button_new_with_label(_("Clear"));
    gtk_box_pack_start(GTK_BOX(vbox2), button_clear, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(button_clear), "clicked",
		     G_CALLBACK(cb_clear_button_clicked), pane);
    gtk_widget_show(button_clear);

    pane->pane = GTK_WIDGET(hbox);
    pane->phon = GTK_WIDGET(entry_phon);
    pane->desc = GTK_WIDGET(entry_desc);
    pane->cclass_code = GTK_WIDGET(entry_cclass_code_narrow);
    pane->add = GTK_WIDGET(button_add);
    pane->clear = GTK_WIDGET(button_clear);
    return pane;
}

CannaViewWordPane *canna_viewword_pane_new(void) {
    CannaViewWordPane *pane;
    GtkWidget *hbox1, *hbox2;
    GtkWidget *vbox1, *vbox2;
    GtkWidget *wordlist;
    GtkWidget *button_load, *button_delete;
    GtkWidget *label;
    GtkWidget *searchword;
    pane = g_malloc0(sizeof(CannaViewWordPane));

    hbox1 = gtk_hbox_new(FALSE, 10);
    gtk_container_set_border_width(GTK_CONTAINER(hbox1), 15);

    vbox1 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox1);
    gtk_box_pack_start(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 0);

    hbox2 = gtk_hbox_new(FALSE, 10);
    gtk_widget_show(hbox2);
    gtk_box_pack_start(GTK_BOX(vbox1), hbox2, FALSE, FALSE, 0);

    label = gtk_label_new_with_mnemonic(_("Search:"));
    gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 5);
    gtk_widget_show(label);

    searchword = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox2), searchword, FALSE, FALSE, 5);
    gtk_widget_show(searchword);
    gtk_widget_set_sensitive(searchword, FALSE);

    g_signal_connect(G_OBJECT(searchword), "focus-out-event",
		    	G_CALLBACK(cb_searchword_focusout), pane);

    wordlist = wordlist_view_new_with_attributes("editable", FALSE,
						 "freq_show", FALSE,
						 "cclass_code_show", TRUE,
						 "selection_mode", GTK_SELECTION_MULTIPLE,
						 NULL);
    gtk_widget_show(wordlist);
    gtk_box_pack_start(GTK_BOX(vbox1), wordlist, TRUE, TRUE, 10);

    vbox2 = gtk_vbutton_box_new();
    gtk_widget_show(vbox2);
    gtk_box_pack_start(GTK_BOX(hbox1), vbox2, FALSE, FALSE, 10);

    button_load = gtk_button_new_with_label(_("Load"));
    gtk_widget_show(button_load);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(vbox2), GTK_BUTTONBOX_START);
    gtk_box_set_spacing(GTK_BOX(vbox2), 10);
    gtk_box_pack_start(GTK_BOX(vbox2), button_load, FALSE, FALSE, 10);

    g_signal_connect(G_OBJECT(button_load), "clicked",
		     G_CALLBACK(cb_canna_viewwordpane_button_load_clicked),
		     pane);

    button_delete = gtk_button_new_with_label(_("Delete"));
    gtk_widget_show(button_delete);
    gtk_box_pack_start(GTK_BOX(vbox2), button_delete, FALSE, FALSE, 10);
    gtk_widget_set_sensitive(button_delete, FALSE);

    g_signal_connect(G_OBJECT(button_delete), "clicked",
		     G_CALLBACK(cb_canna_viewwordpane_button_delete_clicked),
		     pane);



    pane->pane = hbox1;
    pane->wordlist_canna = wordlist;
    pane->button_load = button_load;
    pane->button_delete = button_delete;
    pane->searchword = searchword;
    return pane;
}

CannaImportWordPane *canna_importword_pane_new(void) {
    CannaImportWordPane *pane;

    GtkWidget *vbox1, *vbox2;
    GtkWidget *hbox1, *hbox2;
    GtkWidget *hbox3, *hbox4;
    GtkWidget *label1, *label2;
    GtkWidget *frame1, *frame2;
    GtkWidget *menu, *menuitem;
    gint i, dictionary_type_num;

    pane = g_malloc0(sizeof(CannaImportWordPane));
    dictionary_type_num = sizeof(dictionary_type) / sizeof(dictionary_type[0]);

    pane->pane = gtk_hbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(pane->pane), 15);

    vbox1 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox1);
    gtk_box_pack_start(GTK_BOX(pane->pane), vbox1, FALSE, FALSE, 0);

    /* Import */	
    frame1 = gtk_frame_new(_("Import"));
    gtk_widget_show(frame1);
    gtk_box_pack_start(GTK_BOX(vbox1), frame1, FALSE, FALSE, 0);
    
    hbox2 = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(hbox2);
    gtk_container_set_border_width(GTK_CONTAINER(hbox2), 10);
    gtk_container_add(GTK_CONTAINER(frame1), hbox2);
    
    label1 = gtk_label_new_with_mnemonic(_("Filename:"));
    gtk_widget_show(label1);
    gtk_box_pack_start(GTK_BOX(hbox2), label1, FALSE, FALSE, 0);
    
    pane->importfilename = gtk_entry_new();
    gtk_widget_show(pane->importfilename);
    gtk_box_pack_start(GTK_BOX(hbox2), pane->importfilename, FALSE, FALSE, 5);
	
    pane->button_import_getfilename = gtk_button_new_with_label(_("Browse..."));
    gtk_widget_show(pane->button_import_getfilename);
    gtk_box_pack_start(GTK_BOX(hbox2), pane->button_import_getfilename, FALSE, FALSE, 10);
    g_signal_connect(G_OBJECT(pane->button_import_getfilename), "clicked",
		     G_CALLBACK(cb_browse_button_clicked), GTK_ENTRY(pane->importfilename));

    pane->opt_dic_import_type = gtk_option_menu_new();
    gtk_widget_show(pane->opt_dic_import_type);
	
    menu = gtk_menu_new();
    for(i = 0; i < dictionary_type_num; i++) {
	menuitem = gtk_menu_item_new_with_label(dictionary_type[i]);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);
	gtk_widget_show(menuitem);
    }
    gtk_option_menu_set_menu(GTK_OPTION_MENU(pane->opt_dic_import_type), menu);
    gtk_box_pack_start(GTK_BOX(hbox2), pane->opt_dic_import_type, FALSE, FALSE, 10);
    


    pane->button_doimport = gtk_button_new_with_label(_("Import"));
    gtk_box_pack_start(GTK_BOX(hbox2), pane->button_doimport, FALSE, FALSE, 20);
    gtk_widget_show(pane->button_doimport);
    g_signal_connect(G_OBJECT(pane->button_doimport), "clicked",
		     G_CALLBACK(cb_import_button_clicked), pane);
    

    /* Export */
    frame2 = gtk_frame_new(_("Export"));
    gtk_widget_show(frame2);
    gtk_box_pack_start(GTK_BOX(vbox1), frame2, FALSE, FALSE, 10);
	
    hbox3 = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(hbox3);
    gtk_container_set_border_width(GTK_CONTAINER(hbox3), 10);
    gtk_container_add(GTK_CONTAINER(frame2), hbox3);
    
    label2 = gtk_label_new_with_mnemonic(_("Filename:"));
    gtk_widget_show(label2);
    gtk_box_pack_start(GTK_BOX(hbox3), label2, FALSE, FALSE, 0);
	
    pane->exportfilename = gtk_entry_new();
    gtk_widget_show(pane->exportfilename);
    gtk_box_pack_start(GTK_BOX(hbox3), pane->exportfilename, FALSE, FALSE, 5);
	
    pane->button_export_getfilename = gtk_button_new_with_label(_("Browse..."));
    gtk_widget_show(pane->button_export_getfilename);
    gtk_box_pack_start(GTK_BOX(hbox3), pane->button_export_getfilename, FALSE, FALSE, 10);
    g_signal_connect(G_OBJECT(pane->button_export_getfilename), "clicked",
		     G_CALLBACK(cb_browse_button_clicked), GTK_ENTRY(pane->exportfilename));


    pane->opt_dic_export_type = gtk_option_menu_new();
    gtk_widget_show(pane->opt_dic_export_type);

    menu = gtk_menu_new();
    for(i = 0; i < dictionary_type_num; i++) {
	menuitem = gtk_menu_item_new_with_label(dictionary_type[i]);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);
	gtk_widget_show(menuitem);
    }
    gtk_option_menu_set_menu(GTK_OPTION_MENU(pane->opt_dic_export_type), menu);
    gtk_box_pack_start(GTK_BOX(hbox3), pane->opt_dic_export_type, FALSE, FALSE, 10);
    
    pane->button_doexport = gtk_button_new_with_label(_("Export"));
    gtk_box_pack_start(GTK_BOX(hbox3), pane->button_doexport, FALSE, FALSE, 20);
    gtk_widget_show(pane->button_doexport);

    g_signal_connect(G_OBJECT(pane->button_doexport), "clicked",
		     G_CALLBACK(cb_export_button_clicked), pane);



    return pane;
}

void show_canna_container(CannaContainer *container, gboolean flags) {
    if(flags) {
	gtk_widget_show(container->container);
    }
}

/* callbacks */
static void cb_canna_viewwordpane_button_load_clicked(GtkButton *button,
						      CannaViewWordPane *pane)
{
    word *list = NULL, *pos = NULL;

    wordlist_view_clear(WORDLIST_VIEW(pane->wordlist_canna));
    list = canna_get_word_text_priv_dic();
    if(list == NULL)
	return;

    for(pos = list; pos != NULL; pos = pos->next) {
/* for test */
	gchar *eucjp_phonetic = NULL, *eucjp_literal = NULL;
	gchar *eucjp_pos = NULL;

	eucjp_phonetic = pos->phon;
	eucjp_literal = pos->desc;
	eucjp_pos = pos->cclass_code;
	if(eucjp_phonetic != NULL)
	    pos->phon = eucjp_to_utf8(eucjp_phonetic);
	if(eucjp_literal != NULL)
	    pos->desc = eucjp_to_utf8(eucjp_literal);
	if(eucjp_pos != NULL)
	    pos->cclass_code = eucjp_to_utf8(find_pos_from_code(eucjp_pos, SUPPORT_CANNA));

	if(pos->phon != NULL && pos->desc != NULL &&
	   pos->cclass_code != NULL) {
	    wordlist_view_set_values(WORDLIST_VIEW(pane->wordlist_canna),
				     pos);
	    if(eucjp_phonetic != NULL)
		free(eucjp_phonetic);
	    if(eucjp_literal != NULL)
		free(eucjp_literal);
	    if(eucjp_pos != NULL)
		free(eucjp_pos);
	}
    }
    word_free_list(list);
    gtk_button_set_label(button, _("Reload"));
    gtk_widget_set_sensitive(pane->button_delete, TRUE);
    gtk_widget_set_sensitive(pane->searchword, TRUE);
}

static
void cb_canna_viewwordpane_button_delete_clicked (GtkButton *button,
						  CannaViewWordPane *pane)
{
    GList *list, *pos;
    word *data;
    char *utf8_wordinfo, *eucjp_wordinfo, *code, *eucjp_pass;
    int phonlen, desclen, wordinfolen, codelen;
    int ret;

    list = wordlist_view_get_selected_data_list(WORDLIST_VIEW(pane->wordlist_canna));

    for(pos = g_list_first(list); pos != NULL; pos = g_list_next(pos)) {
	data = pos->data;
	eucjp_pass = utf8_to_eucjp(data->cclass_code);
	code = find_code_from_pos(eucjp_pass, SUPPORT_CANNA);
	phonlen = strlen(data->phon);
	desclen = strlen(data->desc);
	codelen = strlen(code);
	wordinfolen = phonlen + desclen + codelen + 2 + 1;
	utf8_wordinfo = g_malloc(sizeof(char) * wordinfolen);

	snprintf(utf8_wordinfo, wordinfolen, "%s %s %s",
		 data->phon, code, data->desc);
	eucjp_wordinfo = utf8_to_eucjp(utf8_wordinfo);
	ret = canna_delete_dic("user", eucjp_wordinfo);
	free(code);
	free(eucjp_pass);
	g_free(eucjp_wordinfo);
	g_free(utf8_wordinfo);
    }
    /* TODO: must implement free(list) func. */
    wordlist_view_remove_selected_data(WORDLIST_VIEW(pane->wordlist_canna));
}

static void cb_button_cclass_browse_clicked(GtkButton *button,
					    CannaAddWordPane *pane)
{
    gint id;
    gchar *code;
    id = gtk_option_menu_get_history(GTK_OPTION_MENU(pane->optmenu_cclass_code));
    code = cannadic_pos_dialog(id, SUPPORT_CANNA);
    if(code != NULL) {
	gtk_entry_set_text(GTK_ENTRY(pane->cclass_code), code);
	g_free(code);
    }
}

static void cb_add_button_clicked (GtkButton *button,
				   CannaAddWordPane *pane)
{
    const gchar *utf8_phon, *utf8_desc, *utf8_cclass_code;
    gchar *utf8_word_line;
    gint word_len;
    unsigned char *eucjp_word_line;
    int ret;

    utf8_phon = gtk_entry_get_text(GTK_ENTRY(pane->phon));
    utf8_desc = gtk_entry_get_text(GTK_ENTRY(pane->desc));
    utf8_cclass_code = gtk_entry_get_text(GTK_ENTRY(pane->cclass_code));

    word_len = strlen(utf8_phon) + strlen(utf8_desc) +
	       strlen(utf8_cclass_code) + 2 + 1;
    utf8_word_line = g_malloc(sizeof(gchar) * word_len);
    if(utf8_word_line != NULL) {
	g_snprintf(utf8_word_line, word_len, "%s %s %s",
		   utf8_phon, utf8_cclass_code, utf8_desc);

	eucjp_word_line = utf8_to_eucjp(utf8_word_line);
	if(eucjp_word_line != NULL) {
	    ret = canna_define_dic("user", eucjp_word_line);
	    g_free(eucjp_word_line);
	}
	g_free(utf8_word_line);
    }

    gtk_entry_set_text(GTK_ENTRY(pane->phon), "");
    gtk_entry_set_text(GTK_ENTRY(pane->desc), "");
    gtk_entry_set_text(GTK_ENTRY(pane->cclass_code), "");
    gtk_option_menu_set_history(GTK_OPTION_MENU(pane->optmenu_cclass_code), 0);

    g_free(utf8_phon);
    g_free(utf8_desc);
    g_free(utf8_cclass_code);
}

static void cb_clear_button_clicked (GtkButton *button,
				     CannaAddWordPane *pane)
{
    gtk_entry_set_text(GTK_ENTRY(pane->phon), "");
    gtk_entry_set_text(GTK_ENTRY(pane->desc), "");
    gtk_entry_set_text(GTK_ENTRY(pane->cclass_code), "");
    gtk_option_menu_set_history(GTK_OPTION_MENU(pane->optmenu_cclass_code), 0);
}

static void cb_browse_button_clicked(GtkButton *button, GtkEntry *entry) {
	GtkWidget *filebrowser;
	gint ret;
	const gchar *filename;

	filebrowser = gtk_file_selection_new(_("Select File"));
	ret = gtk_dialog_run(GTK_DIALOG(filebrowser));

	switch(ret) {
		case GTK_RESPONSE_OK:
			filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filebrowser));
			gtk_entry_set_text(entry, filename);
			gtk_widget_destroy(filebrowser);
			break;
		case GTK_RESPONSE_CANCEL:
			gtk_widget_destroy(filebrowser);
			break;
	}
}

static void cb_export_button_clicked(GtkButton *button, CannaImportWordPane *pane)
{
	const gchar *filename;
	gint type, ret;
	int (*export_commands[])(const char*, int) = {
		dixchange_export,
		cannadic_export,
/*		atok_export,
		msime_export, */
	};

	filename = gtk_entry_get_text(GTK_ENTRY(pane->exportfilename));
	type = gtk_option_menu_get_history(GTK_OPTION_MENU(pane->opt_dic_export_type));

	if(filename != NULL && filename[0] != '\0')
		ret = export_commands[type](filename, 1);
}

static void cb_import_button_clicked(GtkButton *button, CannaImportWordPane *pane)
{
	const gchar *filename;
	gint type, ret;
	int (*import_commands[])(const char*, int) = {
		dixchange_import,
		cannadic_import,
/*		atok_import,
		msime_import, */
	};

	filename = gtk_entry_get_text(GTK_ENTRY(pane->importfilename));
	type = gtk_option_menu_get_history(GTK_OPTION_MENU(pane->opt_dic_import_type));

	if(filename != NULL && filename[0] != '\0')
		ret = import_commands[type](filename, 1);
}



static gboolean cb_searchword_focusout(GtkWidget *widget, GdkEventFocus *focus, CannaViewWordPane *pane)
{
    int num;
    gchar *searchword;
    gchar *eucjp_phonetic, *eucjp_literal, *eucjp_pos;
    word *list = NULL, *pos = NULL;

    searchword = gtk_editable_get_chars (GTK_EDITABLE(pane->searchword),0,-1);

    wordlist_view_clear(WORDLIST_VIEW(pane->wordlist_canna));
    if(searchword != NULL) {
	num = read_anthy_priv_dic_list(&list);
	if(list == NULL)
	    return;
	
	for(pos = list; pos != NULL; pos = pos->next) {
	    eucjp_phonetic = pos->phon;
	    eucjp_literal = pos->desc;
	    eucjp_pos = pos->cclass_code;
	    pos->phon = eucjp_to_utf8(eucjp_phonetic);
	    pos->desc = eucjp_to_utf8(eucjp_literal);
	    pos->cclass_code = eucjp_to_utf8(find_pos_from_code(eucjp_pos, SUPPORT_ANTHY));
	    g_free(eucjp_phonetic);
	    g_free(eucjp_literal);
	    g_free(eucjp_pos);
	    if(strstr(pos->phon, searchword) != NULL) {
		wordlist_view_set_values(WORDLIST_VIEW(pane->wordlist_canna), pos);
	    }
	}
	word_free_list(list);

        g_free(searchword);
    }
    return FALSE;
}
