/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The plex86 developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include "bios.h"
#include "multiboot.h"


/************************************************************************/
/* Main code                                                            */
/************************************************************************/

int
bin_load (char *guest_file_name, Bit32u address)
{
    enum { TYPE_BIN, TYPE_ELF } type;
    int virtno;
    struct stat stat_buf;
    char *image;


    /* load guest code */

    fprintf (stderr, "bios: loading guest code: %s ", guest_file_name);

    virtno = open (guest_file_name, O_RDONLY);
    if (virtno < 0)
    {
        fprintf(stderr, "\nbios: open %s: %s\n", guest_file_name, strerror(errno));
	return 1;
    }

    if (fstat (virtno, &stat_buf) != 0)
    {
        fprintf(stderr, "\nbios: fstat %s: %s\n", guest_file_name, strerror(errno));
	return 1;
    }

    fprintf (stderr, "(%lu bytes) ", stat_buf.st_size);

    if ((image = (void *) malloc (stat_buf.st_size)) == NULL)
    {
	perror ("bios: malloc");
	return 1;
    }

    if (read (virtno, image, stat_buf.st_size) != stat_buf.st_size)
    {
	perror ("bios: read");
	return 1;
    }

    close (virtno);



    /* Check whether binary is an ELF executable */

    if (image[0] == 0x7f && 
        image[1] == 'E' && image[2] == 'L' && image[3] == 'F')
    {
        if (address != -1) {
          fprintf(stderr, "ELF kernel image specifies load address:"
                          "don't pass this as an argument\n");
          return 1;
          }
	fprintf (stderr, "(ELF)\n");
	type = TYPE_ELF;
    }
    else
    {
        if (address == -1) {
          fprintf(stderr, "binary kernel image requires load address\n");
          return 1;
          }
	fprintf (stderr, "(BIN)\n");
	type = TYPE_BIN;
    }


    /* Load image into guest address space */

    switch (type)
    {
    case TYPE_BIN:
        /* Load complete binary image */
        vm_conf.text_address = address;
        if (pluginWritePhyMem(vm_conf.text_address, stat_buf.st_size, image))
        {
            fprintf (stderr, "bios: trying to load beyond available VM memory.\n");
            return 1;
        }  
        break;

    case TYPE_ELF:
    {
        Elf32_Ehdr *eh = (Elf32_Ehdr *) image;
        Elf32_Shdr *sh = (Elf32_Shdr *)(image + eh->e_shoff);
        int i;

	if (eh->e_type != ET_EXEC)
	{
	    fprintf (stderr, "bios: ELF file is not an executable\n");
	    return 1;
	}

	if (eh->e_machine != EM_386)
	{
	    fprintf (stderr, "bios: plex86 isn't able to run non-x86 binaries\n");
	    fprintf (stderr, "bios: please use an emulator for your binary!\n");
	    return 1;
	}

        /* Load all sections that occupy space and have bits */
	for (i = 0; i < eh->e_shnum; i++)
	    if ((sh[i].sh_flags & SHF_ALLOC) && (sh[i].sh_type != SHT_NOBITS))
                if (pluginWritePhyMem(sh[i].sh_addr, sh[i].sh_size, 
                                       image + sh[i].sh_offset))
                {
                    fprintf (stderr, "bios: trying to load beyond available VM memory.\n");
                    return 1;
                }  

        /* Retrieve entry point address */
        vm_conf.text_address = eh->e_entry;

        break;
    }
    }

    free (image);

    return 0;
}

  Bit32u
image_load(char *path, Bit32u paddr)
{
  struct stat stat_buf;
  int fd, ret;
  unsigned long size;
  Bit32u page_size, paddr_orig;
#define BUFF_SIZE 4096
  Bit8u buffer[BUFF_SIZE];
 
  /* read in ROM BIOS image file */
  fd = open(path, O_RDONLY
#ifdef O_BINARY
                  | O_BINARY
#endif
           );
  if (fd < 0) {
    fprintf(stderr, "image_load: couldn't open image file '%s'.\n", path);
    return 0;
    }
  ret = fstat(fd, &stat_buf);
  if (ret) {
    fprintf(stderr, "image_load: couldn't stat image file '%s'.\n", path);
    return 0;
    }
 
  size = stat_buf.st_size;
  page_size = ((Bit32u)size + 0xfff) & ~0xfff;
 
  paddr_orig = paddr;
 
  while (size > 0) {
    ret = read(fd, buffer, (size>BUFF_SIZE)?BUFF_SIZE:size);
    if (ret <= 0) {
      fprintf(stderr, "image_load: read failed on image\n");
      return 0;
      }
    if (pluginWritePhyMem(paddr, (size>BUFF_SIZE)?BUFF_SIZE:size, buffer))
      {
      fprintf (stderr, "image_load: trying to load beyond available "
                       "VM memory.\n");
      return 0;
      }
    size -= ret;
    paddr += ret;
    }
  close(fd);
  fprintf(stderr, "image_load: '%s', size=%u read into memory at %08x\n",
          path, (unsigned) stat_buf.st_size, (unsigned) paddr_orig);
 
  return page_size;
/* +++ fix this wierdness */
}

/************************************************************************/
/* Setup initial guest context                                          */
/************************************************************************/

void
bin_context(unsigned hack_linux)
{
    struct multiboot_header *mbh = NULL;
    char text[1024];
    int i, mbi_offset = 0;
    guest_cpu_t guest_cpu;

    /* Is the binary multiboot compliant? */

    if (!pluginReadPhyMem(vm_conf.text_address, 1024, text))
    {
        for (i = 0; i < 1024-sizeof(struct multiboot_header); i++)
            if (*(Bit32u *)(text + i) == MULTIBOOT_MAGIC)
            {
	        fprintf(stderr, "bios: This is a multiboot compliant kernel.\n");
                mbh = (struct multiboot_header *)(text + i);
                break;
            }
    }

    if (mbh)
    {
	struct multiboot_info mbi;
	memset(&mbi, 0, sizeof(mbi));

        if (mbh->flags & MULTIBOOT_PAGE_ALIGN)
            fprintf(stderr,"Should align all boot modules to page boundaries.\n"); 

        mbi.flags |= MULTIBOOT_MEMORY;
        mbi.mem_lower = 1024;
        mbi.mem_upper = (vm_conf.max_memory - 1) * 1024;

        /* FIXME: find a nice, warm and dry place for multiboot info structure */
        mbi_offset = vm_conf.max_memory * 1024 * 1024 - sizeof(mbi);
        if (pluginWritePhyMem(mbi_offset, sizeof(mbi), (char *)&mbi))
            mbi_offset = 0;
    }

    
    /* Setup initial context */

    if (hack_linux) {
      void get_cpu_linux_values(guest_cpu_t *);
      get_cpu_linux_values(&guest_cpu);
      }
    else {
      void get_cpu_test_values(guest_cpu_t *);
      get_cpu_test_values(&guest_cpu);
      }

    guest_cpu.eax = mbi_offset? MULTIBOOT_VALID : 0;
    guest_cpu.ebx = mbi_offset;
    guest_cpu.ecx = vm_conf.text_address;
    guest_cpu.esp = vm_conf.stack_address;
    guest_cpu.eip = vm_conf.text_address;
    vm_set_cpu(&guest_cpu);
}


/************************************************************************/
/* Memory dump routine                                                  */
/************************************************************************/

void
bin_dump (char *dump_file_name)
{
    int   dumpno, counter, p, i, j;
    char  page[4096], outbuf[330];

    counter = 0;

    fprintf (stderr, "bios: dumping memory to %s\n", dump_file_name);

    dumpno = open (dump_file_name, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR);

    for (p = 0; p < vm_conf.max_memory * 256; p++)
    {
        if (pluginReadPhyMem(p*4096, 4096, page))
            break;

        for (i = 0; i < 4096; i++)
        {
            static char hex[] = "0123456789abcdef";

            if (!(i%32)) 
            {
                int addr = p*4096 + i;

                for (j = 20; j >= 0; j -= 4 )
                    outbuf[counter++] = hex[ (addr >> j) & 0x0f ];

                outbuf[counter++] = ':';
                outbuf[counter++] = ' ';
            }

            outbuf[counter++] = hex[ (page[i] >> 4) & 0x0f ];
            outbuf[counter++] = hex[  page[i]       & 0x0f ];
        
            if (!((i+1)%4)) 
                outbuf[counter++]=' ';
            if (!((i+1)%32)) 
                outbuf[counter++]='\n';
            if (!((i+1)%128))
            {   
                write(dumpno, &outbuf, 324);
                counter = 0;
            }
        }
    }

    close (dumpno);
}
