/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  emulation.h: features relating to x86 emulation
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#ifndef __EMULATE_H__
#define __EMULATE_H__

/*
 * Features exported by the instruction emulation to the entire monitor.
 */
#define EmulateStopReasonCycles          0x01
#define EmulateStopReasonDT              0x02
#define EmulateStopReasonExecuteNative   0x04
#define EmulateStopReasonModeChange      0x08
#define EmulateStopReasonInstructions    0x10
unsigned cpuEmulate(vm_t *vm, guest_context_t *context,
                    unsigned *instructions);

void   emulate_interrupt(vm_t *vm, unsigned vector);
void   emulate_exception(vm_t *vm, unsigned vector, Bit32u gerror);
void   emulate_async_checks(vm_t *vm);

extern Bit16u OpcodeInfo[];
 
#define G_GetCPL(vm)  vm->guest_cpu.cpl
#define IsNullSelector(selector) ( ((selector.raw) & 0xfffc) == 0 )

#define G_EIP(vm)  (vm->guest.addr.guest_context->eip)

#define G_GetIOPL(vm) (vm->guest_cpu.veflags.fields.iopl)
#define G_GetVIP(vm)  (vm->guest_cpu.veflags.fields.vip)
#define G_GetVIF(vm)  (vm->guest_cpu.veflags.fields.vif)
#define G_GetVM(vm)   (vm->guest_cpu.veflags.fields.vm)
#define G_GetIF(vm)   (vm->guest_cpu.veflags.fields.if_)


/* */
/* Features exported only to the x86 emulation code */
/* */

#ifdef EMULATION
void iPrefetch(vm_t *);

void LGDT_Ms(vm_t *);
void LIDT_Ms(vm_t *);
void CALL_Ed(vm_t *);
void MOV_SwEw(vm_t *);
void JMP_Ap(vm_t *);
void PUSH_Iv(vm_t *);
void PUSHF_Fv(vm_t *);
void POPF_Fv(vm_t *);
void CALL_Ad(vm_t *);
void CALL_Aw(vm_t *);
void MOV_GdEd(vm_t *);
void MOV_GwEw(vm_t *);
void MOV_GbEb(vm_t *);
void MOV_EwSw(vm_t *);
void MOVZX_GdEb(vm_t *);
void MOVZX_GwEb(vm_t *);
void MOV_EdGd(vm_t *);
void MOV_EwGw(vm_t *);
void LEA_GdM(vm_t *);
void LEA_GwM(vm_t *);
void LEAVE(vm_t *);
void RETnear32(vm_t *);
void RETnear16(vm_t *);
void AND_EdId(vm_t *);
void LTR_Ew(vm_t *);
void IRET32(vm_t *);
void IRET16(vm_t *);
void INT3(vm_t *);
void INT_Ib(vm_t *);
void MOV_EwIw(vm_t *);
void MOV_EdId(vm_t *);
void ROL_Ed(vm_t *);
void ROR_Ed(vm_t *);
void RCL_Ed(vm_t *);
void RCR_Ed(vm_t *);
void SHL_Ed(vm_t *);
void SHR_Ed(vm_t *);
void SAR_Ed(vm_t *);
void ROL_Ew(vm_t *);
void ROR_Ew(vm_t *);
void RCL_Ew(vm_t *);
void RCR_Ew(vm_t *);
void SHL_Ew(vm_t *);
void SHR_Ew(vm_t *);
void SAR_Ew(vm_t *);
void OUT_IbAL(vm_t *);
void OUT_IbeAX(vm_t *);
void IN_ALIb(vm_t *);
void IN_eAXIb(vm_t *);
void MOV_OdEAX(vm_t *);
void MOV_OwAX(vm_t *);
void MOV_EbGb(vm_t *);
void MOV_ObAL(vm_t *);
void MOV_CdRd(vm_t *);
void MOV_RdCd(vm_t *);
void INC_Ed(vm_t *);
void CMP_EdGd(vm_t *);
void CMP_EwGw(vm_t *);
void CPUID(vm_t *);
void LAR_GvEw(vm_t *);
void LSL_GvEw(vm_t *);
void CLTS(vm_t *);
void WBINVD(vm_t *);
void LxS_GvMp(vm_t *, unsigned sreg);
void LLDT_Ew(vm_t *);
void CLI_guest(vm_t *);
void STI_guest(vm_t *);
void PUSH_SREG(vm_t *, unsigned sreg);
void MOV_DdRd(vm_t *);
void JMP_Jd(vm_t *);
void JMP_Jw(vm_t *);
void JCC_Jd(vm_t *);
void JCC_Jw(vm_t *);
void POP_SREG(vm_t *, unsigned sreg);
void JMP_Ed(vm_t *);
void ADD_EbIb(vm_t *);
void OR_EbIb(vm_t *);
void ADC_EbIb(vm_t *);
void SBB_EbIb(vm_t *);
void AND_EbIb(vm_t *);
void SUB_EbIb(vm_t *);
void XOR_EbIb(vm_t *);
void CMP_EbIb(vm_t *);
void MOV_EAXOd(vm_t *);
void MOV_AXOw(vm_t *);
void PUSHAD32(vm_t *);
void PUSHAD16(vm_t *);
void MUL_EAXEd(vm_t *);
void IMUL_EAXEd(vm_t *);
void MOVSX_GdEb(vm_t *);
void MOVSX_GwEb(vm_t *);
void MOV_RdDd(vm_t *);
void CMP_EdId(vm_t *);
void HLT(vm_t *);
void XOR_EdGd(vm_t *);
void XOR_EwGw(vm_t *);
void CALL32_Ap(vm_t *);
void CALL16_Ap(vm_t *);
void JMP_Ed(vm_t *);
void JMP_Ew(vm_t *);
void LODSB_ALXb(vm_t *);
void TEST_EbIb(vm_t *);
void LOOP_Jb(vm_t *);
void INSW_YvDX(vm_t *);
void CALL16_Ep(vm_t *);
void CALL_Ew(vm_t *);
void RETfar32(vm_t *);
void RETfar16(vm_t *);
void JMP16_Ep(vm_t *);
void MOVSB_XbYb(vm_t *);
void INC_Eb(vm_t *);
void DEC_Eb(vm_t *);
void CMP_ALIb(vm_t *);
void STOSB_YbAL(vm_t *);
void STOSW_YveAX(vm_t *);
void INC_ERX(vm_t *);
void INC_RX(vm_t *);
void DEC_ERX(vm_t *);
void DEC_RX(vm_t *);
void AND_ALIb(vm_t *);
void OR_ALIb(vm_t *);
void AND_EAXId(vm_t *);
void AND_AXIw(vm_t *);
void ROL_Eb(vm_t *);
void ROR_Eb(vm_t *);
void RCL_Eb(vm_t *);
void RCR_Eb(vm_t *);
void SHL_Eb(vm_t *);
void SHR_Eb(vm_t *);
void SHL_Eb(vm_t *);
void SAR_Eb(vm_t *);
void TEST_EbIb(vm_t *);
void NOT_Eb(vm_t *);
void NEG_Eb(vm_t *);
void MUL_ALEb(vm_t *);
void IMUL_ALEb(vm_t *);
void DIV_ALEb(vm_t *);
void IDIV_ALEb(vm_t *);
void ADD_EbGb(vm_t *);
void DIV_EAXEd(vm_t *);
void IDIV_EAXEd(vm_t *);
void ADD_EdGd(vm_t *);
void ADD_EwGw(vm_t *);
void XOR_EbGb(vm_t *);
void CMP_EAXId(vm_t *);
void CMP_AXIw(vm_t *);
void SUB_GdEd(vm_t *);
void SUB_GwEw(vm_t *);
void POPAD32(vm_t *);
void POPAD16(vm_t *);
void MOV_EbIb(vm_t *);
void MOV_ALOb(vm_t *);
void OR_GbEb(vm_t *);
void XCHG_EbGb(vm_t *);
void SUB_GbEb(vm_t *);
void OR_GdEd(vm_t *);
void OR_GwEw(vm_t *);
void CMP_EbGb(vm_t *);
void LAHF(vm_t *);
void SAHF(vm_t *);
void OR_EbGb(vm_t *);
void ADD_ALIb(vm_t *);
void CWDE(vm_t *vm);
void CBW(vm_t *vm);
void ADD_GdEd(vm_t *vm);
void ADD_GwEw(vm_t *vm);
void JCXZ_Jb(vm_t *vm);

void ADD_EwIw(vm_t *vm);
void OR_EwIw(vm_t *vm);
void ADC_EwIw(vm_t *vm);
void SBB_EwIw(vm_t *vm);
void AND_EwIw(vm_t *vm);
void SUB_EwIw(vm_t *vm);
void XOR_EwIw(vm_t *vm);
void CMP_EwIw(vm_t *vm);

void TEST_EbGb(vm_t *vm);
void TEST_ALIb(vm_t *vm);
void CMP_GbEb(vm_t *vm);
void ADD_GbEb(vm_t *vm);
void XCHG_ERXEAX(vm_t *vm);
void XCHG_RXAX(vm_t *vm);
void LOOPNE_Jb(vm_t *vm);
void LOOPE_Jb(vm_t *vm);
void SUB_ALIb(vm_t *vm);
void TEST_EdGd(vm_t *vm);
void TEST_EwGw(vm_t *vm);
void PUSH_Ew(vm_t *vm);
void CMP_GdEd(vm_t *vm);
void CMP_GwEw(vm_t *vm);

void TEST_EwIw(vm_t *);
void NOT_Ew(vm_t *);
void NEG_Ew(vm_t *);
void MUL_AXEw(vm_t *);
void IMUL_AXEw(vm_t *);
void DIV_AXEw(vm_t *);
void IDIV_AXEw(vm_t *);

void ADD_EAXId(vm_t *);
void ADD_AXIw(vm_t *);
void SUB_EAXId(vm_t *);
void SUB_AXIw(vm_t *);
void XOR_GdEd(vm_t *);
void XOR_GwEw(vm_t *);
void MOVSW_XvYv(vm_t *);
void SUB_EbGb(vm_t *);
void OR_EdGd(vm_t *);
void OR_EwGw(vm_t *);
void CMPSB_XbYb(vm_t *);
void LODSW_eAXXv(vm_t *);
void ADC_GdEd(vm_t *);
void ADC_GwEw(vm_t *);
void SUB_EdGd(vm_t *);
void SUB_EwGw(vm_t *);
void CDQ(vm_t *);
void CWD(vm_t *);
void SBB_GdEd(vm_t *);
void SBB_GwEw(vm_t *);
void INC_Ew(vm_t *);
void DEC_Ew(vm_t *);
void OR_EAXId(vm_t *);
void OR_AXIw(vm_t *);
void RETfar32_Iw(vm_t *);
void RETfar16_Iw(vm_t *);
void SBB_EAXId(vm_t *);
void SBB_AXIw(vm_t *);
void ADC_EdGd(vm_t *);
void ADC_EwGw(vm_t *);
void RETnear32_Iw(vm_t *);
void RETnear16_Iw(vm_t *);
void AND_GdEd(vm_t *);
void AND_GwEw(vm_t *);
void SBB_EdGd(vm_t *);
void SBB_EwGw(vm_t *);
void SCASB_ALXb(vm_t *);
void SCASW_eAXXv(vm_t *);
void XCHG_EdGd(vm_t *);
void XCHG_EwGw(vm_t *);
void POP_Ed(vm_t *);
void POP_Ew(vm_t *);
void XOR_GbEb(vm_t *);
void TEST_EAXId(vm_t *);
void TEST_AXIw(vm_t *);
void IMUL_GdEd(vm_t *);
void IMUL_GwEw(vm_t *);
void AND_GbEb(vm_t *);
void ADC_EAXId(vm_t *);
void ADC_AXIw(vm_t *);
void XLAT(vm_t *);
void AAD(vm_t *);
void CMC(vm_t *);
void CMPSW_XvYv(vm_t *);
void ADD_EdId(vm_t *);
void OR_EdId(vm_t *);
void ADC_EdId(vm_t *);
void SBB_EdId(vm_t *);
void AND_EdId(vm_t *);
void SUB_EdId(vm_t *);
void XOR_EdId(vm_t *);
void CMP_EdId(vm_t *);
void XOR_EAXId(vm_t *);
void XOR_AXIw(vm_t *);
void XOR_ALIb(vm_t *);
void AND_EdGd(vm_t *);
void AND_EwGw(vm_t *);
void AND_EbGb(vm_t *);
void DEC_Ed(vm_t *);
void SETCC_Eb(vm_t *);
void NOT_Ed(vm_t *);
void ADC_EbGb(vm_t *);
void ADC_GbEb(vm_t *);
void ADC_ALIb(vm_t *);
void NEG_Ed(vm_t *);
void TEST_EdId(vm_t *);
void PUSH_Ed(vm_t *);
void AAA(vm_t *);
void AAM(vm_t *);
void AAS(vm_t *);
void DAA(vm_t *);
void DAS(vm_t *);
void SBB_EbGb(vm_t *);
void SBB_GbEb(vm_t *);
void SBB_ALIb(vm_t *);
void SALC(vm_t *);
void XADD_EwGw(vm_t *);
void XADD_EdGd(vm_t *);
void XADD_EbGb(vm_t *);
void MOVSX_GdEw(vm_t *);
void MOVSX_GwEw(vm_t *);
void MOVZX_GdEw(vm_t *);
void MOVZX_GwEw(vm_t *);
void SHLD_EdGd(vm_t *);
void SHLD_EwGw(vm_t *);
void SHRD_EdGd(vm_t *);
void SHRD_EwGw(vm_t *);
void CMPXCHG_EbGb(vm_t *);
void CMPXCHG_EdGd(vm_t *);
void CMPXCHG_EwGw(vm_t *);
void CMOV_GdEd(vm_t *);
void CMOV_GwEw(vm_t *);
void IMUL_GdEdId(vm_t *);
void IMUL_GwEwIw(vm_t *);

void BSF_GvEv(vm_t *);
void BSR_GvEv(vm_t *);
void BT_EvGv(vm_t *);
void BTS_EvGv(vm_t *);
void BTR_EvGv(vm_t *);
void BTC_EvGv(vm_t *);
void BT_EvIb(vm_t *);
void BTS_EvIb(vm_t *);
void BTC_EvIb(vm_t *);
void BTR_EvIb(vm_t *);
void BSWAP_ERX(vm_t *);
void OUTSW_DXXv(vm_t *);
void OUTSB_DXXb(vm_t *);
void SGDT_Ms(vm_t *);
void SIDT_Ms(vm_t *);
void SMSW_Ew(vm_t *);
void LMSW_Ew(vm_t *);
void INVLPG(vm_t *);
void FWAIT(vm_t *);
void CALL32_Ep(vm_t *);
void JMP32_Ep(vm_t *);
void IN_ALDX(vm_t *);
void IN_eAXDX(vm_t *);
void SLDT_Ew(vm_t *);
void STR_Ew(vm_t *);
void ENTER_IwIb(vm_t *);
void ARPL_EwGw(vm_t *);
void VERR_Ew(vm_t *);
void VERW_Ew(vm_t *);
void OUT_IbeAX(vm_t *);
void RDTSC(vm_t *);
void INSB_YbDX(vm_t *);

void ESC0(vm_t *);
void ESC1(vm_t *);
void ESC2(vm_t *);
void ESC3(vm_t *);
void ESC4(vm_t *);
void ESC5(vm_t *);
void ESC6(vm_t *);
void ESC7(vm_t *);


#define read_virtual_dword(vm, seg, off, ptr) \
  read_virtual(vm, seg, off, ptr, 4)
#define read_virtual_word(vm, seg, off, ptr) \
  read_virtual(vm, seg, off, ptr, 2)
#define read_virtual_byte(vm, seg, off, ptr) \
  read_virtual(vm, seg, off, ptr, 1)
 
#define write_virtual_dword(vm, seg, off, ptr) \
  write_virtual(vm, seg, off, ptr, 4)
#define write_virtual_word(vm, seg, off, ptr) \
  write_virtual(vm, seg, off, ptr, 2)
#define write_virtual_byte(vm, seg, off, ptr) \
  write_virtual(vm, seg, off, ptr, 1)
 
#define read_RMW_virtual_dword(vm, seg, off, ptr) \
  read_RMW_virtual(vm, seg, off, ptr, 4)
#define read_RMW_virtual_word(vm, seg, off, ptr) \
  read_RMW_virtual(vm, seg, off, ptr, 2)
#define read_RMW_virtual_byte(vm, seg, off, ptr) \
  read_RMW_virtual(vm, seg, off, ptr, 1)
 
#define write_RMW_virtual_dword(vm, ptr) \
  write_RMW_virtual(vm, ptr, 4)
#define write_RMW_virtual_word(vm, ptr) \
  write_RMW_virtual(vm, ptr, 2)
#define write_RMW_virtual_byte(vm, ptr) \
  write_RMW_virtual(vm, ptr, 1)
 
void read_virtual(vm_t *vm, unsigned seg, Bit32u off, void *ptr,
                  unsigned size);
void write_virtual(vm_t *vm, unsigned seg, Bit32u off, void *ptr,
                   unsigned size);
void read_RMW_virtual(vm_t *vm, unsigned seg, Bit32u off, void *ptr,
                      unsigned size);
void write_RMW_virtual(vm_t *vm, void *ptr, unsigned size);

void IOPermCheck(vm_t *vm, Bit32u port, unsigned len);
void push32(vm_t *vm, Bit32u val32);
void push16(vm_t *vm, Bit16u val16);
unsigned can_push(vm_t *vm, descriptor_cache_t *ss_cache, Bit32u esp,
                  unsigned bytes);
unsigned can_pop(vm_t *vm, unsigned bytes);
void access_linear(vm_t *vm, Bit32u laddr, unsigned length, unsigned pl,
                   unsigned rw, void *data);
void read_physical(vm_t *vm, Bit32u paddr, unsigned length, void *data);
void write_physical(vm_t *vm, Bit32u paddr, unsigned length, void *data);
void load_seg_reg(vm_t *vm, unsigned sreg, Bit16u selector);
void task_switch(vm_t *, selector_t, descriptor_cache_t *, unsigned source);
void descriptor2cache(vm_t *vm, descriptor_cache_t *cache);
void jump_protected(vm_t *vm, Bit16u cs_raw, Bit32u disp32);
void call_protected(vm_t *vm, Bit16u cs_raw, Bit32u disp32);
void return_protected(vm_t *vm, Bit16u pop_bytes);

void fetch_raw_descriptor(vm_t *vm, selector_t selector, descriptor_t *desc,
                     unsigned exception_no);
unsigned fetch_descriptor2(vm_t *vm, selector_t selector,
                           descriptor_t *desc);
void exception(vm_t *vm, unsigned vector,
               Bit32u error_code) __attribute__ ((noreturn));
void load_cs_pro(vm_t *, selector_t selector, descriptor_cache_t *cache,
             unsigned cpl);
void load_ss_pro(vm_t *, selector_t selector, descriptor_cache_t *cache,
             unsigned cpl);
void load_sreg_pro(vm_t *, unsigned sreg, selector_t selector,
                   descriptor_cache_t *cache);
void invalidate_sreg_pro(vm_t *vm, unsigned sreg, selector_t selector);

void validate_page_timestamp(vm_t *vm, Bit32u paddr);
void   pop32(vm_t *, Bit32u *val32);
void   pop16(vm_t *, Bit16u *val16);
void   WriteReg32(vm_t *vm, unsigned reg, Bit32u val);
void   WriteReg16(vm_t *vm, unsigned reg, Bit16u val);
void   WriteReg8(vm_t *vm, unsigned reg, Bit8u val);
Bit32u ReadReg32(vm_t *vm, unsigned reg);
Bit16u ReadReg16(vm_t *vm, unsigned reg);
Bit8u  ReadReg8(vm_t *vm, unsigned reg);
void   UndefinedOpcode(vm_t *vm) __attribute__ ((noreturn));
void   iret_protected(vm_t *);
void   stack_return_to_v86(vm_t *vm, Bit32u new_eip, selector_t cs,
                           Bit32u flags32);
void   stack_return_from_v86(vm_t *vm);
void   validate_seg_regs(vm_t *);
void   interrupt(vm_t *vm, Bit8u vector, unsigned is_INT,
                 unsigned is_error_code, Bit16u error_code);
void   async_checks(vm_t *vm);

#define OSZAPCMask 0x000008d5
  static inline void
SetOSZAPC(vm_t *vm, Bit32u eflags)
{
  vm->guest.addr.guest_context->eflags.raw =
    (vm->guest.addr.guest_context->eflags.raw & ~OSZAPCMask) |
    (eflags & OSZAPCMask);
}

#define OSZAPMask 0x000008d4
  static inline void
SetOSZAP(vm_t *vm, Bit32u eflags)
{
  vm->guest.addr.guest_context->eflags.raw =
    (vm->guest.addr.guest_context->eflags.raw & ~OSZAPMask) |
    (eflags & OSZAPMask);
}

#define SZAPCMask 0x000000d5
  static inline void
SetSZAPC(vm_t *vm, Bit32u eflags)
{
  vm->guest.addr.guest_context->eflags.raw =
    (vm->guest.addr.guest_context->eflags.raw & ~SZAPCMask) |
    (eflags & SZAPCMask);
}

  static inline void
SetFlagsFromImageZ(vm_t *vm, Bit32u eflags)
{
  vm->guest.addr.guest_context->eflags.raw =
    (vm->guest.addr.guest_context->eflags.raw & ~FLG_ZF) |
    (eflags & FLG_ZF);
}
  
 
void get_SS_ESP_from_TSS(vm_t *, unsigned pl, selector_t *ss, Bit32u *esp);

#define InhibitInterrupts 0x01
#define InhibitDebug      0x02

#define invalidate_prefetch_q()
#define revalidate_prefetch_q()

#define G_CPL(vm)  vm->guest_cpu.cpl
#define G_ESP(vm)  (vm->guest.addr.guest_context->esp)

#define G_EAX(vm)  (vm->guest.addr.guest_context->eax)
#define G_EBX(vm)  (vm->guest.addr.guest_context->ebx)
#define G_ECX(vm)  (vm->guest.addr.guest_context->ecx)
#define G_EDX(vm)  (vm->guest.addr.guest_context->edx)
#define G_EDI(vm)  (vm->guest.addr.guest_context->edi)
#define G_ESI(vm)  (vm->guest.addr.guest_context->esi)
#define G_EBP(vm)  (vm->guest.addr.guest_context->ebp)

#define G_IP(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->eip))
#define G_SP(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->esp))

#define G_AX(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->eax))
#define G_BX(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->ebx))
#define G_CX(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->ecx))
#define G_DX(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->edx))
#define G_DI(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->edi))
#define G_SI(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->esi))
#define G_BP(vm)   (* ((Bit16u*) &vm->guest.addr.guest_context->ebp))

#define G_AL(vm)   (* ((Bit8u*) &vm->guest.addr.guest_context->eax))
#define G_BL(vm)   (* ((Bit8u*) &vm->guest.addr.guest_context->ebx))
#define G_CL(vm)   (* ((Bit8u*) &vm->guest.addr.guest_context->ecx))
#define G_DL(vm)   (* ((Bit8u*) &vm->guest.addr.guest_context->edx))

#define G_AH(vm)   (* (((Bit8u*) &vm->guest.addr.guest_context->eax)+1))
#define G_BH(vm)   (* (((Bit8u*) &vm->guest.addr.guest_context->ebx)+1))
#define G_CH(vm)   (* (((Bit8u*) &vm->guest.addr.guest_context->ecx)+1))
#define G_DH(vm)   (* (((Bit8u*) &vm->guest.addr.guest_context->edx)+1))

#define G_FLG_NT(vm) (vm->guest.addr.guest_context->eflags.fields.nt)

#define G_SetIOPL(vm, val) vm->guest_cpu.veflags.fields.iopl = (val)
#define G_SetVIP(vm, val)  vm->guest_cpu.veflags.fields.vip = (val)
#define G_SetVIF(vm, val)  vm->guest_cpu.veflags.fields.vif = (val)


#define G_SetIF(vm, val) vm->guest_cpu.veflags.fields.if_ = (val)

#if 0
#define G_GetTF(vm)   (vm->guest.addr.guest_context->eflags.fields.tf)
#define G_SetTF(vm, val) \
  vm->guest.addr.guest_context->eflags.fields.tf = (val)
#endif
 
#define G_GetTF(vm)      (vm->guest_cpu.veflags.fields.tf)
#define G_SetTF(vm, val)  vm->guest_cpu.veflags.fields.tf = (val)

 
/* Arithmetic flags; return 0 or 1 */
#define G_GetOF(vm) (vm->guest.addr.guest_context->eflags.fields.of)
#define G_SetOF(vm, val) \
  vm->guest.addr.guest_context->eflags.fields.of = (val)

#define G_GetSF(vm) (vm->guest.addr.guest_context->eflags.fields.sf)
#define G_GetPF(vm) (vm->guest.addr.guest_context->eflags.fields.pf)

#define G_GetZF(vm) (vm->guest.addr.guest_context->eflags.fields.zf)
#define G_SetZF(vm, val) \
  vm->guest.addr.guest_context->eflags.fields.zf = (val)

#define G_GetCF(vm) (vm->guest.addr.guest_context->eflags.fields.cf)
#define G_SetCF(vm, val) \
  vm->guest.addr.guest_context->eflags.fields.cf = (val)
 
#define G_GetRF(vm) (vm->guest.addr.guest_context->eflags.fields.rf)
#define G_SetRF(vm, val) \
  vm->guest.addr.guest_context->eflags.fields.rf = (val)

#define G_GetDF(vm) (vm->guest.addr.guest_context->eflags.fields.df)
#define G_SetDF(vm, val) \
  vm->guest.addr.guest_context->eflags.fields.df = (val)

#define TASK_FROM_JUMP         10
#define TASK_FROM_CALL_OR_INT  11
#define TASK_FROM_IRET         12

Bit32u dtranslate_linear(vm_t *, Bit32u laddr, unsigned pl, unsigned rw);
void   CR3_change(vm_t *, Bit32u val);

#endif  /* EMULATION */

#endif  /* __EMULATE_H__ */
