/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: presentation.cxx,v $
 *
 *  $Revision: 1.12 $
 *
 *  last change: $Author: rt $ $Date: 2006/02/09 14:48:38 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "canvas/debug.hxx"
#include "unoviewcontainer.hxx"
#include "transitionfactory.hxx"
#include "eventmultiplexer.hxx"
#include "usereventqueue.hxx"
#include "eventqueue.hxx"
#include "activitiesqueue.hxx"
#include "activitiesfactory.hxx"
#include "interruptabledelayevent.hxx"
#include "slide.hxx"
#include "tools.hxx"
#include "unoview.hxx"
#include "slidebitmap.hxx"
#include "rehearsetimingsactivity.hxx"
#include "waitsymbol.hxx"

#include "cppuhelper/compbase3.hxx"
#include "cppuhelper/factory.hxx"
#include "cppuhelper/implementationentry.hxx"
#include "cppuhelper/compbase2.hxx"
#include "cppuhelper/interfacecontainer.h"
#include "comphelper/make_shared_from_uno.hxx"
#include "comphelper/scopeguard.hxx"
#include "comphelper/optional.hxx"
#include "cppcanvas/spritecanvas.hxx"
#include "cppcanvas/vclfactory.hxx"
#include "cppcanvas/basegfxfactory.hxx"
#include "tools/debug.hxx"
#include "basegfx/point/b2dpoint.hxx"
#include "basegfx/polygon/b2dpolygon.hxx"
#include "basegfx/matrix/b2dhommatrix.hxx"
#include "basegfx/polygon/b2dpolygontools.hxx"
#include "basegfx/polygon/b2dpolypolygontools.hxx"
#include "basegfx/tools/canvastools.hxx"
#include "com/sun/star/beans/XPropertySet.hpp"
#include "com/sun/star/util/XModifyListener.hpp"
#include "com/sun/star/util/XUpdatable.hpp"
#include "com/sun/star/awt/XPaintListener.hpp"
#include "com/sun/star/awt/SystemPointer.hpp"
#include "com/sun/star/animations/TransitionType.hpp"
#include "com/sun/star/animations/TransitionSubType.hpp"
#include "com/sun/star/presentation/XSlideShow.hpp"
#include "com/sun/star/lang/XServiceInfo.hpp"
#include "com/sun/star/lang/XServiceName.hpp"

#include "boost/utility.hpp"
#include "boost/bind.hpp"
#include <map>
#include <vector>
#include <iterator>
#include <algorithm>

namespace css = com::sun::star;
using namespace css;
using namespace ::presentation::internal;

namespace {

/******************************************************************************
   
   SlideShowImpl

   This class encapsulates the slideshow presentation viewer.
    
   With an instance of this class, it is possible to statically
   and dynamically show a presentation, as defined by the
   constructor-provided draw model (represented by a sequence
   of ::com::sun::star::drawing::XDrawPage objects).
    
   It is possible to show the presentation on multiple views
   simultaneously (e.g. for a multi-monitor setup). Since this
   class also relies on user interaction, the corresponding
   XSlideShowView interface provides means to register some UI
   event listeners (mostly borrowed from awt::XWindow interface).
   
   Since currently (mid 2004), OOo isn't very well suited to
   multi-threaded rendering, this class relies on <em>very
   frequent</em> external update() calls, which will render the
   next frame of animations. This works as follows: after the
   displaySlide() has been successfully called (which setup and
   starts an actual slide show), the update() method must be
   called until it returns false.
   Effectively, this puts the burden of providing
   concurrency to the clients of this class, which, as noted
   above, is currently unavoidable with the current state of
   affairs (I've actually tried threading here, but failed
   miserably when using the VCL canvas as the render backend -
   deadlocked).
   
 ******************************************************************************/

typedef cppu::WeakComponentImplHelper3<css::presentation::XSlideShow,
                                       lang::XServiceInfo,
                                       lang::XServiceName> SlideShowImplBase;

class SlideShowImpl : private comphelper::OBaseMutex,
                      public SlideShowImplBase,
                      private boost::noncopyable
{
public:
    /// UNO stuff
    static rtl::OUString SAL_CALL getImplementationName_()
    {
        return OUSTR("Slideshow::SlideShow");
    }
    static uno::Sequence<rtl::OUString> SAL_CALL getSupportedServiceNames_()
    {
        rtl::OUString const name(
            RTL_CONSTASCII_USTRINGPARAM("com.sun.star.presentation.SlideShow"));
        return uno::Sequence<rtl::OUString>( &name, 1 );
    }
    static uno::Reference<uno::XInterface> SAL_CALL create(
        uno::Reference<uno::XComponentContext> const& xContext )
        throw (uno::Exception);
    
    /** Notify that the transition phase of the current slide
        has ended.

        The life of a slide has three phases: the transition
        phase, when the previous slide vanishes, and the
        current slide becomes visible, the shape animation
        phase, when shape effects are running, and the phase
        after the last shape animation has ended, but before
        the next slide transition starts.

        This method notifies the end of the first phase.
    */
    void notifySlideTransitionEnded();

    /** Notify that the shape animation phase of the current slide
        has ended.

        The life of a slide has three phases: the transition
        phase, when the previous slide vanishes, and the
        current slide becomes visible, the shape animation
        phase, when shape effects are running, and the phase
        after the last shape animation has ended, but before
        the next slide transition starts.

        This method notifies the end of the second phase.
    */
    void notifySlideAnimationsEnded();

    /** Notify that the slide has ended.

        The life of a slide has three phases: the transition
        phase, when the previous slide vanishes, and the
        current slide becomes visible, the shape animation
        phase, when shape effects are running, and the phase
        after the last shape animation has ended, but before
        the next slide transition starts.
        
        This method notifies the end of the third phase.
    */
    void notifySlideEnded();

    /** Notify that the view has changed.

        Currently, this method call denotes a change in view
        size and/or scaling.
    */
    void notifyViewChange();

    /** Notification from eventmultiplexer that a hyperlink
        has been clicked.
    */
    bool notifyHyperLinkClicked( rtl::OUString const& hyperLink );
    
private:
    // XSlideShow:
    virtual sal_Bool SAL_CALL nextEffect() throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL startShapeActivity(
        uno::Reference<drawing::XShape> const& xShape )
        throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL stopShapeActivity(
        uno::Reference<drawing::XShape> const& xShape )
        throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL pause( sal_Bool bPauseShow )
        throw (uno::RuntimeException);
    virtual uno::Reference<drawing::XDrawPage> SAL_CALL getCurrentSlide()
        throw (uno::RuntimeException);
    virtual void SAL_CALL displaySlide(
        uno::Reference<drawing::XDrawPage> const& xSlide,
        uno::Reference<animations::XAnimationNode> const& xRootNode,
        uno::Sequence<beans::PropertyValue> const& rProperties )
        throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL setProperty(
        beans::PropertyValue const& rProperty ) throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL addView(
        uno::Reference<css::presentation::XSlideShowView> const& xView )
        throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL removeView(
        uno::Reference<css::presentation::XSlideShowView> const& xView )
        throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL update( double & nNextTimeout )
        throw (uno::RuntimeException);
    virtual void SAL_CALL addSlideShowListener(
        uno::Reference<css::presentation::XSlideShowListener> const& xListener )
        throw (uno::RuntimeException);
    virtual void SAL_CALL removeSlideShowListener(
        uno::Reference<css::presentation::XSlideShowListener> const& xListener )
        throw (uno::RuntimeException);
    virtual void SAL_CALL addShapeEventListener(
        uno::Reference<css::presentation::XShapeEventListener> const& xListener,
        uno::Reference<drawing::XShape> const& xShape )
        throw (uno::RuntimeException);
    virtual void SAL_CALL removeShapeEventListener(
        uno::Reference<css::presentation::XShapeEventListener> const& xListener,
        uno::Reference<drawing::XShape> const& xShape )
        throw (uno::RuntimeException);
    virtual void SAL_CALL setShapeCursor(
        uno::Reference<drawing::XShape> const& xShape, sal_Int16 nPointerShape )
        throw (uno::RuntimeException);
    
    // XServiceInfo
    virtual rtl::OUString SAL_CALL getImplementationName()
        throw (uno::RuntimeException);
    virtual sal_Bool SAL_CALL supportsService(
        rtl::OUString const& rServiceName ) throw (uno::RuntimeException);
    virtual uno::Sequence<rtl::OUString> SAL_CALL getSupportedServiceNames()
        throw (uno::RuntimeException);
    // XServiceName
    virtual rtl::OUString SAL_CALL getServiceName()
        throw (uno::RuntimeException);
    
private:
    // WeakComponentImplHelperBase
    virtual void SAL_CALL disposing();    
    virtual ~SlideShowImpl();
    explicit SlideShowImpl(
        uno::Reference<uno::XComponentContext> const& xContext );
    
    bool isDisposed( bool bCheckInDisposeToo = true ) const {
        return (rBHelper.bDisposed ||
                (bCheckInDisposeToo && rBHelper.bInDispose));
    }
    
private:
    struct SlideAnimationsEndHandler; friend struct SlideAnimationsEndHandler;
    struct ShapeListenerRegisterer; friend struct ShapeListenerRegisterer;
    struct PrefetchPropertiesFunc; friend struct PrefetchPropertiesFunc;
    
    /// Stop currently running show.
    void stopShow();
    
    /// Creates a new slide.
    SlideSharedPtr createSlide(
        uno::Reference<drawing::XDrawPage> const& xDrawPage,
        uno::Reference<animations::XAnimationNode> const& xRootNode );
    
    /// the previously running slide
    SlideSharedPtr mpPreviousSlide;
    /// the currently running slide
    SlideSharedPtr mpCurrentSlide;
    /// the already prefetched slide: best candidate for upcoming slide
    SlideSharedPtr mpPrefetchSlide;
    /// slide to be prefetched: best candidate for upcoming slide
    uno::Reference<drawing::XDrawPage> mxPrefetchSlide;
    /// slide animation to be prefetched:
    uno::Reference<animations::XAnimationNode> mxPrefetchAnimationNode;

    /// Checks whether the given slide/animation node matches mpPrefetchSlide
    static bool matches(
        SlideSharedPtr const& pSlide,
        uno::Reference<drawing::XDrawPage> const& xSlide,
        uno::Reference<animations::XAnimationNode> const& xNode )
    {
        if (pSlide)
            return (pSlide->getXDrawPage() == xSlide &&
                    pSlide->getXAnimationNode() == xNode);
        else
            return (!xSlide.is() && !xNode.is());
    }
    
    /// iterates over all buffered slides:
    template <typename FuncT>
    void forEachSlide( FuncT const& func )
    {
        if (mpPreviousSlide)
            func( mpPreviousSlide );
        if (mpCurrentSlide)
            func( mpCurrentSlide );
        if (mpPrefetchSlide)
            func( mpPrefetchSlide );
    }
    
    /// Resets the current slide transition sound object with a new one:
    SoundPlayerSharedPtr const& resetSlideTransitionSound(
        rtl::OUString const& url = rtl::OUString() );
    
    /** Prepare a slide transition
                
        This method registers all necessary events and
        activities for a slide transition.
        
        @return the slide change activity, or NULL for no transition effect
    */
    ActivitySharedPtr createSlideTransition(
        const uno::Reference< drawing::XDrawPage >&    xDrawPage,
        const SlideSharedPtr&                          rLeavingSlide,
        const SlideSharedPtr&                          rEnteringSlide,
        const EventSharedPtr&                          rTransitionEndEvent );
    
    /// Display/hide wait symbol on all views
    void setWaitState( bool bOn );
    /// sets mouse cursor related to flags.
    void updateMouseCursor();

    /// all registered views
    UnoViewContainer                        maViewContainer;
    
    /// all registered slide show listeners
    cppu::OInterfaceContainerHelper         maListenerContainer;
    
    typedef std::map<uno::Reference<drawing::XShape>,
                     std::vector<uno::Reference<
                                 css::presentation::XShapeEventListener> >
                     > ShapeEventListenerMap;
    typedef ::std::map<uno::Reference<drawing::XShape>,
                       sal_Int16> ShapeCursorMap;
    
    /// map of vectors, containing all registered listeners for a shape
    ShapeEventListenerMap                   maShapeEventListeners;
    /// map of sal_Int16 values, specifying the mouse cursor for every shape
    ShapeCursorMap                          maShapeCursors;
    
    boost::optional<RGBColor>               maUserPaintColor;

    boost::shared_ptr<canvas::tools::ElapsedTime> mpPresTimer;
    EventQueue                              maEventQueue;
    rtl::Reference<EventMultiplexer> const  mpEventMultiplexer;
    ActivitiesQueue                         maActivitiesQueue;
    UserEventQueue                          maUserEventQueue;

    EventHandlerSharedPtr                   mpSlideAnimationsEndHandler;

    boost::shared_ptr<RehearseTimingsActivity> mpRehearseTimingsActivity;
    boost::shared_ptr<WaitSymbol>           mpWaitSymbol;

    /// the current slide transition sound object:
    SoundPlayerSharedPtr                    mpCurrentSlideTransitionSound;
    
    uno::Reference<uno::XComponentContext>  mxComponentContext;
    
    bool                                    mbWaitState;
    bool                                    mbAutomaticAdvancementMode;
    bool                                    mbImageAnimationsAllowed;
    bool                                    mbNoSlideTransitions;
    bool                                    mbMouseVisible;
    bool                                    mbForceManualAdvance;
    bool                                    mbShowPaused;
    bool                                    mbSlideShowIdle;
};


/******************************************************************************
    
    SlideView
    
 ******************************************************************************/
typedef cppu::WeakComponentImplHelper2<
    util::XModifyListener, awt::XPaintListener> SlideViewBase;

class SlideView : private comphelper::OBaseMutex,
                  public SlideViewBase,
                  public UnoView,
                  private boost::noncopyable
{
public:
    static UnoViewSharedPtr create(
        uno::Reference<css::presentation::XSlideShowView> const& xView,
        EventQueue & rEventQueue, SlideShowImpl & rShow );
    
private:
    // View:
    virtual ViewLayerSharedPtr createViewLayer() const;
    virtual bool updateScreen() const;
    virtual bool isContentDestroyed() const;
    virtual void clear() const;
    virtual cppcanvas::CanvasSharedPtr getCanvas() const;
    virtual basegfx::B2DHomMatrix getTransformation() const;
    virtual void setViewSize( const basegfx::B2DSize& rSize );
    virtual void setClip( const basegfx::B2DPolyPolygon& rClip );
    virtual void setMouseCursor( sal_Int16 nPointerShape );
    virtual cppcanvas::CustomSpriteSharedPtr createSprite(
        const basegfx::B2DSize& rSpriteSizePixel ) const;
    virtual double getSpritePriority( double nSpritePrio ) const;
    virtual void setPriority( double nPrio );
    // UnoView:
    virtual uno::Reference<css::presentation::XSlideShowView> getUnoView()const;
    
    // XEventListener:
    virtual void SAL_CALL disposing( lang::EventObject const& evt )
        throw (uno::RuntimeException);
    // XModifyListener:
    virtual void SAL_CALL modified( const lang::EventObject& aEvent )
        throw (uno::RuntimeException);
    // XPaintListener:
    virtual void SAL_CALL windowPaint( const awt::PaintEvent& e )
        throw (uno::RuntimeException);
    
private:
    // WeakComponentImplHelperBase:
    virtual void SAL_CALL disposing();
    virtual ~SlideView();    
    SlideView( uno::Reference<css::presentation::XSlideShowView> const& xView,
               EventQueue & rEventQueue, SlideShowImpl & rShow );
    
private:
    class SlideViewLayer;
    
    void updateCanvas();
    
    uno::Reference<css::presentation::XSlideShowView>    mxView;
    cppcanvas::SpriteCanvasSharedPtr                     mpCanvas;
    
    EventQueue &                                         mrEventQueue;
    SlideShowImpl &                                      mrShow;
    
    const basegfx::B2DPolygon                            maUnitRectPoly;
    basegfx::B2DPolyPolygon                              maClip;
    
    basegfx::B2DHomMatrix                                maViewTransform;
    basegfx::B2DSize                                     maUserSize;
    
    mutable bool                                         mbContentValid;
};


/** Event handler for slide end events.
    This handler is registered for slide animation end
    events at the global EventMultiplexer, and internally
    queues an event that in turn calls
    SlideShowImpl::notifySlideAnimationsEnded()
*/
struct SlideShowImpl::SlideAnimationsEndHandler : public EventHandler,
                                                  private boost::noncopyable
{
    SlideShowImpl & mrShow;
    EventQueue & mrEventQueue;
    SlideAnimationsEndHandler( SlideShowImpl & rShow, EventQueue & rEventQueue )
        : mrShow( rShow ), mrEventQueue( rEventQueue ) {}
    virtual void dispose() {}
    virtual bool handleEvent()
    {
        // DON't call notifySlideAnimationsEnded()
        // directly, but queue an event. handleEvent()
        // might be called from e.g.
        // showNext(), and notifySlideAnimationsEnded() must not be called
        // in recursion.
        mrEventQueue.addEvent( 
            makeEvent( boost::bind( &SlideShowImpl::notifySlideAnimationsEnded,
                                    boost::ref(mrShow) ) ) );
        return true;
    }
};

SlideShowImpl::SlideShowImpl(
    uno::Reference<uno::XComponentContext> const& xContext )
    : SlideShowImplBase(m_aMutex),
      maViewContainer(),
      maListenerContainer( m_aMutex ),
      maShapeEventListeners(),
      maShapeCursors(),
      maUserPaintColor(),
      mpPresTimer( new canvas::tools::ElapsedTime ),
      maEventQueue( mpPresTimer ),
      mpEventMultiplexer( EventMultiplexer::create( maEventQueue ) ),
      maActivitiesQueue( mpPresTimer, *mpEventMultiplexer ),
      maUserEventQueue( *mpEventMultiplexer, maEventQueue ),
      mbAutomaticAdvancementMode(false),
      mxComponentContext( xContext ),
      mbWaitState(false),
      mbImageAnimationsAllowed( true ),
      mbNoSlideTransitions( false ),
      mbMouseVisible( true ),
      mbForceManualAdvance( false ),
      mpRehearseTimingsActivity(),
      mpWaitSymbol(),
      mpCurrentSlideTransitionSound(),
      mbShowPaused( false ),
      mbSlideShowIdle( true )
{
    // keep care not constructing any UNO references to this inside ctor,
    // shift that code to create()!
    
    mpSlideAnimationsEndHandler.reset( new SlideAnimationsEndHandler(
                                           *this, maEventQueue ) );
    mpEventMultiplexer->addSlideAnimationsEndHandler(
        mpSlideAnimationsEndHandler );
    mpEventMultiplexer->addHyperlinkHandler(
        EventMultiplexer::HyperLinkHandlerFunc(
            boost::bind( &SlideShowImpl::notifyHyperLinkClicked,
                         // this valid at least as long as multiplexer
                         this, _1 ) ),
        this /* handler id */ );
}

uno::Reference<uno::XInterface> SlideShowImpl::create(
    uno::Reference<uno::XComponentContext> const& xContext )
    throw (uno::Exception)
{
    return static_cast<cppu::OWeakObject *>( new SlideShowImpl(xContext) );
}

SlideShowImpl::~SlideShowImpl()
{
    // WeakComponentImplHelperBase self disposes if undisposed
}

// we are about to be disposed (someone call dispose() on us)
void SlideShowImpl::disposing()
{
    // stop show, if needed:
    stopShow();
    
    mxComponentContext.clear();
    mpCurrentSlideTransitionSound.reset();
    if (mpWaitSymbol) {
        mpWaitSymbol->dispose();
        mpWaitSymbol.reset();
    }
    if (mpRehearseTimingsActivity) {
        mpRehearseTimingsActivity->dispose();
        mpRehearseTimingsActivity.reset();
    }
    if (mpSlideAnimationsEndHandler) {
        mpEventMultiplexer->removeSlideAnimationsEndHandler(
            mpSlideAnimationsEndHandler );
        mpSlideAnimationsEndHandler.reset();
    }    
    maUserEventQueue.clear();
    maActivitiesQueue.clear();
    mpEventMultiplexer->removeHyperlinkHandler(this);
    mpEventMultiplexer->clear();
    maEventQueue.clear();
    mpPresTimer.reset();
    maShapeCursors.clear();
    // TODO(Q3): need to dispose the listeners. Maybe also use
    // OListenerContainerBase here.
    maShapeEventListeners.clear();
    // send all listeners a disposing() that we are going down:
    maListenerContainer.disposeAndClear(
        lang::EventObject( static_cast<cppu::OWeakObject *>(this) ) );
    maViewContainer.clear();
    // release slides:
    mxPrefetchAnimationNode.clear();
    mxPrefetchSlide.clear();
    mpPrefetchSlide.reset();
    mpCurrentSlide.reset();
    mpPreviousSlide.reset();
}

SoundPlayerSharedPtr const& SlideShowImpl::resetSlideTransitionSound(
    rtl::OUString const& url )
{
    if (mpCurrentSlideTransitionSound) {
        mpCurrentSlideTransitionSound->stopPlayback();
        mpCurrentSlideTransitionSound->dispose();
        mpCurrentSlideTransitionSound.reset();
    }
    
    if (url.getLength() > 0) {
        try {
            mpCurrentSlideTransitionSound = SoundPlayer::create(
                *mpEventMultiplexer, url, mxComponentContext );
        }
        catch (lang::NoSupportException const&) {
            // catch possible exceptions from SoundPlayer, since
            // being not able to playback the sound is not a hard
            // error here (still, the slide transition should be
            // shown).
        }
    }
    return mpCurrentSlideTransitionSound;
}

ActivitySharedPtr SlideShowImpl::createSlideTransition(
    const uno::Reference< drawing::XDrawPage >& xDrawPage,
    const SlideSharedPtr&                       rLeavingSlide,
    const SlideSharedPtr&                       rEnteringSlide,
    const EventSharedPtr&                       rTransitionEndEvent )
{
    ENSURE_AND_THROW( !maViewContainer.empty(),
                      "createSlideTransition(): No views" );
    
    // return empty transition, if slide transitions
    // are disabled.
    if (mbNoSlideTransitions)
        return ActivitySharedPtr();
    
    // retrieve slide change parameters from XDrawPage
    uno::Reference< beans::XPropertySet > xPropSet( xDrawPage,
                                                    uno::UNO_QUERY_THROW );

    sal_Int16 nTransitionType;
    if( !extractValue( nTransitionType,
                       xPropSet->getPropertyValue( OUSTR("TransitionType") ),
                       ShapeSharedPtr(),
                       LayerManagerSharedPtr() ) )
    {
        OSL_ENSURE( false,
                    "createSlideTransition(): "
                    "Could not extract slide transition type from XDrawPage" );
        return ActivitySharedPtr();
    }
    
    sal_Int16 nTransitionSubType;
    if( !extractValue( nTransitionSubType,
                       xPropSet->getPropertyValue( OUSTR("TransitionSubtype") ),
                       ShapeSharedPtr(),
                       LayerManagerSharedPtr() ) )
    {
        OSL_ENSURE(
            false, "createSlideTransition(): "
            "Could not extract slide transition subtype from XDrawPage" );
        return ActivitySharedPtr();
    }
    
    bool bTransitionDirection;
    if( !extractValue( bTransitionDirection,
                       xPropSet->getPropertyValue(OUSTR("TransitionDirection")),
                       ShapeSharedPtr(),
                       LayerManagerSharedPtr() ) )
    {
        OSL_ENSURE(
            false, "createSlideTransition(): "
            "Could not extract slide transition direction from XDrawPage" );
        return ActivitySharedPtr();
    }
    
    RGBColor aTransitionFadeColor;
    if( !extractValue( aTransitionFadeColor,
                       xPropSet->getPropertyValue(OUSTR("TransitionFadeColor")),
                       ShapeSharedPtr(),
                       LayerManagerSharedPtr() ) )
    {
        OSL_ENSURE(
            false, "createSlideTransition(): "
            "Could not extract slide transition fade color from XDrawPage" );
        return ActivitySharedPtr();
    }

    rtl::OUString aSoundURL;
    if( !extractValue( aSoundURL,
                       xPropSet->getPropertyValue( OUSTR("Sound") ),
                       ShapeSharedPtr(),
                       LayerManagerSharedPtr() ) )
    {
        OSL_ENSURE(
            false, "createSlideTransition(): "
            "Could not determine transition sound effect URL from XDrawPage" );
        return ActivitySharedPtr();
    }
    
    SlideChangeAnimationSharedPtr pTransition(
        TransitionFactory::createSlideTransition(
            rLeavingSlide,
            rEnteringSlide,
            nTransitionType,
            nTransitionSubType,
            bTransitionDirection,
            aTransitionFadeColor,
            resetSlideTransitionSound( aSoundURL ) ) );
    
    if( !pTransition.get() )
        return ActivitySharedPtr(); // no transition effect
    // has been
    // generated. Normally,
    // that means that simply
    // no transition is set on
    // this slide.

    // add all current views to the transition
    std::for_each( maViewContainer.begin(),
                   maViewContainer.end(),
                   boost::bind( &SlideChangeAnimation::addView,
                                boost::ref(pTransition), _1 ) );
    
    double nTransitionDuration;
    if( !extractValue( nTransitionDuration,
                       xPropSet->getPropertyValue(OUSTR("TransitionDuration")),
                       ShapeSharedPtr(),
                       LayerManagerSharedPtr() ) )
    {
        OSL_ENSURE(
            false, "createSlideTransition(): "
            "Could not extract slide transition duration from XDrawPage" );
        return ActivitySharedPtr();
    }
    
    return ActivitySharedPtr(
        ActivitiesFactory::createSimpleActivity(
            ActivitiesFactory::CommonParameters(
                rTransitionEndEvent,
                maEventQueue,
                maActivitiesQueue,
                nTransitionDuration,
                5,
                false,
                boost::optional<double>(1.0),
                0.0,
                0.0,
                ShapeSharedPtr(),
                LayerManagerSharedPtr() ),
            pTransition,
            true ) );
}

struct SlideShowImpl::ShapeListenerRegisterer {
    SlideSharedPtr const& mpSlide;
    explicit ShapeListenerRegisterer( SlideSharedPtr const& pSlide )
        : mpSlide(pSlide) {}
    void operator()( ShapeEventListenerMap::value_type const& rEntry ) const {
        std::for_each( rEntry.second.begin(), rEntry.second.end(),
                       boost::bind( &Slide::addShapeEventListener,
                                    boost::ref(mpSlide), _1,
                                    boost::cref(rEntry.first) ) );
    }
};
SlideSharedPtr SlideShowImpl::createSlide(
    uno::Reference<drawing::XDrawPage> const& xDrawPage,
    uno::Reference<animations::XAnimationNode> const& xRootNode )
{
    if (! xDrawPage.is())
        return SlideSharedPtr();
    
    SlideSharedPtr const pSlide( new Slide( xDrawPage, xRootNode,
                                            maEventQueue,
                                            maActivitiesQueue,
                                            *mpEventMultiplexer,
                                            maUserEventQueue,
                                            mxComponentContext ) );
    
    // add all registered views also to the slide
    std::for_each( maViewContainer.begin(), maViewContainer.end(),
                   boost::bind( &Slide::addView, boost::ref(pSlide), _1) );
    // add all registered shape listeners also to the shape
    std::for_each( maShapeEventListeners.begin(), maShapeEventListeners.end(),
                   ShapeListenerRegisterer(pSlide) );
    // add all registered shape cursors also to the shape
    std::for_each( maShapeCursors.begin(), maShapeCursors.end(),
                   boost::bind(
                       &Slide::setShapeCursor, boost::ref(pSlide),
                       boost::bind(
                           std::select1st<ShapeCursorMap::value_type>(), _1 ),
                       boost::bind(
                           std::select2nd<ShapeCursorMap::value_type>(), _1 )
                       ) );
    
    // forward applicable slideshow attributes to newly
    // generated slide
    pSlide->setUserPaintColor( maUserPaintColor );
    pSlide->setImageAnimationsAllowed( mbImageAnimationsAllowed );
    
    updateMouseCursor();
            
    // prefetch show content (reducing latency for slide 
    // bitmap and effect start later on)
    pSlide->prefetchShow();

    return pSlide;
}

void SlideShowImpl::setWaitState( bool bOn )
{
    mbWaitState = bOn;
    if (! mpWaitSymbol) // fallback to cursor
        updateMouseCursor();
    else if (mbWaitState)
        mpWaitSymbol->show();
    else
        mpWaitSymbol->hide();
}
    
void SlideShowImpl::updateMouseCursor()
{
    sal_Int16 mouseCursor = awt::SystemPointer::ARROW;
    if (mbWaitState && !mpWaitSymbol) // fallback to cursor
        mouseCursor = awt::SystemPointer::WAIT;
    else if (! mbMouseVisible) // INVISIBLE overrides user paint overlay
        mouseCursor = awt::SystemPointer::INVISIBLE;
    else if (maUserPaintColor)
        mouseCursor = awt::SystemPointer::PEN;
            
    mpEventMultiplexer->setMouseCursor( mouseCursor );
}
    
void SlideShowImpl::stopShow()
{
    // Force-end running animation
    // ===========================
    if (mpCurrentSlide)
        mpCurrentSlide->end();
    
    // clear all queues
    maEventQueue.clear();
    maActivitiesQueue.clear();

    // Attention: we MUST clear the user event queue here,
    // this is because the current slide might have registered
    // shape events (click or enter/leave), which might
    // otherwise dangle forever in the queue (because of the
    // shared ptr nature). If someone needs to change this:
    // somehow unregister those shapes at the user event queue
    // on notifySlideEnded().
    maUserEventQueue.clear();

    // re-enable automatic effect advancement
    // (maEventQueue.clear() above might have killed
    // mpEventMultiplexer's tick events)
    if (mbAutomaticAdvancementMode) {
        // toggle automatic mode (enabling just again is
        // ignored by EventMultiplexer)
        mpEventMultiplexer->setAutomaticMode( false );
        mpEventMultiplexer->setAutomaticMode( true );
    }
    
    // stop slide transition sound, if any:
    resetSlideTransitionSound();
}

struct SlideShowImpl::PrefetchPropertiesFunc {
    SlideShowImpl *const that;
    PrefetchPropertiesFunc( SlideShowImpl * that_ ) : that(that_) {}
    void operator()( beans::PropertyValue const& rProperty ) const {
        if (rProperty.Name.equalsAsciiL(
                RTL_CONSTASCII_STRINGPARAM("Prefetch") )) {
            uno::Sequence<uno::Any> seq;
            if ((rProperty.Value >>= seq) && seq.getLength() == 2) {
                seq[0] >>= that->mxPrefetchSlide;
                seq[1] >>= that->mxPrefetchAnimationNode;
            }
        }
        else {
            OSL_ENSURE( false, rtl::OUStringToOString(
                            rProperty.Name, RTL_TEXTENCODING_UTF8 ).getStr() );
        }
    }
};
void SlideShowImpl::displaySlide(
    uno::Reference<drawing::XDrawPage> const& xSlide,
    uno::Reference<animations::XAnimationNode> const& xRootNode,
    uno::Sequence<beans::PropertyValue> const& rProperties )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return;
    
    stopShow();  // MUST call that: results in
    // maUserEventQueue.clear(). What's more,
    // stopShow()'s currSlide->end() call is
    // now also required, notifySlideEnded()
    // relies on that
    // unconditionally. Otherwise, genuine
    // shape animations (drawing layer and
    // GIF) will not be stopped.
    
    std::for_each( rProperties.getConstArray(),
                   rProperties.getConstArray() + rProperties.getLength(),
                   PrefetchPropertiesFunc(this) );
    
    OSL_ENSURE( !maViewContainer.empty(), "### no views!" );
    if (maViewContainer.empty())
        return;
    
    // this here might take some time
    {
        comphelper::ScopeGuard const guard(
            boost::bind( &SlideShowImpl::setWaitState, this, false ) );
        setWaitState(true);
        
        mpPreviousSlide = mpCurrentSlide;
        mpCurrentSlide.reset();
        if (matches( mpPrefetchSlide, xSlide, xRootNode )) {
            // prefetched slide matches:
            mpCurrentSlide = mpPrefetchSlide;
        }
        else {
            mpCurrentSlide = createSlide( xSlide, xRootNode );
        }
        
        OSL_ASSERT( mpCurrentSlide );
        if (mpCurrentSlide) {
            // push new transformation to all views
            basegfx::B2ISize const slideSize_( mpCurrentSlide->getSlideSize() );
            basegfx::B2DSize const slideSize( slideSize_.getX(),
                                              slideSize_.getY());
            std::for_each( maViewContainer.begin(), maViewContainer.end(),
                           boost::bind( &View::setViewSize, _1,
                                        boost::cref(slideSize) ) );
            
            // explicitly notify view change here,
            // because transformation might have changed:
            // optimization, this->notifyViewChange() would
            // repaint slide which is not necessary.
            if (mpRehearseTimingsActivity)
                mpRehearseTimingsActivity->notifyViewChange();
            if (mpWaitSymbol)
                mpWaitSymbol->notifyViewChange();
            
            EventSharedPtr const pTransitionEndEvent(
                makeEvent( boost::bind(
                               &SlideShowImpl::notifySlideTransitionEnded,
                               this ) ) );
            
            // create slide transition, and add proper end event 
            // (which then starts the slide effects
            // via CURRENT_SLIDE.show())
            ActivitySharedPtr const pSlideChangeActivity(
                createSlideTransition( mpCurrentSlide->getXDrawPage(),
                                       mpPreviousSlide,
                                       mpCurrentSlide,
                                       pTransitionEndEvent ) );
            
            if (pSlideChangeActivity) {
                // factory generated a slide transition - activate it!
                maActivitiesQueue.addActivity( pSlideChangeActivity );
            }
            else {
                // no transition effect on this slide - schedule slide
                // effect start event right away.
                maEventQueue.addEvent( pTransitionEndEvent );
            }
        }
    } // finally
    
    updateMouseCursor();
}

sal_Bool SlideShowImpl::nextEffect() throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return false;
    
    if (mbShowPaused)
        return true;
    else
        return mpEventMultiplexer->notifyNextEffect();
}

sal_Bool SlideShowImpl::startShapeActivity(
    uno::Reference<drawing::XShape> const& xShape )
    throw (uno::RuntimeException)
{
    // TODO(F3): NYI
    OSL_ENSURE( false, "not yet implemented!" );
    return false;
}

sal_Bool SlideShowImpl::stopShapeActivity(
    uno::Reference<drawing::XShape> const& xShape )
    throw (uno::RuntimeException)
{
    // TODO(F3): NYI
    OSL_ENSURE( false, "not yet implemented!" );
    return false;
}

sal_Bool SlideShowImpl::pause( sal_Bool bPauseShow )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return false;
    
    if (bPauseShow)
        mpPresTimer->pauseTimer();
    else
        mpPresTimer->continueTimer();
    
    mpEventMultiplexer->notifyPauseMode(bPauseShow);
    
    mbShowPaused = bPauseShow;
    return true;
}

uno::Reference<drawing::XDrawPage> SlideShowImpl::getCurrentSlide()
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return uno::Reference<drawing::XDrawPage>();
    
    if (mpCurrentSlide)
        return mpCurrentSlide->getXDrawPage();
    else
        return uno::Reference<drawing::XDrawPage>();
}

sal_Bool SlideShowImpl::addView(
    uno::Reference<css::presentation::XSlideShowView> const& xView )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return false;
    
    // first of all, check if view has a valid canvas
    ENSURE_AND_RETURN( xView.is(), "addView(): Invalid view" );
    ENSURE_AND_RETURN( xView->getCanvas().is(),
                       "addView(): View does not provide a valid canvas" );
    
    UnoViewSharedPtr const pView( SlideView::create(
                                      xView, maEventQueue, *this ) );
    if (! maViewContainer.addView( pView ))
        return false;
    
    // forward to subordinates
    // =======================

    forEachSlide( boost::bind( &Slide::addView, _1, boost::cref(pView) ) );
    
    if (mpRehearseTimingsActivity) {
        // new view for timer:
        mpRehearseTimingsActivity->addView( pView );
    }
    
    if (mpWaitSymbol) {
        // new view for wait symbol:
        mpWaitSymbol->addView( pView );
    }
            
    // forward new view to EventMultiplexer (mouse events etc.)
    mpEventMultiplexer->addView( pView );
    
    // initialize view content
    // =======================

    if (mpCurrentSlide) {
        // set view transformation
        const basegfx::B2ISize slideSize = mpCurrentSlide->getSlideSize();
        pView->setViewSize( basegfx::B2DSize( slideSize.getX(),
                                              slideSize.getY() ) );
    }

    // clear view area (since its newly added, 
    // we need a clean slate)
    pView->clear();

    return true;
}

sal_Bool SlideShowImpl::removeView(
    uno::Reference<css::presentation::XSlideShowView> const& xView )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    
    ENSURE_AND_RETURN( xView.is(), "removeView(): Invalid view" );
    
    UnoViewSharedPtr const pView( maViewContainer.removeView( xView ) );
    if (! pView)
        return false;
    
    if (mpRehearseTimingsActivity) {
        // disconnect timer from view:
        mpRehearseTimingsActivity->removeView( pView );
    }

    if (mpWaitSymbol) {
        // disconnect wait symbol from view:
        mpWaitSymbol->removeView( pView );
    }
            
    // remove view from all slides:
    forEachSlide( boost::bind( &Slide::removeView, _1, boost::cref(pView) ) );
    
    // remove view from EventMultiplexer (mouse events etc.)
    if (mpEventMultiplexer.is())
        mpEventMultiplexer->removeView( pView );
    
    return true;
}

sal_Bool SlideShowImpl::setProperty( beans::PropertyValue const& rProperty )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return false;
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("AutomaticAdvancement") ))
    {
        double nTimeout;
        mbAutomaticAdvancementMode = (rProperty.Value >>= nTimeout);
        if (mbAutomaticAdvancementMode) {
            mpEventMultiplexer->setAutomaticTimeout( nTimeout );
        }
        mpEventMultiplexer->setAutomaticMode( mbAutomaticAdvancementMode );
        return true;
    }
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("UserPaintColor") ))
    {
        sal_Int32 nColor;
        if (rProperty.Value >>= nColor) {
            OSL_ENSURE( mbMouseVisible,
                        "setProperty(): User paint overrides invisible mouse" );
            // enable user paint
            maUserPaintColor.reset( unoColor2RGBColor( nColor ) );
        }
        else {
            // disable user paint
            maUserPaintColor.reset();
        }
        forEachSlide( boost::bind( &Slide::setUserPaintColor, _1,
                                   boost::cref(maUserPaintColor) ) );
        updateMouseCursor();
        return true;
    }
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("AdvanceOnClick") )) {
        sal_Bool bAdvanceOnClick;
        if (! (rProperty.Value >>= bAdvanceOnClick))
            return false;
        maUserEventQueue.setAdvanceOnClick( bAdvanceOnClick );
        return true;
    }            
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("ImageAnimationsAllowed") ) )
    {
        if (! (rProperty.Value >>= mbImageAnimationsAllowed))
            return false;
        forEachSlide( boost::bind( &Slide::setImageAnimationsAllowed,
                                   _1, mbImageAnimationsAllowed ) );
        return true;
    }
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("MouseVisible") ))
    {
        if (! (rProperty.Value >>= mbMouseVisible))
            return false;
        updateMouseCursor();
        return true;
    }
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("ForceManualAdvance") )) {
        return (rProperty.Value >>= mbForceManualAdvance);
    }
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("RehearseTimings") ))
    {
        bool bRehearseTimings = false;
        if (! (rProperty.Value >>= bRehearseTimings))
            return false;
        
        if (bRehearseTimings) {
            mpRehearseTimingsActivity = RehearseTimingsActivity::create(
                maEventQueue, *mpEventMultiplexer, maActivitiesQueue );
            // add all previously registered views:
            std::for_each( maViewContainer.begin(),
                           maViewContainer.end(),
                           boost::bind( &RehearseTimingsActivity::addView,
                                        boost::cref(mpRehearseTimingsActivity),
                                        _1 ) );
        }
        else if (mpRehearseTimingsActivity) {
            // removes timer from all views:
            mpRehearseTimingsActivity->dispose();
            mpRehearseTimingsActivity.reset();
        }
        return true;
    }
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("WaitSymbolBitmap") ))
    {
        uno::Reference<rendering::XBitmap> xBitmap;
        if (! (rProperty.Value >>= xBitmap))
            return false;
        
        if (mpWaitSymbol)
            mpWaitSymbol->dispose();
        mpWaitSymbol.reset( new WaitSymbol( xBitmap, *mpEventMultiplexer ) );
        
        // add all previously registered views:
        std::for_each( maViewContainer.begin(), maViewContainer.end(),
                       boost::bind( &WaitSymbol::addView,
                                    boost::cref(mpWaitSymbol), _1 ) );
        return true;
    }
    
    if (rProperty.Name.equalsAsciiL(
            RTL_CONSTASCII_STRINGPARAM("NoSlideTransitions") )) {
        return (rProperty.Value >>= mbNoSlideTransitions);
    }
    
    return false;
}

void SlideShowImpl::addSlideShowListener(
    uno::Reference<css::presentation::XSlideShowListener> const& xListener )
    throw (uno::RuntimeException)
{
    if (isDisposed())
        return;
    // container syncs with passed mutex ref
    maListenerContainer.addInterface(xListener);
}

void SlideShowImpl::removeSlideShowListener(
    uno::Reference<css::presentation::XSlideShowListener> const& xListener )
    throw (uno::RuntimeException)
{
    // container syncs with passed mutex ref
    maListenerContainer.removeInterface(xListener);
}

void SlideShowImpl::addShapeEventListener(
    uno::Reference<css::presentation::XShapeEventListener> const& xListener,
    uno::Reference<drawing::XShape> const& xShape )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return;
    
    ShapeEventListenerMap::iterator aIter;
    if( (aIter=maShapeEventListeners.find( xShape )) ==
        maShapeEventListeners.end() )
    {
        // no entry for this shape -> create one
        maShapeEventListeners.insert( 
            ShapeEventListenerMap::value_type(
                xShape,
                ShapeEventListenerMap::value_type::second_type(1, xListener) ));
    }
    else {
        // aIter references to an existing entry. Just add the
        // listener there
        aIter->second.push_back( xListener );
    }
    
    // add to each already buffered slide:
    forEachSlide( boost::bind(
                      &Slide::addShapeEventListener,
                      _1, boost::cref(xListener), boost::cref(xShape) ) );
}

void SlideShowImpl::removeShapeEventListener(
    uno::Reference<css::presentation::XShapeEventListener> const& xListener,
    uno::Reference<drawing::XShape> const& xShape )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    
    ShapeEventListenerMap::iterator aIter;
    if( (aIter=maShapeEventListeners.find( xShape )) !=
        maShapeEventListeners.end() )
    {
        // entry for this shape found -> remove listener from the list
        ShapeEventListenerMap::value_type::second_type::iterator aListener(
            aIter->second.end() );
        const ShapeEventListenerMap::value_type::second_type::iterator aEnd(
            aIter->second.end() );
        if( (aListener=::std::remove( aIter->second.begin(), aEnd, xListener ))
            != aEnd ) {
            // actually erase from container
            aIter->second.erase( aListener, aEnd );
        }
    }

    // remove from each already buffered slide:
    forEachSlide( boost::bind(
                      &Slide::removeShapeEventListener,
                      _1, boost::cref(xListener), boost::cref(xShape) ) );
}

void SlideShowImpl::setShapeCursor(
    uno::Reference<drawing::XShape> const& xShape, sal_Int16 nPointerShape )
    throw (uno::RuntimeException)
{            
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return;
    
    ShapeCursorMap::iterator aIter;
    if( (aIter=maShapeCursors.find( xShape )) == maShapeCursors.end() ) {
        // no entry for this shape -> create one
        if( nPointerShape != awt::SystemPointer::ARROW ) {
            // add new entry, unless shape shall display
            // normal pointer arrow -> no need to handle that
            // case
            maShapeCursors.insert(
                ShapeCursorMap::value_type(xShape, nPointerShape) );
        }
    }
    else if( nPointerShape == awt::SystemPointer::ARROW ) {
        // shape shall display normal cursor -> can disable
        // the cursor and clear the entry
        maShapeCursors.erase( xShape );
    }
    else {
        // existing entry found, update with new cursor ID
        aIter->second = nPointerShape;
    }

    // forward to slides
    forEachSlide( boost::bind( &Slide::setShapeCursor, _1,
                               boost::cref(xShape), nPointerShape ) );
}

sal_Bool SlideShowImpl::update( double & nNextTimeout )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    if (isDisposed())
        return false;
    
    // precondition: update() must only be called from the
    // main thread!
    DBG_TESTSOLARMUTEX();

    if( mbShowPaused ) {
        return false;
    }
    else {
        // hold timer, while processing the queues (ensures
        // same time for all activities and events)
        {
            comphelper::ScopeGuard const guard(
                boost::bind( &canvas::tools::ElapsedTime::releaseTimer,
                             boost::cref(mpPresTimer) ) );
            mpPresTimer->holdTimer();
                    
            // process queues:
            maEventQueue.process();
            maActivitiesQueue.process();
        }
        // Time held until here
                
        const bool bActivitiesLeft = (! maActivitiesQueue.isEmpty());
        const bool bTimerEventsLeft = (! maEventQueue.isEmpty());
        const bool bRet = (bActivitiesLeft || bTimerEventsLeft);
                
        if (bRet) {
            // calc nNextTimeout value:
            if (bActivitiesLeft) {
                // activities left: requires immediate updates
                nNextTimeout = 0.0; // come back ASAP
            }
            else {
                // timer events left:
                // difference from current time (nota bene:
                // time no longer held here!) to the next event in
                // the event queue.

                // #i61190# Retrieve next timeout only _after_
                // processing activity queue

                // ensure positive value:
                nNextTimeout = std::max( 0.0, maEventQueue.nextTimeout() );
            }

            mbSlideShowIdle = false;
        }

#if defined(VERBOSE) && defined(DBG_UTIL)
        // when slideshow is idle, issue an XUpdatable::update() call
        // exactly once after a previous animation sequence finished -
        // this might trigger screen dumps on some canvas
        // implementations
        if( !mbSlideShowIdle &&
            (!bRet ||
             nNextTimeout > 1.0) )
        {
            UnoViewVector::const_iterator       aCurr(maViewContainer.begin());
            const UnoViewVector::const_iterator aEnd(maViewContainer.end());
            while( aCurr != aEnd )
            {
                try
                {
                    uno::Reference< css::presentation::XSlideShowView > xView( (*aCurr)->getUnoView(),
                                                                               uno::UNO_QUERY_THROW );
                    uno::Reference< util::XUpdatable >             xUpdatable( xView->getCanvas(),
                                                                               uno::UNO_QUERY_THROW );
                    xUpdatable->update();
                } 
                catch( uno::Exception& )
                {
                }

                ++aCurr;
            }

            mbSlideShowIdle = true;
        }
#endif
        
        return bRet;
    }
}

void SlideShowImpl::notifySlideTransitionEnded()
{
    OSL_ENSURE( !isDisposed(), "### already disposed!" );
    OSL_ENSURE( mpCurrentSlide,
                "notifySlideTransitionEnded(): Invalid current slide" );
    if (mpCurrentSlide) {
        // first init show, to give the animations
        // the chance to register SlideStartEvents
        mpCurrentSlide->show();
        mpEventMultiplexer->notifySlideStartEvent();
    }
}

void SlideShowImpl::notifySlideAnimationsEnded()
{
    OSL_ENSURE( !isDisposed(), "### already disposed!" );
    
    // This struct will receive the (interruptable) event,
    // that triggers the notifySlideEnded() method.
    InterruptableEventPair aNotificationEvents;

    if( mpEventMultiplexer->getAutomaticMode() ) {
        OSL_ENSURE( ! mpRehearseTimingsActivity,
                    "unexpected: RehearseTimings mode!" );
                
        // schedule a slide end event, with automatic mode's
        // delay
        aNotificationEvents = makeInterruptableDelay(
            boost::bind( &SlideShowImpl::notifySlideEnded, this ),
            mpEventMultiplexer->getAutomaticTimeout() );
    }
    else {
        OSL_ENSURE( mpCurrentSlide,
                    "notifySlideAnimationsEnded(): Invalid current slide!" );
        
        // check whether slide transition should happen
        // 'automatically'. If yes, simply schedule the
        // specified timeout.
        // NOTE: mbForceManualAdvance and mpRehearseTimingsActivity
        // override any individual slide setting, to always
        // step slides manually.
        if( !mbForceManualAdvance &&
            !mpRehearseTimingsActivity &&
            mpCurrentSlide && mpCurrentSlide->hasAutomaticNextSlide() )
        {
            aNotificationEvents = makeInterruptableDelay(
                boost::bind( &SlideShowImpl::notifySlideEnded, this ),
                mpCurrentSlide->getAutomaticNextSlideTimeout() );
            
            // TODO(F2): Provide a mechanism to let the user override 
            // this automatic timeout via next()
        }
        else {
            if (mpRehearseTimingsActivity)
                mpRehearseTimingsActivity->start();
                    
            // generate click event. Thus, the user must
            // trigger the actual end of a slide. No need to
            // generate interruptable event here, there's no
            // timeout involved.
            aNotificationEvents.mpImmediateEvent = 
                makeEvent( boost::bind(
                               &SlideShowImpl::notifySlideEnded, this ) );
        }
    }
    
    // register events on the queues. To make automatic slide
    // changes interruptable, register the interruption event
    // as a nextEffectEvent target. Note that the timeout
    // event is optional (e.g. manual slide changes don't
    // generate a timeout)            
    maUserEventQueue.registerNextEffectEvent(
        aNotificationEvents.mpImmediateEvent );
    
    if( aNotificationEvents.mpTimeoutEvent.get() != NULL )
        maEventQueue.addEvent( aNotificationEvents.mpTimeoutEvent );
    
    // current slide's main sequence is over. Now should be
    // the time to prefetch the next slide (if any), and
    // prepare the initial slide bitmap (speeds up slide
    // change setup time a lot). Show the wait cursor, this
    // indeed might take some seconds.
    {
        comphelper::ScopeGuard const guard(
            boost::bind( &SlideShowImpl::setWaitState, this, false ) );
        setWaitState(true);
        
        if (! matches( mpPrefetchSlide,
                       mxPrefetchSlide, mxPrefetchAnimationNode )) {
            mpPrefetchSlide = createSlide( mxPrefetchSlide,
                                           mxPrefetchAnimationNode );
        }
        if (mpPrefetchSlide) {
            // ignore return value, this is just to populate
            // Slide's internal bitmap buffer, such that the time
            // needed to generate the slide bitmap is not spent
            // when the slide change is requested.
            mpPrefetchSlide->getCurrentSlideBitmap( *maViewContainer.begin() );
        }
    } // finally
}

void SlideShowImpl::notifySlideEnded()
{
    OSL_ENSURE( !isDisposed(), "### already disposed!" );
    
    if (mpRehearseTimingsActivity) {
        const double time = mpRehearseTimingsActivity->stop();
        if (mpRehearseTimingsActivity->hasBeenClicked()) {
            // save time at current drawpage:
            uno::Reference<beans::XPropertySet> xPropSet(
                mpCurrentSlide->getXDrawPage(), uno::UNO_QUERY );
            OSL_ASSERT( xPropSet.is() );
            if (xPropSet.is()) {
                xPropSet->setPropertyValue(
                    OUSTR("Change"),
                    uno::Any( static_cast<sal_Int32>(1) ) );
                xPropSet->setPropertyValue(
                    OUSTR("Duration"),
                    uno::Any( static_cast<sal_Int32>(time) ) );
            }
        }
    }
    
    mpEventMultiplexer->notifySlideEndEvent();
    
    stopShow();  // MUST call that: results in
    // maUserEventQueue.clear(). What's more,
    // stopShow()'s currSlide->end() call is
    // now also required, notifySlideEnded()
    // relies on that
    // unconditionally. Otherwise, genuine
    // shape animations (drawing layer and
    // GIF) will not be stopped.
    
    // force default cursor here:
    mpEventMultiplexer->setMouseCursor( awt::SystemPointer::ARROW );
    
    // TODO(T2): Attention, this is not thread safe. We're 
    // calling out listeners with locked mutex!
    maListenerContainer.forEach<css::presentation::XSlideShowListener>(
        boost::mem_fn( &css::presentation::XSlideShowListener::slideEnded ) );
}

void SlideShowImpl::notifyViewChange()
{
    OSL_ENSURE( !isDisposed(), "### already disposed!" );
    
    // TODO(P1): This should be restricted to the view which 
    // actually _change_ (and not to all views)

    // TODO(Q3): handle this via the new EventMultiplexer scheme!

    // forward view change to active slide
    if (mpCurrentSlide)
        mpCurrentSlide->paint();

    if (mpWaitSymbol)
        mpWaitSymbol->notifyViewChange();
            
    if (mpRehearseTimingsActivity)
        mpRehearseTimingsActivity->notifyViewChange();
}

bool SlideShowImpl::notifyHyperLinkClicked( rtl::OUString const& hyperLink )
{
    maListenerContainer.forEach<css::presentation::XSlideShowListener>(
        boost::bind( &css::presentation::XSlideShowListener::hyperLinkClicked,
                     _1, boost::cref(hyperLink) ) );
    return true;
}


/******************************************************************************
    
    SlideView implementation
    
 ******************************************************************************/

SlideView::SlideView(
    const uno::Reference<css::presentation::XSlideShowView>& xView,
    EventQueue & rEventQueue, SlideShowImpl & rShow )
    : SlideViewBase( m_aMutex ),
      mxView( xView ),
      mpCanvas(),
      mrEventQueue( rEventQueue ),
      mrShow( rShow ),
      maUnitRectPoly( basegfx::tools::createPolygonFromRect(
                          basegfx::B2DRectangle( 0.0, 0.0,
                                                 1.0, 1.0 ) ) ),
      maClip(),
      maViewTransform(),
      maUserSize( 1.0, 1.0 ), // default size: one-by-one rectangle
      mbContentValid( false )
{
    // keep care not constructing any UNO references to this inside ctor,
    // shift that code to create()!
    
    ENSURE_AND_THROW( mxView.is(),
                      "SlideView::SlideView(): Invalid view" );
    
    mpCanvas = cppcanvas::VCLFactory::getInstance().createSpriteCanvas(
        xView->getCanvas() );
    ENSURE_AND_THROW( mpCanvas.get(), "Could not create cppcanvas" );
    
    basegfx::unotools::homMatrixFromAffineMatrix(
        maViewTransform, xView->getTransformation() );    
}

UnoViewSharedPtr SlideView::create(
    uno::Reference<css::presentation::XSlideShowView> const& xView,
    EventQueue & rEventQueue, SlideShowImpl & rShow )
{
    boost::shared_ptr<SlideView> const that(
        comphelper::make_shared_from_UNO(
            new SlideView(xView, rEventQueue, rShow) ) );
    
    // register listeners with XSlideShowView
    xView->addTransformationChangedListener( that.get() );
    xView->addPaintListener( that.get() );
    
    // set new transformation
    that->updateCanvas();
    
    return that;
}

SlideView::~SlideView()
{
    // WeakComponentImplHelperBase self disposes if undisposed
}

// we are about to be disposed (someone call dispose() on us)
void SlideView::disposing()
{
    // additionally, also de-register from XSlideShowView
    if (mxView.is()) {
        mxView->removeTransformationChangedListener( this );
        mxView->removePaintListener( this );
        mxView.clear();
    }
    mpCanvas.reset();
    WeakComponentImplHelperBase::disposing();
}

class SlideView::SlideViewLayer : public ViewLayer, private boost::noncopyable
{
    cppcanvas::SpriteCanvasSharedPtr const mpCanvas;
public:
    explicit SlideViewLayer( cppcanvas::SpriteCanvasSharedPtr const& pCanvas )
        : mpCanvas(pCanvas) {}
    virtual ::cppcanvas::CustomSpriteSharedPtr createSprite(
        const ::basegfx::B2DSize& rSpriteSizePixel ) const
    {
        return mpCanvas->createCustomSprite( rSpriteSizePixel );
    }
    virtual double getSpritePriority( double nSpritePrio ) const
    {
        // TODO(F2): Sprite/Layer/View priority
        return nSpritePrio;
    }
    virtual void setPriority( double nPrio )
    {
        // TODO(F2): Sprite/Layer/View priority
    }
    virtual cppcanvas::CanvasSharedPtr getCanvas() const
    {
        return mpCanvas;
    }
};
ViewLayerSharedPtr SlideView::createViewLayer() const
{
    osl::MutexGuard aGuard( m_aMutex );
    ENSURE_AND_THROW( mpCanvas, "SlideView::createViewLayer(): Disposed" );
    return ViewLayerSharedPtr( new SlideViewLayer(mpCanvas) );
}

bool SlideView::updateScreen() const
{
    osl::MutexGuard aGuard( m_aMutex );
    ENSURE_AND_THROW( mpCanvas.get(),
                      "SlideView::updateScreen(): Disposed" );
    return mpCanvas->updateScreen(
#if defined(VERBOSE) && OSL_DEBUG_LEVEL > 1
        true
#else
        false
#endif
        );
}

void SlideView::clear() const
{
    osl::MutexGuard aGuard( m_aMutex );
    OSL_ENSURE( mxView.is() && mpCanvas, "SlideView::clear(): Disposed" );
    // only clear the _complete_ view, if content
    // became invalid. Otherwise, since the
    // View::clear does not respect our clip,
    // partial updates might be spoiled (in that
    // the whole view gets cleared, but only a
    // small rectangle updated with slide content)
    if (!mbContentValid && mxView.is())
        mxView->clear();
    mbContentValid = true;
}

bool SlideView::isContentDestroyed() const
{
    return !mbContentValid;
}

cppcanvas::CanvasSharedPtr SlideView::getCanvas() const
{
    osl::MutexGuard aGuard( m_aMutex );
    OSL_ENSURE( mpCanvas, "SlideView::getCanvas(): Disposed" );
    return mpCanvas;
}

basegfx::B2DHomMatrix SlideView::getTransformation() const
{
    osl::MutexGuard aGuard( m_aMutex );
    basegfx::B2DHomMatrix aMatrix;
    aMatrix.scale( 1.0/maUserSize.getX(), 1.0/maUserSize.getY() );    
    return maViewTransform * aMatrix;
}

void SlideView::setViewSize( const basegfx::B2DSize& rSize )
{
    osl::MutexGuard aGuard( m_aMutex );
    maUserSize = rSize;
    updateCanvas();
    // view resized, have to update whole area
    mbContentValid = false;
}

cppcanvas::CustomSpriteSharedPtr SlideView::createSprite(
    const basegfx::B2DSize& rSpriteSizePixel ) const          
{
    osl::MutexGuard aGuard( m_aMutex );
    OSL_ENSURE( mpCanvas, "SlideView::createSprite(): Disposed" );
    if (mpCanvas)
        return mpCanvas->createCustomSprite( rSpriteSizePixel );
    else
        return cppcanvas::CustomSpriteSharedPtr();
}

void SlideView::setClip( const basegfx::B2DPolyPolygon& rClip )
{
    if( rClip.count() == 0 && maClip.count() == 0 ) {
        // bail out early for no-ops
        return;
    }
    maClip = rClip;
    
    if( maClip.areControlPointsUsed() )
        maClip = basegfx::tools::adaptiveSubdivideByAngle( maClip );
    
    // normalize polygon, preparation for clipping
    // in updateCanvas()
    maClip = basegfx::tools::correctOrientations( maClip );
    maClip = basegfx::tools::removeAllIntersections(maClip);
    maClip = basegfx::tools::removeNeutralPolygons(maClip, sal_True);
    
    updateCanvas();
}

void SlideView::setMouseCursor( sal_Int16 nPointerShape )
{
    osl::MutexGuard const guard( m_aMutex );
    if (mxView.is())
        mxView->setMouseCursor( nPointerShape );
}

double SlideView::getSpritePriority( double nSpritePrio ) const
{
    osl::MutexGuard aGuard( m_aMutex );
    // TODO(F2): Sprite/Layer/View priority
    return nSpritePrio;
}

void SlideView::setPriority( double nPrio )
{
//     osl::MutexGuard aGuard( m_aMutex );
    // TODO(F2): Sprite/Layer/View priority
}

uno::Reference<css::presentation::XSlideShowView> SlideView::getUnoView() const
{
    osl::MutexGuard aGuard( m_aMutex );
    return mxView;
}

// XEventListener
void SlideView::disposing( lang::EventObject const& evt )
    throw (uno::RuntimeException)
{
    // no deregistration necessary anymore, XView has left:
    osl::MutexGuard const guard( m_aMutex );
    if (mxView.is()) {
        OSL_ASSERT( evt.Source == mxView );
        mxView.clear();
    }
}

// XModifyListener
void SlideView::modified( const lang::EventObject& aEvent )
    throw (uno::RuntimeException)
{
    osl::MutexGuard const guard( m_aMutex );
    OSL_ENSURE( mxView.is(), "SlideView::modified(): "
                "Disposed, but event received from XSlideShowView?!");
    if (! mxView.is())
        return;
    
    // view transformation changed. Modify canvas matrix, then
    basegfx::unotools::homMatrixFromAffineMatrix(
        maViewTransform, mxView->getTransformation() );
    
    updateCanvas();
    
    // view resized, have to update whole area
    mbContentValid = false;
    
    // notify view change. Don't call notifyViewChange 
    // directly, this might not be the main thread!
    mrEventQueue.addEvent(
        makeEvent( boost::bind( &SlideShowImpl::notifyViewChange,
                                boost::ref( mrShow ) ) ) );
}

// XPaintListener
void SlideView::windowPaint( const awt::PaintEvent& e )
    throw (uno::RuntimeException)
{
    osl::MutexGuard aGuard( m_aMutex );
    OSL_ENSURE( mxView.is() && mpCanvas, "Disposed, but event received?!" );
    
    // TODO(F2): Don't always call update screen here,
    //           this might tear running 
    //           animations. Implementation should synchronize
    //           with ActivitiesQueue here, 
    //           and only update when no animations are active.
    
    // TODO(Q2): Watch out here, windowPaint()
    // might not be called from the main thread
    // (asking for SolarMutex trouble...)
    
    // repaint _whole_ screen, not only changed parts
    if (mpCanvas)
        mpCanvas->updateScreen( true );
}

void SlideView::updateCanvas()
{
    OSL_ENSURE( mpCanvas, "SlideView::updateCanvasTransform(): Disposed" );
    if (! mpCanvas)
        return;
    
    // setup canvas transformation
    // ===========================
    mpCanvas->setTransformation( getTransformation() );
    
    // setup canvas clipping
    // =====================
    
    // our view displays a one-by-one rectangle, modified by maUserSize 
    // (i.e. maViewTransform alone transforms a one-by-one rect to final
    // device coordinates). Thus, to display maUnitRectPoly correctly under
    // the new total view transform, we need to scale it up by maUserSize
    basegfx::B2DPolyPolygon aClipPoly( maUnitRectPoly );
    
    basegfx::B2DHomMatrix aMatrix;
    aMatrix.scale( maUserSize.getX(), maUserSize.getY() );
    
    aClipPoly.transform( aMatrix );
    
    // now clip with user-supplied clip poly (if any)
    if( maClip.count() ) {
        // no need to normalize aClipPoly, it's
        // just created positively oriented,
        // without any self-intersections
        aClipPoly.append( maClip );
        aClipPoly = basegfx::tools::removeAllIntersections(aClipPoly);
        aClipPoly = basegfx::tools::removeNeutralPolygons(
            aClipPoly, sal_False );
    }
    
    mpCanvas->setClip( aClipPoly );
}


/******************************************************************************
    
    UNO service stuff
    
 ******************************************************************************/

// XServiceInfo
sal_Bool SlideShowImpl::supportsService(
    rtl::OUString const& rServiceName ) throw (uno::RuntimeException)
{
    return getSupportedServiceNames_()[0].equals( rServiceName );
}

uno::Sequence<rtl::OUString> SlideShowImpl::getSupportedServiceNames()
    throw (uno::RuntimeException)
{
    return getSupportedServiceNames_();
}

rtl::OUString SlideShowImpl::getImplementationName()
    throw (uno::RuntimeException)
{
    return getImplementationName_();
}

// XServiceName
rtl::OUString SlideShowImpl::getServiceName() throw (uno::RuntimeException)
{
    return getSupportedServiceNames_()[0];
}

cppu::ImplementationEntry component_entries[] = {
    {
        SlideShowImpl::create, SlideShowImpl::getImplementationName_,
        SlideShowImpl::getSupportedServiceNames_,
        cppu::createSingleComponentFactory, 0, 0
    },
    { 0, 0, 0, 0, 0, 0 }
};

} // anon namespace

extern "C" {

void SAL_CALL component_getImplementationEnvironment(
    sal_Char const** ppEnvTypeName, uno_Environment ** ppEnv )
{
    *ppEnvTypeName = CPPU_CURRENT_LANGUAGE_BINDING_NAME;
}

sal_Bool SAL_CALL component_writeInfo( lang::XMultiServiceFactory * xMgr,
                                       registry::XRegistryKey * xRegistry )
{
    return cppu::component_writeInfoHelper( xMgr, xRegistry, component_entries);
}

void * SAL_CALL component_getFactory( sal_Char const* pImplName, 
                                      lang::XMultiServiceFactory * xMgr,
                                      registry::XRegistryKey * xRegistry )
{
    return cppu::component_getFactoryHelper(
        pImplName, xMgr, xRegistry, component_entries );
}

} // extern "C"

