--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Containers.Vectors;
with Ada.Strings.Unbounded;
with Ada.Unchecked_Deallocation;

with McKae.XML.EZ_Out.String_Stream;

with Exec_Command;
with Error_Message;
with Exception_Message;

package body Group_Command is

   use Ada.Strings.Unbounded;
   type Actor_Full_Name is record
      Sub_System_Name : Unbounded_String;
      Actor_Name : Unbounded_String;
   end record;
   package Actor_Vector is new Ada.Containers.Vectors (Positive,
                                                       Actor_Full_Name,
                                                       "=");
   use type Actor_Vector.Cursor;
   type Group_Type is record
      Name : Unbounded_String;
      Actors : Actor_Vector.Vector;
   end record;
   type Group_Access is access Group_Type;
   procedure Free is new Ada.Unchecked_Deallocation (Group_Type, Group_Access);
   package Group_Vector is new Ada.Containers.Vectors (Positive,
                                                       Group_Access,
                                                       "=");
   use type Group_Vector.Cursor;
   The_Groups : Group_Vector.Vector;

   -----------
   -- Group --
   -----------

   function Find_Group (Name : Unbounded_String) return Group_Vector.Cursor;
   function Find_Group (Name : Unbounded_String) return Group_Vector.Cursor is
      Group_Cursor : Group_Vector.Cursor;
      use Group_Vector;
   begin
      if Is_Empty (The_Groups) then
         return No_Element;
      end if;
      Group_Cursor := First (The_Groups);
      loop
         declare
            Group : constant Group_Access := Element (Group_Cursor);
         begin
            if Group.Name = Name then
               return Group_Cursor;
            end if;
         end;
         Group_Cursor := Next (Group_Cursor);
         exit when Group_Cursor = No_Element;
      end loop;
      return Group_Cursor;
   end Find_Group;

   function Find_Actor (Actors : Actor_Vector.Vector;
                        Sub_System_Name : Unbounded_String;
                        Name : Unbounded_String) return Actor_Vector.Cursor;
   function Find_Actor (Actors : Actor_Vector.Vector;
                        Sub_System_Name : Unbounded_String;
                        Name : Unbounded_String) return Actor_Vector.Cursor is
      Actor_Cursor : Actor_Vector.Cursor;
      use Actor_Vector;
   begin
      if Is_Empty (Actors) then
         return No_Element;
      end if;
      Actor_Cursor := First (Actors);
      loop
         declare
            Actor : constant Actor_Full_Name := Element (Actor_Cursor);
         begin
            if Actor.Actor_Name = Name and
              Actor.Sub_System_Name = Sub_System_Name then
               return Actor_Cursor;
            end if;
         end;
         Actor_Cursor := Next (Actor_Cursor);
         exit when Actor_Cursor = No_Element;
      end loop;
      return Actor_Cursor;
   end Find_Actor;

   function Group (Shell_Line : String) return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
      Group_Cursor : Group_Vector.Cursor;
   begin
      Clear (Xml_Buffer);
      declare
         Arguments : constant Shell_Commands.Arguments_Array :=
           Shell_Commands.Arguments (Shell_Line);
      begin
         if Arguments'Length = 1 then
            if Arguments (1) = "list" then
               if Group_Vector.Is_Empty (The_Groups) then
                  return "<error><severity>WARNING</severity>" &
                    "<msg>no group</msg></error>";
               end if;
               Start_Element (Xml_Buffer, "result");
               Group_Cursor := Group_Vector.First (The_Groups);
               loop
                  Output_Element (Xml_Buffer, "name",
                                  To_String
                                  (Group_Vector.Element (Group_Cursor).Name));
                  Group_Cursor := Group_Vector.Next (Group_Cursor);
                  exit when Group_Cursor = Group_Vector.No_Element;
               end loop;
               End_Element (Xml_Buffer, "result");
               declare
                  String_To_Return : constant String :=
                    Get_String (Xml_Buffer);
               begin
                  Full_Clear (Xml_Buffer);
                  return String_To_Return;
               end;
            else
               return Syntax_String;
            end if;
         elsif Arguments'Length = 2 then
            if Arguments (1) = "list" then
               Group_Cursor := Find_Group (Arguments (2));
               if Group_Cursor = Group_Vector.No_Element then
                  return Error_Message ("group",
                                        "WARNING",
                                        To_String (Arguments (2)) &
                                        " isn't a valid group name");
               end if;
               Start_Element (Xml_Buffer, "result");
               declare
                  Group : constant Group_Access :=
                    Group_Vector.Element (Group_Cursor);
                  Actor_Cursor : Actor_Vector.Cursor;
               begin
                  Output_Element (Xml_Buffer, "name",
                                  To_String (Group.Name));
                  Actor_Cursor := Actor_Vector.First (Group.Actors);
                  loop
                     Start_Element (Xml_Buffer, "actor");
                     declare
                        Actor : constant Actor_Full_Name :=
                          Actor_Vector.Element (Actor_Cursor);
                     begin
                        Output_Element
                          (Xml_Buffer,
                           "sub_system",
                           To_String (Actor.Sub_System_Name));
                        Output_Element (Xml_Buffer,
                                        "name",
                                        To_String (Actor.Actor_Name));
                     end;
                     End_Element (Xml_Buffer, "actor");
                     Actor_Cursor := Actor_Vector.Next (Actor_Cursor);
                     exit when Actor_Cursor = Actor_Vector.No_Element;
                  end loop;
               end;
               End_Element (Xml_Buffer, "result");
               declare
                  String_To_Return : constant String :=
                    Get_String (Xml_Buffer);
               begin
                  Full_Clear (Xml_Buffer);
                  return String_To_Return;
               end;
            else
               return Syntax_String;
            end if;
         elsif Arguments'Length = 3 then
            if Arguments (1) = "get" then
               Group_Cursor := Find_Group (Arguments (3));
               if Group_Cursor = Group_Vector.No_Element then
                  return Error_Message ("group",
                                        "WARNING",
                                        To_String (Arguments (3)) &
                                        " isn't a valid group name");
               end if;
               declare
                  Group : constant Group_Access :=
                    Group_Vector.Element (Group_Cursor);
                  Actor_Cursor : Actor_Vector.Cursor;
               begin
                  Start_Element (Xml_Buffer, "result");
                  Actor_Cursor := Actor_Vector.First (Group.Actors);
                  loop
                     Start_Element (Xml_Buffer, "actor");
                     declare
                        Actor : constant Actor_Full_Name :=
                          Actor_Vector.Element (Actor_Cursor);
                        Command : constant Unbounded_String := "get " &
                          Arguments (2) & " " & Actor.Sub_System_Name &
                          " " & Actor.Actor_Name;
                     begin
                        Output_Element
                          (Xml_Buffer,
                           "sub_system",
                           To_String (Actor.Sub_System_Name));
                        Output_Element (Xml_Buffer,
                                        "name",
                                        To_String (Actor.Actor_Name));
                        Put (Xml_Buffer,
                             Exec_Command (To_String (Command)));
                     end;
                     End_Element (Xml_Buffer, "actor");
                     Actor_Cursor := Actor_Vector.Next (Actor_Cursor);
                     exit when Actor_Cursor = Actor_Vector.No_Element;
                  end loop;
               end;
               declare
                  String_To_Return : constant String :=
                    Get_String (Xml_Buffer);
               begin
                  Full_Clear (Xml_Buffer);
                  return String_To_Return;
               end;
            else
               return Syntax_String;
            end if;
         elsif Arguments'Length = 4 then
            if Arguments (1) = "add" then
               Group_Cursor := Find_Group (Arguments (2));
               declare
                  Group : Group_Access;
               begin
                  if Group_Cursor = Group_Vector.No_Element then
                     Group := new Group_Type;
                     Group.Name := Arguments (2);
                     Group_Vector.Append (The_Groups, Group);
                  else
                     Group := Group_Vector.Element (Group_Cursor);
                  end if;
                  Actor_Vector.Append (Group.Actors,
                                       (Sub_System_Name => Arguments (3),
                                        Actor_Name => Arguments (4)));
               end;
               return "<result cmd=""group add"" status=""OK""></result>";
            elsif Arguments (1) = "del" then
               Group_Cursor := Find_Group (Arguments (2));
               if Group_Cursor = Group_Vector.No_Element then
                  return Error_Message ("group",
                                        "WARNING",
                                        To_String (Arguments (2)) &
                                        " isn't a valid group name");
               end if;
               declare
                  Group : Group_Access := Group_Vector.Element (Group_Cursor);
                  Actor_Cursor : Actor_Vector.Cursor;
               begin
                  Actor_Cursor := Find_Actor (Group.Actors,
                                              Arguments (3),
                                              Arguments (4));
                  if Actor_Cursor = Actor_Vector.No_Element then
                     return "<error><severity>WARNING</severity><msg>" &
                       To_String (Arguments (3)) & "." &
                       To_String (Arguments (4)) &
                       " isn't in group " & To_String (Arguments (2)) &
                       "</msg></error>";
                  end if;
                  Actor_Vector.Delete (Group.Actors, Actor_Cursor);
                  if Actor_Vector.Is_Empty (Group.Actors) then
                     Actor_Vector.Clear (Group.Actors);
                     Free (Group);
                     Group_Vector.Delete (The_Groups, Group_Cursor);
                  else
                     Group_Vector.Replace_Element (The_Groups,
                                                   Group_Cursor,
                                                   Group);
                  end if;
               end;
               return "<result cmd=""group del""><msg>ok</msg></result>";
            elsif Arguments (1) = "set" then
               Group_Cursor := Find_Group (Arguments (4));
               if Group_Cursor = Group_Vector.No_Element then
                  return "<error><severity>WARNING</severity><msg>" &
                    To_String (Arguments (4)) & " isn't a valid group name" &
                    "</msg></error>";
               end if;
               declare
                  Group : constant Group_Access :=
                    Group_Vector.Element (Group_Cursor);
                  Actor_Cursor : Actor_Vector.Cursor;
               begin
                  Start_Element (Xml_Buffer, "result");
                  Actor_Cursor := Actor_Vector.First (Group.Actors);
                  loop
                     Start_Element (Xml_Buffer, "actor");
                     declare
                        Actor : constant Actor_Full_Name :=
                          Actor_Vector.Element (Actor_Cursor);
                        Command : constant Unbounded_String := "set " &
                          Arguments (2) & " " & Arguments (3) & " " &
                          Actor.Sub_System_Name & " " & Actor.Actor_Name;
                     begin
                        Output_Element
                          (Xml_Buffer,
                           "sub_system",
                           To_String (Actor.Sub_System_Name));
                        Output_Element (Xml_Buffer,
                                        "name",
                                        To_String (Actor.Actor_Name));
                        Put (Xml_Buffer,
                             Exec_Command (To_String (Command)));
                     end;
                     End_Element (Xml_Buffer, "actor");
                     Actor_Cursor := Actor_Vector.Next (Actor_Cursor);
                     exit when Actor_Cursor = Actor_Vector.No_Element;
                  end loop;
               end;
               declare
                  String_To_Return : constant String :=
                    Get_String (Xml_Buffer);
               begin
                  Full_Clear (Xml_Buffer);
                  return String_To_Return;
               end;
            else
               return Syntax_String;
            end if;
         else
            return Syntax_String;
         end if;
      end;
   exception
      when Shell_Commands.No_Argument =>
         return Syntax_String;
      when E : others =>
         return Exception_Message (Cmd => "group", E => E);
   end Group;

   function Syntax_String
     (Status : Shell_Commands.Status_Type := Shell_Commands.Syntax_Error)
     return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Clear (Xml_Buffer);
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "group", "status" = Status'Img));
      Start_Element (Xml_Buffer, "message", "type" = "help");
      Output_Element (Xml_Buffer, "syntax", "group list");
      Output_Element (Xml_Buffer, "syntax", "group list grp_name");
      Output_Element (Xml_Buffer, "syntax",
                      "group add grp_name sub_system_name actor_name");
      Output_Element (Xml_Buffer, "syntax",
                      "group del grp_name sub_system_name actor_name");
      Output_Element (Xml_Buffer, "syntax",
                      "group get parameter_name grp_name");
      Output_Element (Xml_Buffer, "syntax",
                      "group set parameter_name parameter_value grp_name");
      End_Element (Xml_Buffer, "message");
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Syntax_String;

end Group_Command;
