--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Containers.Vectors;
with Ada.Containers.Vectors.Select_Item;
with Ada.Strings.Unbounded;
with Ada.Unchecked_Deallocation;

with DOM.Core;

with Uuid;
with Log4ada;

with Narval.Actors;

package Narval.Configurator.Actors_Description is

   use Ada.Strings.Unbounded;

   package String_Vector is new Ada.Containers.Vectors (Positive,
                                                        Unbounded_String);

   type Receiver_Descriptor_Type is record
      Name : Unbounded_String;
      Data_Source : Unbounded_String;
      Size : Natural := 0;
      Port : Unbounded_String;
   end record;
   type Receiver_Descriptor_Access is access Receiver_Descriptor_Type;
   procedure Free is new Ada.Unchecked_Deallocation
     (Receiver_Descriptor_Type, Receiver_Descriptor_Access);
   function Selection (Item : Receiver_Descriptor_Access;
                       Select_On : Unbounded_String) return Boolean;

   type Sender_Descriptor_Type is record
      Name : Unbounded_String;
      Size : Natural := 0;
      Port : Unbounded_String;
      Consumers_Number : Actor_Numbering_Type;
   end record;
   type Sender_Descriptor_Access is access Sender_Descriptor_Type;
   procedure Free is new Ada.Unchecked_Deallocation (Sender_Descriptor_Type,
                                                     Sender_Descriptor_Access);
   function Selection (Item : Sender_Descriptor_Access;
                       Select_On : Unbounded_String) return Boolean;

   package Receivers_Vector is new Ada.Containers.Vectors
     (Positive, Receiver_Descriptor_Access);
   package Senders_Vector is new Ada.Containers.Vectors
     (Positive, Sender_Descriptor_Access);
   procedure Free (Receivers : in out Receivers_Vector.Vector);
   procedure Free (Senders : in out Senders_Vector.Vector);

   type Actor_Description_Type;
   type Actor_Description_Access is access all Actor_Description_Type;
   package Actors_Description_Vector is new Ada.Containers.Vectors
     (Positive, Actor_Description_Access);
   function "<" (Left, Right : Actor_Description_Access) return Boolean;
   package Actors_Sorting is new
     Actors_Description_Vector.Generic_Sorting ("<");
   procedure Free (Actors : in out Actors_Description_Vector.Vector);
   procedure Fill (Node_Root : DOM.Core.Node;
                   Actors : in out Actors_Description_Vector.Vector);
   procedure Fill (Actors : in out Actors_Description_Vector.Vector;
                   File_Name : String);

   type Actors_Vector_Access is access all Actors_Description_Vector.Vector;

   task type Register_Synchro_Task is
      entry Clear;
      entry Post (Pointer : Actors.Actor_Class_Access);
      entry Get_Pointer (Pointer : out Actors.Actor_Class_Access);
   end Register_Synchro_Task;
   type Register_Synchro_Access is access Register_Synchro_Task;
   procedure Free is new Ada.Unchecked_Deallocation (Register_Synchro_Task,
                                                     Register_Synchro_Access);

   type Actor_Description_Type is record
      Name : Unbounded_String;
      Parent_Vector : Actors_Vector_Access := null;
      Kind : Actor_Kind_Type;
      Uuid_Value : Uuid.Uuid_Type;
      Host_Name : Unbounded_String;
      Process_Name : Unbounded_String;
      Arguments : String_Vector.Vector;
      Rank_Number : Rank_Type := -1;
      Log_Level : Log4ada.Level_Type := Log4ada.Warn;
      Task_Register : Register_Synchro_Access;
      Inputs : Receivers_Vector.Vector;
      Declared_Inputs : Natural := 0;
      Outputs : Senders_Vector.Vector;
      Declared_Outputs : Natural := 0;
      Rank_Calcul_Running : Boolean := False;
      Check_All : Boolean := True;
   end record;
   procedure Free is new Ada.Unchecked_Deallocation (Actor_Description_Type,
                                                     Actor_Description_Access);
   procedure Display (Actor : Actor_Description_Type);
   function To_String (Actor : Actor_Description_Type) return String;
   procedure Validity_Check (Actor : Actor_Description_Type);
   procedure Fill (Node_Root : DOM.Core.Node;
                   Actor : in out Actor_Description_Type);
   procedure Fill (Actor : in out Actor_Description_Type;
                   Informations : String);
   procedure Validity_Check
     (Actor_Position : Actors_Description_Vector.Cursor);
   procedure Rank_Calcul (Actor_Position : Actors_Description_Vector.Cursor);
   procedure Display (Actor_Position : Actors_Description_Vector.Cursor);
   procedure Validity_Check (Actors : Actors_Description_Vector.Vector);
   procedure Rank_Calcul (Actors : Actors_Description_Vector.Vector);
   procedure Display (Actors : Actors_Description_Vector.Vector);
   procedure Validity_Check (Sender_Position : Senders_Vector.Cursor);
   procedure Validity_Check (Receiver_Position : Receivers_Vector.Cursor);
   function Find_Actor (Actors : Actors_Description_Vector.Vector;
                        Actor_Name : Unbounded_String)
                       return Actor_Description_Access;
   function Selection (Item : Actor_Description_Access;
                       Select_On : Unbounded_String) return Boolean;
   function Find_Actor is new Actors_Description_Vector.Select_Item
     (Unbounded_String, Selection);
   Find_Failure : exception;
   function Find_Input is new Receivers_Vector.Select_Item
     (Unbounded_String, Selection);
   function Find_Output is new Senders_Vector.Select_Item
     (Unbounded_String, Selection);
end Narval.Configurator.Actors_Description;
