/*
 * Copyright 2016 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// primary header
#include "h265Parser.h"

// library headers
#include "common/Array.h"
#include "common/unittest.h"
#include "common/nalreader.h"

namespace YamiParser {
namespace H265 {

    using YamiMediaCodec::NalReader;

    const static std::array<uint8_t, 1192> g_SimpleH265 = {
        0x00, 0x00, 0x00, 0x01, 0x40, 0x01, 0x0c, 0x01, 0xff, 0xff, 0x01, 0x60,
        0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03, 0x00,
        0x99, 0x1b, 0x02, 0x40, 0x00, 0x00, 0x00, 0x01, 0x42, 0x01, 0x01, 0x01,
        0x60, 0x00, 0x00, 0x03, 0x00, 0x90, 0x00, 0x00, 0x03, 0x00, 0x00, 0x03,
        0x00, 0x99, 0xa0, 0x0b, 0x08, 0x04, 0x85, 0xb1, 0xbb, 0xb6, 0xd2, 0x20,
        0x10, 0x00, 0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x03, 0x01, 0xe0, 0x80,
        0x00, 0x00, 0x00, 0x01, 0x44, 0x01, 0xc0, 0x61, 0xc3, 0xe0, 0x09, 0x00,
        0x00, 0x00, 0x01, 0x26, 0x01, 0xef, 0xc0, 0xb7, 0x5f, 0xc9, 0x39, 0xa9,
        0xe5, 0xd7, 0x40, 0x98, 0x5e, 0xf6, 0xb1, 0xe4, 0x38, 0x3a, 0x0f, 0x22,
        0xf5, 0xf1, 0xf6, 0xa5, 0x03, 0xd2, 0xa5, 0x4b, 0x53, 0x50, 0x24, 0x74,
        0x68, 0x82, 0x31, 0x54, 0x9f, 0x13, 0xca, 0x68, 0x70, 0x50, 0x3a, 0x49,
        0x9e, 0x4a, 0xb5, 0x0a, 0x9e, 0x36, 0x0d, 0x36, 0x72, 0x28, 0xc7, 0x91,
        0x27, 0x79, 0x3e, 0x08, 0x27, 0x64, 0x0a, 0xde, 0x89, 0x0f, 0x22, 0xb0,
        0x04, 0xe6, 0x59, 0xb3, 0x20, 0x1f, 0x26, 0x8d, 0xf0, 0xbd, 0x43, 0xf2,
        0xd9, 0x4a, 0x5b, 0x84, 0x46, 0x7b, 0x9b, 0xed, 0xee, 0x31, 0x50, 0x0a,
        0x0b, 0x6e, 0xb8, 0x80, 0x8a, 0x9e, 0xd7, 0xd0, 0x64, 0xe6, 0x66, 0x2a,
        0x18, 0x89, 0x85, 0xbf, 0xfd, 0xb9, 0x20, 0x86, 0x8a, 0x27, 0xa6, 0xc1,
        0xed, 0x5b, 0xb8, 0x67, 0xbe, 0x78, 0x0d, 0x0c, 0x0b, 0x6d, 0x89, 0x14,
        0x56, 0xb4, 0x55, 0xf2, 0xb7, 0xc5, 0x3e, 0xc8, 0x00, 0x1d, 0xd3, 0xef,
        0x8a, 0xeb, 0x47, 0xed, 0x1d, 0x0f, 0x34, 0x0e, 0x71, 0x28, 0xf6, 0xe5,
        0x07, 0xd2, 0xdd, 0x8a, 0xe0, 0xd8, 0x94, 0x4c, 0x98, 0xa3, 0xcc, 0x7f,
        0x63, 0x1e, 0xe1, 0xa6, 0x9b, 0xcd, 0x43, 0x8b, 0xd1, 0xaf, 0xe4, 0x2a,
        0x01, 0x91, 0xce, 0xae, 0x94, 0x6a, 0xed, 0x4a, 0xaf, 0x8d, 0x32, 0x05,
        0xa0, 0xa4, 0x5a, 0x00, 0x07, 0x1c, 0x9a, 0x1d, 0xdb, 0xba, 0x41, 0xd9,
        0xbd, 0xf4, 0x30, 0xbc, 0x06, 0x8a, 0xc7, 0xc7, 0x00, 0x00, 0x03, 0x00,
        0x06, 0x7b, 0x16, 0x08, 0x09, 0x28, 0x3b, 0xd6, 0xbc, 0xdb, 0xaa, 0xee,
        0x75, 0xa4, 0x10, 0x36, 0xb4, 0xfd, 0x71, 0x9d, 0xb9, 0xab, 0x11, 0x10,
        0x13, 0xe6, 0x74, 0x07, 0xf5, 0xa7, 0x7f, 0xae, 0x6a, 0xfd, 0xd1, 0x8a,
        0x14, 0x4a, 0x92, 0x57, 0x93, 0x68, 0xba, 0x9e, 0x88, 0xad, 0x96, 0x8a,
        0xf6, 0xa9, 0xc7, 0x87, 0xef, 0xd3, 0x89, 0x52, 0xb5, 0xe6, 0x41, 0x2d,
        0x9b, 0x8d, 0x33, 0x55, 0xf9, 0x32, 0x3d, 0x36, 0xfa, 0xe3, 0x43, 0x8f,
        0x99, 0x62, 0x46, 0x56, 0xdb, 0x04, 0xf9, 0x5b, 0xc3, 0xc6, 0x5d, 0x33,
        0x17, 0x61, 0xde, 0xef, 0xe8, 0xb9, 0x1f, 0x35, 0xf3, 0xbd, 0x69, 0xcb,
        0x4c, 0x56, 0x69, 0xaa, 0x8d, 0xc8, 0xd5, 0x32, 0x87, 0x20, 0xe7, 0xbd,
        0xeb, 0xc3, 0x68, 0xf3, 0x46, 0x70, 0x07, 0xf1, 0xf5, 0xc1, 0x45, 0x9b,
        0xb3, 0xa0, 0x65, 0xe2, 0xd9, 0xfa, 0x87, 0x70, 0x64, 0x87, 0xda, 0x6f,
        0x60, 0x5e, 0x30, 0x7d, 0x74, 0xd8, 0x68, 0xa2, 0x6c, 0xc5, 0x9b, 0x6c,
        0x31, 0x77, 0x96, 0x45, 0x41, 0x13, 0x60, 0x7e, 0xd2, 0x69, 0x96, 0x54,
        0x00, 0xb2, 0x0b, 0xf2, 0x7e, 0x34, 0x00, 0x9f, 0x36, 0x20, 0x0a, 0x53,
        0xcf, 0x3e, 0xaf, 0xb2, 0x1c, 0xa9, 0x4d, 0x90, 0x47, 0xf0, 0x7a, 0xe2,
        0x25, 0x62, 0x6b, 0xcd, 0x85, 0x37, 0x3e, 0x5c, 0x68, 0xac, 0x7f, 0xd9,
        0x95, 0x24, 0xd2, 0x77, 0x2e, 0x31, 0xf9, 0xa8, 0xf7, 0x27, 0xe4, 0x7c,
        0xf5, 0x4c, 0x3e, 0xd8, 0xda, 0x7f, 0xfa, 0x05, 0x7f, 0x1e, 0x3e, 0x6e,
        0x1a, 0x07, 0xe4, 0x22, 0xc4, 0x95, 0xaa, 0xb0, 0x25, 0xb0, 0x97, 0x02,
        0x3c, 0x0d, 0xb3, 0x5e, 0x9c, 0x49, 0x2a, 0xfb, 0x98, 0x5a, 0x3c, 0xef,
        0x50, 0x3c, 0x47, 0x7a, 0xe8, 0xac, 0xc0, 0x1c, 0x80, 0x6a, 0xe5, 0x87,
        0x98, 0x60, 0x7b, 0x0e, 0x71, 0xdc, 0x41, 0x55, 0x4d, 0xe8, 0x80, 0xa6,
        0x01, 0x61, 0xfb, 0x09, 0xd5, 0x44, 0x51, 0x61, 0x15, 0xeb, 0xe4, 0xa0,
        0x3d, 0x3b, 0x44, 0xa3, 0x8a, 0x74, 0xe9, 0xd3, 0xad, 0x9e, 0x82, 0x73,
        0x16, 0x21, 0xd7, 0x7a, 0xbe, 0x66, 0xe5, 0xde, 0x5b, 0xab, 0x44, 0x87,
        0x11, 0xb9, 0x1b, 0x27, 0x0f, 0xf9, 0x10, 0x84, 0x62, 0x31, 0xe5, 0x65,
        0xb0, 0xd7, 0x94, 0x66, 0xeb, 0x69, 0x76, 0x65, 0x5e, 0x87, 0xe3, 0x22,
        0x43, 0x71, 0x33, 0x52, 0x97, 0x76, 0x71, 0x0a, 0xe9, 0x5f, 0xbf, 0xf8,
        0xf6, 0x41, 0x17, 0xc2, 0x34, 0x8a, 0xa8, 0xda, 0xb0, 0x8d, 0x9e, 0xc2,
        0xbc, 0x54, 0xf5, 0xa2, 0x36, 0xef, 0x67, 0x15, 0xf2, 0xe2, 0xf9, 0xbe,
        0xfd, 0x9c, 0x86, 0x15, 0xed, 0xef, 0x6b, 0xf5, 0xc3, 0x5d, 0xe4, 0xa1,
        0x4d, 0xce, 0xcb, 0x1c, 0x10, 0xd3, 0xb3, 0x1f, 0x50, 0x85, 0x7d, 0x79,
        0x10, 0x9f, 0x10, 0xe5, 0xb4, 0x64, 0xce, 0x22, 0xaf, 0x6a, 0x2d, 0xb6,
        0x70, 0x46, 0x08, 0xf6, 0x6a, 0x35, 0xbc, 0x42, 0xf2, 0xf9, 0x8d, 0x92,
        0x06, 0x7c, 0xcf, 0x8c, 0xda, 0x1f, 0xa1, 0x63, 0x8f, 0x92, 0x70, 0x19,
        0x36, 0x69, 0x20, 0xba, 0x22, 0x4f, 0xf5, 0xd2, 0x85, 0xc7, 0x5f, 0x7c,
        0xda, 0x53, 0x09, 0x0c, 0xfa, 0x37, 0x70, 0x42, 0x79, 0x77, 0x83, 0x99,
        0xf8, 0x0e, 0xeb, 0x68, 0x25, 0x8d, 0xae, 0x96, 0x79, 0x70, 0xfb, 0x85,
        0xdb, 0xd0, 0x2f, 0x62, 0x9a, 0x61, 0xa8, 0xc5, 0x06, 0x29, 0x65, 0x82,
        0x2d, 0x2a, 0x07, 0x87, 0x72, 0x72, 0x11, 0xf1, 0x7a, 0x60, 0x95, 0x93,
        0x76, 0xba, 0x63, 0xd1, 0x19, 0xbe, 0x72, 0x54, 0xd1, 0x48, 0xb6, 0x33,
        0xb4, 0x31, 0x40, 0x79, 0xad, 0x93, 0x02, 0xc1, 0x11, 0xc0, 0x3a, 0x77,
        0x28, 0xe0, 0xdc, 0x38, 0xb4, 0xb8, 0xcc, 0xa7, 0x19, 0x3a, 0x16, 0x34,
        0x26, 0x5a, 0x4e, 0xa0, 0xfb, 0xc1, 0xa1, 0x37, 0x00, 0x5d, 0xf5, 0x39,
        0x8b, 0xc7, 0xc7, 0x33, 0xaf, 0xd3, 0xfa, 0x42, 0xe4, 0x73, 0xce, 0xf7,
        0x75, 0xdf, 0x9f, 0xdd, 0x7d, 0x4b, 0x6b, 0x46, 0xc2, 0x21, 0x96, 0x63,
        0x98, 0xec, 0x11, 0x10, 0x72, 0xa3, 0x77, 0xd9, 0x5b, 0xd9, 0x05, 0xfa,
        0x56, 0xdd, 0xa4, 0x47, 0xfd, 0xec, 0x72, 0x17, 0xfd, 0x3e, 0xb6, 0x3c,
        0x8e, 0xb3, 0x50, 0x02, 0x38, 0x43, 0x69, 0xff, 0x2d, 0x71, 0x93, 0xcb,
        0x15, 0xd4, 0x27, 0x2a, 0xae, 0x50, 0x3c, 0x91, 0xd6, 0xfc, 0xee, 0x6e,
        0x1c, 0x0f, 0xbe, 0xa8, 0x09, 0xe5, 0x84, 0x8e, 0x5d, 0x3a, 0x3e, 0xbb,
        0x8a, 0x6d, 0x0c, 0xd9, 0xa7, 0xc1, 0xed, 0xdd, 0x89, 0x7e, 0x59, 0xf4,
        0xed, 0x6f, 0x7b, 0xdc, 0x77, 0x77, 0xe5, 0x6c, 0x57, 0xff, 0xea, 0xb4,
        0x86, 0xf5, 0xc7, 0xe8, 0xf0, 0xca, 0xff, 0x31, 0x4b, 0x56, 0xed, 0xde,
        0x4d, 0x7d, 0xb8, 0xa3, 0x23, 0x67, 0xac, 0xca, 0xdd, 0x33, 0xb2, 0x1b,
        0xf6, 0xea, 0xfe, 0x43, 0x9b, 0x52, 0x22, 0xf4, 0x38, 0x5f, 0xd9, 0xee,
        0x75, 0x06, 0xc7, 0x63, 0x67, 0xba, 0x07, 0xce, 0x48, 0x9c, 0xc7, 0x63,
        0x6b, 0x4e, 0x61, 0x1b, 0xcd, 0x11, 0x62, 0x5f, 0x20, 0x2d, 0x24, 0x8f,
        0xfc, 0x16, 0xe3, 0xdf, 0x35, 0x2a, 0xd9, 0xec, 0xb4, 0xb7, 0x7c, 0xbb,
        0x56, 0xff, 0xa8, 0xf7, 0xda, 0x90, 0x86, 0xbf, 0xcc, 0xc5, 0x23, 0x41,
        0x44, 0xb6, 0xd5, 0xb2, 0x28, 0xa3, 0x8c, 0x96, 0x4a, 0x57, 0x5a, 0x80,
        0xd3, 0xd8, 0x68, 0xe3, 0x71, 0xf8, 0xfc, 0xd0, 0xe9, 0xbe, 0x3c, 0x7b,
        0xf8, 0x2c, 0x9e, 0xbd, 0x5b, 0xa6, 0xc4, 0xb5, 0x1f, 0x26, 0x17, 0x6e,
        0x2b, 0xd8, 0xbb, 0xe6, 0xe9, 0x53, 0xa0, 0x0d, 0x7a, 0x12, 0x34, 0x1e,
        0x0e, 0xef, 0x96, 0x51, 0x21, 0xed, 0x23, 0x1e, 0x56, 0xb7, 0x4e, 0x02,
        0x5f, 0x74, 0x59, 0xcc, 0xfe, 0x72, 0x5b, 0x3d, 0xe6, 0x86, 0x6d, 0xfe,
        0xce, 0x63, 0x34, 0x1a, 0x5d, 0xf5, 0x70, 0x3b, 0xe4, 0x33, 0x43, 0x06,
        0x46, 0xff, 0x35, 0x73, 0xa1, 0xe5, 0x2b, 0x38, 0x86, 0x4d, 0xc3, 0x16,
        0x07, 0x5f, 0x69, 0xdd, 0xdb, 0x51, 0x3f, 0x1d, 0x77, 0x9f, 0x58, 0x42,
        0x0f, 0xe6, 0x81, 0x52, 0xb6, 0x55, 0x0c, 0x57, 0x8e, 0x06, 0x5b, 0x11,
        0xf5, 0xa0, 0x34, 0x8b, 0x9d, 0x1c, 0xa7, 0xe6, 0x6d, 0x60, 0x7d, 0x14,
        0x40, 0xf7, 0x00, 0x8f, 0xae, 0xd2, 0xf1, 0x15, 0xdf, 0x19, 0x28, 0x3c,
        0x52, 0x5c, 0xf6, 0x2d, 0xd1, 0x59, 0x2e, 0x0d, 0xbf, 0x3f, 0xaf, 0x23,
        0x61, 0x1c, 0xda, 0xd7, 0xed, 0x95, 0xc3, 0x31, 0x91, 0x9f, 0xa2, 0x8d,
        0x0e, 0xe5, 0xd7, 0x38
    };

    class H265ParserTest : public ::testing::Test {

    protected:
        void checkH265Vps(Parser& parser, const NalUnit* nalu)
        {
            EXPECT_EQ(NalUnit::VPS_NUT, nalu->nal_unit_type);
            EXPECT_EQ(true, parser.parseVps(nalu));
            SharedPtr<VPS> vps = parser.getVps(0);
            EXPECT_EQ(true, !!vps);
            EXPECT_EQ(153, vps->profile_tier_level.general_level_idc);
            EXPECT_EQ(5, vps->vps_max_dec_pic_buffering_minus1[0]);
        }

        void checkH265Sps(Parser& parser, const NalUnit* nalu)
        {
            EXPECT_EQ(NalUnit::SPS_NUT, nalu->nal_unit_type);
            EXPECT_EQ(true, parser.parseSps(nalu));
            SharedPtr<SPS> sps = parser.getSps(0);
            EXPECT_EQ(true, !!sps);
            EXPECT_EQ(352, sps->pic_width_in_luma_samples);
            EXPECT_EQ(288, sps->pic_height_in_luma_samples);
            EXPECT_EQ(true, sps->vui_parameters_present_flag);
            EXPECT_EQ(30u, sps->vui_params.vui_time_scale);
        }

        void checkH265Pps(Parser& parser, const NalUnit* nalu)
        {
            EXPECT_EQ(NalUnit::PPS_NUT, nalu->nal_unit_type);
            EXPECT_EQ(true, parser.parsePps(nalu));
            SharedPtr<PPS> pps = parser.getPps(0);
            EXPECT_EQ(true, !!pps);
            EXPECT_EQ(14, pps->init_qp_minus26);
            EXPECT_EQ(true, pps->slice_chroma_qp_offsets_present_flag);
        }

        void checkH265SliceHeader(Parser& parser, NalUnit* nalu)
        {
            EXPECT_EQ(NalUnit::IDR_W_RADL, nalu->nal_unit_type);
            SliceHeader slice;
            EXPECT_EQ(true, parser.parseSlice(nalu, &slice));
            EXPECT_EQ(2, slice.slice_type);
            EXPECT_EQ(0, slice.cr_qp_offset);
        }
    };

#define H265_PARSER_TEST(name) TEST_F(H265ParserTest, name)

    H265_PARSER_TEST(Parse_Simple)
    {
        const uint8_t* nal;
        int32_t size;
        NalUnit nalu;
        NalReader nr(&g_SimpleH265[0], g_SimpleH265.size());
        Parser parser;

        ASSERT_TRUE(nr.read(nal, size));
        ASSERT_TRUE(nalu.parseNaluHeader(nal, size));
        checkH265Vps(parser, &nalu);

        ASSERT_TRUE(nr.read(nal, size));
        ASSERT_TRUE(nalu.parseNaluHeader(nal, size));
        checkH265Sps(parser, &nalu);

        ASSERT_TRUE(nr.read(nal, size));
        ASSERT_TRUE(nalu.parseNaluHeader(nal, size));
        checkH265Pps(parser, &nalu);

        ASSERT_TRUE(nr.read(nal, size));
        ASSERT_TRUE(nalu.parseNaluHeader(nal, size));
        checkH265SliceHeader(parser, &nalu);

        ASSERT_FALSE(HasFailure());
    }

} // namespace H265
} // namespace YamiParser
