/***************************************************************************
*   Copyright (C) 2004-2006 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <qregexp.h>
#include <qstringlist.h>

#include <file.h>
#include <element.h>
#include <entry.h>
#include <encoderxml.h>
#include <macro.h>
#include <comment.h>
#include "fileexporterxml.h"

namespace BibTeX
{

    FileExporterXML::FileExporterXML()
            : FileExporter()
    {
        cancelFlag = FALSE;
    }


    FileExporterXML::~FileExporterXML()
    {
        // nothing
    }

    bool FileExporterXML::save( QIODevice* iodevice, File* bibtexfile, QStringList * /*errorLog*/ )
    {
        bool result = TRUE;
        QTextStream stream( iodevice );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        stream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
        stream << "<bibliography>" << endl;

        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Element *element = *it;
            write( stream, element );
        }

        stream << "</bibliography>" << endl;

        return result && !cancelFlag;
    }

    bool FileExporterXML::save( QIODevice* iodevice, Element* element, QStringList * /*errorLog*/ )
    {
        QTextStream stream( iodevice );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        stream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
        return write( stream, element );
    }

    void FileExporterXML::cancel()
    {
        cancelFlag = TRUE;
    }

    bool FileExporterXML::write( QTextStream&stream, Element *element )
    {
        bool result = FALSE;

        Entry *entry = dynamic_cast<Entry*>( element );
        if ( entry != NULL )
            result |= writeEntry( stream, entry );
        else
        {
            Macro * macro = dynamic_cast<Macro*>( element );
            if ( macro != NULL )
                result |= writeMacro( stream, macro );
            else
            {
                Comment * comment = dynamic_cast<Comment*>( element );
                if ( comment != NULL )
                    result |= writeComment( stream, comment );
            }
        }

        return result;
    }

    bool FileExporterXML::writeEntry( QTextStream &stream, Entry* entry )
    {
        stream << " <entry id=\"" << entry->id() << "\" type=\"" << entry->entryTypeString().lower() << "\">" << endl;
        for ( Entry::EntryFields::iterator it = entry->begin(); it != entry->end(); it++ )
        {
            EntryField *field = *it;
            switch ( field->fieldType() )
            {
            case EntryField::ftAuthor:
            case EntryField::ftEditor:
                {
                    QString tag = field->fieldTypeName().lower();
                    stream << "  <" << tag << "s>" << endl;
                    QStringList persons = QStringList::split( QRegExp( "\\s+(,|and|&)+\\s+", FALSE ), EncoderXML::currentEncoderXML() ->encode( valueToString( field->value() ) ) );
                    for ( QStringList::Iterator it = persons.begin(); it != persons.end(); it++ )
                        stream << "   <person>" << *it << "</person>" << endl;
                    stream << "  </" << tag << "s>" << endl;
                }
                break;
            case EntryField::ftMonth:
                {
                    stream << "  <month";
                    Value *value = field->value();
                    bool ok = false;
                    if ( value->count() == 1 )
                    {
                        ValueItem * item = value->first();
                        if ( item->isStringKey() )
                            for ( int i = 0; i < 12 && !ok; i++ )
                                if ( QString::compare( item->text(), MonthsTriple[ i ] ) == 0 )
                                {
                                    stream << " tag=\"" << MonthsTriple[ i ] << "\">" << Months[ i ];
                                    ok = TRUE;
                                }
                    }
                    if ( !ok )
                        stream << '>' << EncoderXML::currentEncoderXML() ->encode( valueToString( field->value() ) ) ;
                    stream << "</month>" << endl;
                }
                break;
            default:
                {
                    QString tag = field->fieldTypeName().lower();
                    stream << "  <" << tag << ">" << EncoderXML::currentEncoderXML() ->encode( valueToString( field->value() ) ) << "</" << tag << ">" << endl;
                }
                break;
            }

        }
        stream << " </entry>" << endl;

        return TRUE;
    }

    bool FileExporterXML::writeMacro( QTextStream &stream, Macro *macro )
    {
        stream << " <string key=\"" << macro->key() << "\">";
        stream << EncoderXML::currentEncoderXML() ->encode( valueToString( macro->value() ) );
        stream << "</string>" << endl;

        return TRUE;
    }

    bool FileExporterXML::writeComment( QTextStream &stream, Comment *comment )
    {
        stream << " <comment>" ;
        stream << EncoderXML::currentEncoderXML() ->encode( comment->text() );
        stream << "</comment>" << endl;

        return TRUE;
    }

    QString FileExporterXML::valueToString( Value *value )
    {
        QString result;
        bool isFirst = TRUE;

        for ( QValueList<ValueItem*>::const_iterator it = value->begin(); it != value->end(); it++ )
        {
            if ( !isFirst )
                result.append( ' ' );
            isFirst = FALSE;

            result.append( ( *it ) ->text() );
        }

        return result;
    }


}
