/*
 * Grouch.app				Copyright (C) 2006 Andy Sveikauskas
 * ------------------------------------------------------------------------
 * This program is free software under the GNU General Public License
 * --
 * This class provides a way to queue NSInvocations for thread safety.  It
 * creates an NSTimer (which runs in a run loop).  Then other threads will
 * add invocations to its queue, and they will be invoked in the timer call.
 *
 * In addition, GrouchRunLoopHack provides NSRunLoop-style methods to add
 * and remove things from the run loop.  So you can pretend GrouchRunLoopHack
 * is an NSRunLoop if it suits you for this purpose.
 */

#import <Foundation/NSObject.h>
#import <Foundation/NSRunLoop.h>

@class NSRunLoop, NSLock, NSTimer, NSPort, NSString;
@class NSInvocation;

#ifdef GNUSTEP
@class NSEvent;
#endif

@interface GrouchRunLoopHack : NSObject
{
	NSRunLoop *loop;
	struct grouchrunloop_node
	{
		NSInvocation *invocation;
		NSObject *array;
		struct grouchrunloop_node *next;
	} *head, *tail;
	NSLock *lock;
	NSTimer *timer;
	BOOL markedForDestruction;
}

// The init methods MUST be called during the main thread!
- init;				// Initiate for current run loop
- initForRunLoop:(NSRunLoop*)l;

// Add an invocation to the queue.  This can be called from any thread.
// The second arugment is just an object that will be retained and released.
// (If your method has non-object parameters and you don't want to call
// NSInvocation-retainArguments, just put your objects in an array and provide
// it as this argument)
- (void)addInvocation:(NSInvocation*)invok withArguments:(NSObject*)array;

// Same as above, but without the second argument nonsense.
- (void)addInvocation:(NSInvocation*)invok;

// Call from another thread to get your timer object eventually invalidated.
// i.e. you will not add any more invocations from this thread.
- (void)invalidate;

@end

// Code to make GrouchRunLoop seem sort of like an NSRunLoop

@interface GrouchRunLoopHack (RunLoopCompatibility)

- (void)addTimer:(NSTimer*)t forMode:(NSString*)mode;
- (void)addPort:(NSPort*)port forMode:(NSString*)mode;
- (void)removePort:(NSPort*)port forMode:(NSString*)mode;

#ifdef GNUSTEP
- (void)addEvent:(void*)data type:(RunLoopEventType)type
        watcher:(id<RunLoopEvents>)watcher forMode:(NSString*)mode;
- (void)removeEvent:(void*)data type:(RunLoopEventType)type
	forMode:(NSString*)mode all:(BOOL)all;
#endif

@end

