package biss.awt;

import biss.awt.Control;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.*;

/**
 * abstract base for scrollable Canvas widgets inside of Compounds
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
abstract public class ScrollableCanvas
  extends Control
{
	HScrollBar Horz;
	int IdxTop = 0;
	int LineHeight = 0;
	int LineWidth = 0;
	Vector Lines;
	VScrollBar Vert;
	int XBorder = 0;
	int XOffset = 0;
	Graphics ResGraph;
	int YOffset;
	int YBorder;
	static Image SoftImage;
	static Graphics SoftGraph;
	static Dimension SoftSize;

public ScrollableCanvas( HScrollBar h, VScrollBar v) {
	super();
	Horz = h;
	Vert = v;

	setForeground( Color.black);
	setBackground( Color.white);

	registerObserver();
}

boolean atEnd() {
	return ( Vert.PageInc >= Lines.size() - IdxTop);
}

void blankTrailing() {
	int y = -1;

	if ( Lines == null)
		y = 0;
	else if (Lines.size() <= IdxTop + Vert.PageInc)
		y = (Lines.size() - IdxTop) * LineHeight + DrawRect.x;

	if ( y > -1) {	
		ResGraph.setColor( getBackground());
		ResGraph.fillRect( 0, y, Width, Height);
		ResGraph.setColor( getForeground());
	}				
}

void checkSoftImage() {
	int w, h;

	if ( SoftImage == null ) {
		w = Math.max( (2*Awt.ScreenWidth / 3), Width);
		h = Math.max( (2*Awt.SysFontHeight), LineHeight);
		SoftSize = new Dimension( w, h);
		SoftImage    = createImage( w, h);
		SoftGraph = SoftImage.getGraphics();
	}
	else {
		if ( SoftSize.width < Width || SoftSize.height < LineHeight ) {
			SoftSize.width  = Math.max( SoftSize.width, Width);
			SoftSize.height = Math.max( SoftSize.height, LineHeight);
			SoftImage    = createImage( SoftSize.width, SoftSize.height);
			SoftGraph = SoftImage.getGraphics();
		}
	}
}

void lineDown() {
	scrollVertical( 1, true); 			
}

void lineUp() {
	if ( IdxTop > 0)
		scrollVertical( -1, true);
}

Object objectAt( int x, int y) {

	if ( (x<0) || (x>Width) )
		return null;
	if ( (y<0) || (y>Height) )
		return null;
	if ( Lines == null)
		return null;

	int offs = LineHeight + FBorder.Ext;

	for ( int idx = IdxTop; idx < Lines.size(); idx++) {
		if ( y <= offs)
			return Lines.elementAt( idx);
		offs += LineHeight;
	}

	return null;
}

public void paint( Graphics g){
	//first valid resident graphics can be obtained here
	if ( ResGraph == null) {
		ResGraph = getGraphics();
		
		if ( ResGraph != null ) {
			clip( ResGraph);
			if ( Awt.SoftScroll )
				checkSoftImage();
		}
	}

	super.paint( g);
}

protected void posChanged() {
	//dispose resized (cliped) Graphics and obtain new one
	if ( ResGraph != null){
		ResGraph.dispose();
		ResGraph = getGraphics();
		if ( ResGraph != null )
			clip( ResGraph);
	}
	if ( Awt.SoftScroll && (LineHeight > 0) )
		checkSoftImage();
}

boolean positionHorizontal( int pos, boolean updateScroll) {
	return scrollHorizontal( pos + XOffset - XBorder - FBorder.Ext, updateScroll);
}

boolean positionVertical( int pos, boolean updateScroll){
	return scrollVertical( pos - IdxTop, updateScroll);
}

public void redraw( Graphics g){
	if ( ResGraph == null)
		return;
	if (( LineHeight == 0) || (Lines == null) || (Lines.size() == 0))
		blank( ResGraph );
	else
		redrawLines( IdxTop, -1);
}

abstract void redrawLine( int idx);

synchronized void redrawLines( int start, int end) {

	int idx, max;

	if ( ResGraph == null )
		return;

	if ( Lines == null)
		max = -1;
	else	
		max = Math.min(IdxTop + Vert.PageInc, Lines.size() - 1);

	if ( end == -1)	end = max;
	else				end = Math.min( end, max);

	start = Math.max( start, IdxTop);

	for ( idx = start; idx <= end; idx++)
		redrawLine( idx);

	blankTrailing();
}

public void registerObserver(){
	Vert.OsScroll.addObserver( this);
	if ( Horz != null)
		Horz.OsScroll.addObserver( this);

	super.registerObserver();
}

synchronized boolean scrollHorizontal( int steps, boolean updateScroll){
	int dx = XBorder + FBorder.Ext;

	if ( (steps == 0) || (Lines == null) )
		return false;

	if ( updateScroll && ( Horz != null) ){
		Horz.setPosition( steps - XOffset + dx);
		steps = Horz.getPosition() + XOffset - dx;
		if (steps == 0)
			return false;
	}

	XOffset = Math.min( XOffset - steps, dx);
	redrawLines( IdxTop, -1);

	return true;
}

synchronized boolean scrollVertical( int steps, boolean updateScroll) {

	if ( (steps == 0) || (Lines == null))
		return false;

	int s = Lines.size();

	if ( updateScroll){      //watch limits on keyboard scroll
		Vert.setPosition( IdxTop + steps);
		steps = Vert.getPosition() - IdxTop;
		if ( (steps == 0) || (IdxTop + steps >= s))
			return false;
	}

	if ( ResGraph != null) {
		int as = Math.abs( steps);
		if ( Vert.IsSoftScroll )
			softScroll( steps);
		else if ( as <= Vert.PageInc )
			ResGraph.copyArea( DrawRect.x, DrawRect.y, DrawRect.width, DrawRect.height,
		                   0, -steps * LineHeight);
	}

	IdxTop += steps;
	vPosChange();

	if ( steps < 0)
		redrawLines( IdxTop, IdxTop - steps - 1);
	else
		redrawLines( IdxTop + Vert.PageInc - steps - 1, -1);

	return true;
}

public void setBorder( int style ) {
	super.setBorder( style);

	XOffset = XBorder + FBorder.Ext;
	YOffset = YBorder + FBorder.Ext;
}

void setLineHeight( int h) {
	LineHeight = h;

	updateVScroll();

	if ( (Lines != null) && (ResGraph != null ))
		redrawLines( IdxTop, -1);
}

void showArea( int sLine, int eLine) {
	if ( ResGraph == null) //already showing?
		return;
	if ( sLine < IdxTop)
		positionVertical( sLine, true);
	else if ( eLine >= IdxTop + Vert.PageInc)
		positionVertical( eLine - Vert.PageInc + 1, true);
}

void softScroll( int steps) {
	int imgY = 0;

	if ( steps == 1) {		//contents up
		updateScrollImage( IdxTop + Vert.PageInc);
		imgY = DrawRect.y + Vert.PageInc * LineHeight;
	}
	else if ( steps == -1) {	//contents down
		updateScrollImage( IdxTop - 2);
		imgY = DrawRect.y -2 * LineHeight ;
	}
	else
		return;

	for ( int i=1; i<=LineHeight; i++) {
		ResGraph.copyArea( DrawRect.x, DrawRect.y, DrawRect.width, DrawRect.height,
		                   0, -steps );
		imgY -= steps;
		ResGraph.drawImage( SoftImage, 0, imgY, this);

		// do an active defer (without explicitly rescheduling the current thread)
		// to avoid jerky scrolling
		int n = (Vert.SoftScrollPixDelay >= 0) ? Vert.SoftScrollPixDelay : Vert.SoftScrollInitDelay;
		for ( int k=0; k < n; k++ );
	}

}

public void unregisterObserver(){
	Vert.OsScroll.deleteObserver( this);
	if ( Horz != null)
		Horz.OsScroll.deleteObserver( this);

	super.unregisterObserver();
}

public void update( Observable o, Object arg){
	if ( o == Vert.OsScroll )
		scrollVertical( Vert.Pos - IdxTop, false);
	else if ( (Horz != null) && (o == Horz.OsScroll) )  
		scrollHorizontal( Horz.Pos + XOffset - XBorder - FBorder.Ext, false);
	else 
		super.update( o, arg);
}

void updateScrollImage( int sIdx) {
}

void updateVScroll() {
	if ( Lines != null){
		Vert.setStepPels( LineHeight);
		Vert.setValues( IdxTop, 0, Lines.size() );
	}
}

void vPosChange(){}
}
