/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: main.c 1.6.1.2 Mon, 13 Oct 1997 19:19:20 -0700 jmacd $
 */

#include "xdelta.h"
#include "getopt.h"

/* $Format: "static const char xdelta_version[] = \"$ReleaseVersion$\"; " $ */
static const char xdelta_version[] = "0.14"; 

typedef struct _Command Command;

struct _Command {
  gchar* name;
  gint (* func) (gint argc, gchar** argv);
  gint nargs;
  gint nvers;
};

static gint  checkout_command (gint argc, gchar** argv);
static gint  checkin_command (gint argc, gchar** argv);
static gint register_command (gint argc, gchar** argv);
static gint     info_command (gint argc, gchar** argv);
static gint    delta_command (gint argc, gchar** argv);
static gint    patch_command (gint argc, gchar** argv);

static const Command commands[] =
{
  { "register", register_command,  1, 0 },
  { "checkin",  checkin_command,   2, 0 },
  { "checkout", checkout_command,  2, 1 },
  { "info",     info_command,      1, 1 },
  { "delta",    delta_command,     3, 0 },
  { "patch",    patch_command,     3, 0 },
  { NULL, NULL, 0, 0 }
};

static struct option const long_options[] =
{
  {"help",                no_argument, 0, 'h'},
  {"version",             no_argument, 0, 'v'},
  {"revision",            no_argument, 0, 'r'},
  {"text",                no_argument, 0, 't'},
  {"separator",           required_argument, 0, 's'},
  {"alignment",           required_argument, 0, 'a'},
  {0,0,0,0}
};

static const gchar* program_name;
static gint         nvers = 0;
static gboolean     use_text = FALSE;
static gboolean     use_separator = FALSE;
static gboolean     use_aligned = FALSE;
static gint         ver1;
static gint         ver2;
static gchar        separator;
static gint         alignment;

static void
usage ()
{
  g_print ("%s: usage: %s COMMAND [OPTIONS] [ARG1 ...]\n", program_name, program_name);
  g_print ("%s: use --help for more help\n", program_name);
  exit (2);
}

static void
help ()
{
  g_print ("%s: usage: %s COMMAND [OPTIONS] [ARG1 ARG2 ...]\n", program_name, program_name);
  g_print ("%s: COMMAND is one of:\n", program_name);
  g_print ("%s:   register  Create a versionfile\n", program_name);
  g_print ("%s:   checkin   Checkin a new version\n", program_name);
  g_print ("%s:   checkout  Checkout a version (latest or -r VERSION)\n", program_name);
  g_print ("%s:   info      Show information on a version (all or -r VERSION)\n", program_name);
  g_print ("%s:   delta     Produce a delta from ARG1 to ARG2 producing ARG3\n", program_name);
  g_print ("%s:   patch     Patch file ARG1 with ARG2 producing ARG3\n", program_name);
  g_print ("%s: OPTIONS are:\n", program_name);
  g_print ("%s:   -t, --text          \n", program_name);
  g_print ("%s:   -a N, --alignment=N \n", program_name);
  g_print ("%s:   -s C, --separator=C \n", program_name);
  g_print ("%s:   -r N, --revision=N  \n", program_name);
  g_print ("%s:   -v, --version       \n", program_name);
  g_print ("%s:   -h, --help          \n", program_name);
  exit (2);
}

static void
version ()
{
  g_print ("%s: version %s\n", program_name, xdelta_version);
  exit (2);
}

gint
main(gint argc, gchar** argv)
{
  const Command *cmd = NULL;
  gint c;
  gint longind;

  program_name = strip_leading_path (argv[0]);

  if (argc < 2)
    usage ();

  for (cmd = commands; cmd->name; cmd += 1)
    if (strcmp (cmd->name, argv[1]) == 0)
      break;

  if (strcmp (argv[1], "-h") == 0 ||
      strcmp (argv[1], "--help") == 0)
    help ();

  if (strcmp (argv[1], "-v") == 0 ||
      strcmp (argv[1], "--version") == 0)
    version ();

  if (!cmd->name)
    {
      g_print ("%s: unrecognized command\n", program_name);
      help ();
    }

  argc -= 1;
  argv += 1;

  while ((c = getopt_long(argc,
			  argv,
			  "+mthvr:s:a:",
			  long_options,
			  &longind)) != EOF)
    {
      switch (c)
	{
	case 'r':
	  {
	    char* end = NULL;
	    gint ver = strtol (optarg, &end, 10);

	    if (end[0] || ver < 0)
	      {
		g_print ("%s: illegal -r argument\n", program_name);
		exit (2);
	      }

	    if (nvers == 0)
	      ver1 = ver;
	    else if (nvers == 1)
	      ver2 = ver;

	    nvers += 1;
	  }
	break;
	case 't': use_text = TRUE; break;
	case 'h': help (); break;
	case 'v': version (); break;
	case 's':
	  {
	    use_separator = TRUE;
	    if (optarg[0] == '\\' && optarg[1] && !optarg[2])
	      {
		switch (optarg[1])
		  {
		  case 'n':  separator = '\n'; break;
		  case 't':  separator = '\t'; break;
		  case '\\': separator = '\\'; break;
		  default:
		    g_print ("%s: illegal -s argument\n", program_name);
		    exit (2);
		  }
	      }
	    else if (!optarg[1])
	      separator = optarg[0];
	    else
	      {
		g_print ("%s: illegal -s argument\n", program_name);
		exit (2);
	      }
	  }
	break;
	case 'a':
	  {
	    char* end = NULL;

	    alignment = strtol (optarg, &end, 10);

	    use_aligned = TRUE;

	    if (end[0])
	      {
		g_print ("%s: illegal -a argument\n", program_name);
		exit (2);
	      }
	  }
	break;
	case '?':
	default:
	  g_print ("%s: illegal argument, use --help for help\n", program_name);
	  exit (2);
	  break;
	}
    }

  argc -= optind;
  argv += optind;

  if (cmd->nargs >= 0 && argc != cmd->nargs)
    {
      g_print ("%s: wrong number of arguments\n", program_name);
      exit (2);
    }

  if (nvers > cmd->nvers)
    {
      g_print ("%s: too many -r arguments\n", program_name);
      exit (2);
    }

  return (* cmd->func) (argc, argv);
}

/* Commands */

static gint
register_command (gint argc, gchar** argv)
{
  XdFile *xd;

  if (fs_file_exists (argv[0]))
    {
      g_print ("%s: file already exists\n", program_name);
      return 2;
    }

  xd = xd_create (argv[0]);

  if (!xd)
    {
      g_print ("%s: create failed: %s\n", program_name, gdbm_strerror (gdbm_errno));
      return 2;
    }

  xd_close (xd);

  return 0;
}

static gint
info_command (gint argc, gchar** argv)
{
  XdFile* xd;
  gint i;

  xd = xd_open_read (argv[0]);

  if (!xd)
    return 2;

  for (i = 0; i < xd->versions; i += 1)
    {
      gchar* date, *md5;
      gint len;

      if (nvers > 0 && ver1 != i)
	continue;

      if (!(date = xd_get_date_str (xd, i)))
	goto bail;

      if (!(md5 = xd_get_md5_str (xd, i)))
	goto bail;

      if ((len = xd_get_len (xd, i)) < 0)
	goto bail;

      g_print ("%d %s %s %d\n", i, date, md5, len);
    }

  xd_close (xd);

  return 0;

bail:
  g_print ("%s: xdelta file corrupted\n", program_name);
  xd_close (xd);
  return 2;
}

static gint
checkin_command (gint argc, gchar** argv)
{
  XdFile* xd;
  gint ret;

  xd = xd_open_write (argv[0]);

  if (!xd)
    {
      g_print ("%s: open %s failed: %s\n", program_name, argv[0], gdbm_strerror (gdbm_errno));
      return 2;
    }

  ret = xd_checkin (xd, argv[1]);

  xd_close (xd);

  return ret;
}

static gint
checkout_command (gint argc, gchar** argv)
{
  XdFile* xd;
  gint ret;

  xd = xd_open_read (argv[0]);

  if (!xd)
    {
      g_print ("%s: open %s failed: %s\n", program_name, argv[0], gdbm_strerror (gdbm_errno));
      return 2;
    }

  ret = xd_checkout (xd, argv[1], nvers > 0 ? ver1 : -1);

  xd_close (xd);

  return ret;
}

static gint
delta_command (gint argc, gchar** argv)
{
  FromSegment *from;
  MappedFile *from_map, *to_map;
  FILE* patch_out;
  MatchQuery *query;
  datum delta;
  gchar md5[16];

  if (!(from_map = map_file (argv[0])))
    {
      g_print ("%s: open/read %s failed: %s\n", program_name, argv[0], strerror (errno));
      return 2;
    }

  if (!(to_map = map_file (argv[1])))
    {
      g_print ("%s: open/read %s failed: %s\n", program_name, argv[1], strerror (errno));
      return 2;
    }

  if (!(patch_out = fopen (argv[2], FOPEN_WRITE_ARG)))
    {
      g_print ("%s: open/write %s failed: %s\n", program_name, argv[2], strerror (errno));
      return 2;
    }

  from = from_segment_new (from_map->seg, from_map->len);

  query = match_query_new (NULL, to_map->seg, to_map->len, 4, &from, 1);

  xdelta (query);

  delta = xdelta_to_bytes (query);

  if (fwrite (XDELTA_PREFIX, XDELTA_PREFIX_LEN, 1, patch_out) != 1)
    goto bail;

  md5_buffer (from_map->seg, from_map->len, md5);

  if (fwrite (md5, 16, 1, patch_out) != 1)
    goto bail;

  md5_buffer (to_map->seg, to_map->len, md5);

  if (fwrite (md5, 16, 1, patch_out) != 1)
    goto bail;

  if (fwrite (delta.dptr, delta.dsize, 1, patch_out) != 1)
    goto bail;

  if (fclose (patch_out))
    goto bail;

  return 0;

bail:
  g_print ("%s: write failed: %s\n", program_name, strerror (errno));

  return 2;
}

static gint
patch_command (gint argc, gchar** argv)
{
  MappedFile *from_map, *patch_map;
  FILE* reconst_out;
  datum delta, deltaoff, from, reconst;
  gchar md5[16];

  if (!(from_map = map_file (argv[0])))
    {
      g_print ("%s: open/read %s failed: %s\n", program_name, argv[0], strerror (errno));
      return 2;
    }

  if (!(patch_map = map_file (argv[1])))
    {
      g_print ("%s: open/read %s failed: %s\n", program_name, argv[1], strerror (errno));
      return 2;
    }

  if (!(reconst_out = fopen (argv[2], FOPEN_WRITE_ARG)))
    {
      g_print ("%s: open/write %s failed: %s\n", program_name, argv[2], strerror (errno));
      return 2;
    }

  delta.dptr = patch_map->seg;
  delta.dsize = patch_map->len;

  from.dptr = from_map->seg;
  from.dsize = from_map->len;

  if (delta.dsize < 40)
    goto bail;

  if (strncmp (delta.dptr, XDELTA_PREFIX, XDELTA_PREFIX_LEN) != 0)
    goto bail;

  md5_buffer (from_map->seg, from_map->len, md5);

  if (memcmp (delta.dptr + XDELTA_PREFIX_LEN, md5, 16) != 0)
    goto badfrom;

  deltaoff.dptr = delta.dptr + XDELTA_PREFIX_LEN + 32;
  deltaoff.dsize = delta.dsize - XDELTA_PREFIX_LEN - 32;

  reconst = xpatch (deltaoff, &from, 1);

  if (!reconst.dptr)
    goto bail;

  if (fwrite (reconst.dptr, reconst.dsize, 1, reconst_out) != 1 ||
      fclose (reconst_out))
    {
      g_print ("%s: open/write %s failed: %s\n", program_name, argv[2], strerror (errno));
      return 2;
    }

  md5_buffer (reconst.dptr, reconst.dsize, md5);

  if (memcmp (delta.dptr + XDELTA_PREFIX_LEN + 16, md5, 16) != 0)
    goto badto;

  return 0;

bail:
  g_print ("%s: invalid patch\n", program_name);

  return 2;

badfrom:
  g_print ("%s: mismatched FROM MD5\n", program_name);

  return 2;

badto:
  g_print ("%s: mismatched TO MD5\n", program_name);

  return 2;

}
