/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: fakeglib.c 1.3 Sun, 12 Oct 1997 20:29:34 -0700 jmacd $
 */

/* This file is derived from GLIB, distributed with GTK, written
 * by Peter Mattis, Josh MacDonald, and Spencer Kimball. */

/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "xdelta.h"
#include <stdlib.h>
#include <string.h>

/* #define MEM_PROFILE */
/* #define MEM_CHECK */


#define MAX_MEM_AREA  65536L
#define MEM_AREA_SIZE 4L

#define MEM_ALIGN     8

typedef struct _GFreeAtom      GFreeAtom;
typedef struct _GMemArea       GMemArea;
typedef struct _GRealMemChunk  GRealMemChunk;

struct _GFreeAtom
{
  GFreeAtom *next;
};

struct _GMemArea
{
  GMemArea *next;            /* the next mem area */
  GMemArea *prev;            /* the previous mem area */
  gulong index;              /* the current index into the "mem" array */
  gulong free;               /* the number of free bytes in this mem area */
  gulong allocated;          /* the number of atoms allocated from this area */
  gulong mark;               /* is this mem area marked for deletion */
  gchar mem[MEM_AREA_SIZE];  /* the mem array from which atoms get allocated
			      * the actual size of this array is determined by
			      *  the mem chunk "area_size". ANSI says that it
			      *  must be declared to be the maximum size it
			      *  can possibly be (even though the actual size
			      *  may be less).
			      */
};

struct _GRealMemChunk
{
  gchar *name;               /* name of this MemChunk...used for debugging output */
  gint type;                 /* the type of MemChunk: ALLOC_ONLY or ALLOC_AND_FREE */
  gint num_mem_areas;        /* the number of memory areas */
  gint num_marked_areas;     /* the number of areas marked for deletion */
  gint atom_size;            /* the size of an atom */
  gulong area_size;          /* the size of a memory area */
  GMemArea *mem_area;        /* the current memory area */
  GMemArea *mem_areas;       /* a list of all the mem areas owned by this chunk */
  GMemArea *free_mem_area;   /* the free area...which is about to be destroyed */
  GFreeAtom *free_atoms;     /* the free atoms list */
  GTree *mem_tree;           /* tree of mem areas sorted by memory address */
  GRealMemChunk *next;       /* pointer to the next chunk */
  GRealMemChunk *prev;       /* pointer to the previous chunk */
};


static gulong g_mem_chunk_compute_size (gulong    size);
static gint   g_mem_chunk_area_compare (GMemArea *a,
					GMemArea *b);
static gint   g_mem_chunk_area_search  (GMemArea *a,
					gchar    *addr);


static GRealMemChunk *mem_chunks = NULL;

#ifdef MEM_PROFILE
static gulong allocations[4096] = { 0 };
static gulong allocated_mem = 0;
static gulong freed_mem = 0;
#endif /* MEM_PROFILE */


#ifndef USE_DMALLOC

gpointer
g_malloc (gulong size)
{
  gpointer p;


#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  gulong *t;
#endif /* MEM_PROFILE || MEM_CHECK */


  if (size == 0)
    return NULL;


#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  size += SIZEOF_LONG;
#endif /* MEM_PROFILE || MEM_CHECK */

#ifdef MEM_CHECK
  size += SIZEOF_LONG;
#endif /* MEM_CHECK */


  p = (gpointer) malloc (size);
  if (!p)
    {
      g_print ("could not allocate %d bytes", (int)size);
      exit (2);
    }


#ifdef MEM_CHECK
  size -= SIZEOF_LONG;

  t = p;
  p = ((guchar*) p + SIZEOF_LONG);
  *t = 0;
#endif /* MEM_CHECK */

#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  size -= SIZEOF_LONG;

  t = p;
  p = ((guchar*) p + SIZEOF_LONG);
  *t = size;

#ifdef MEM_PROFILE
  if (size <= 4095)
    allocations[size-1] += 1;
  else
    allocations[4095] += 1;
  allocated_mem += size;
#endif /* MEM_PROFILE */
#endif /* MEM_PROFILE || MEM_CHECK */


  return p;
}

gpointer
g_malloc0 (gulong size)
{
  gpointer p;


#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  gulong *t;
#endif /* MEM_PROFILE || MEM_CHECK */


  if (size == 0)
    return NULL;


#ifdef MEM_PROFILE
  size += SIZEOF_LONG;
#endif /* MEM_PROFILE */

#ifdef MEM_CHECK
  size += SIZEOF_LONG;
#endif /* MEM_CHECK */


  p = (gpointer) calloc (size, 1);
  if (!p)
    {
      g_print ("could not allocate %d bytes", (int)size);
      exit (2);
    }


#ifdef MEM_CHECK
  size -= SIZEOF_LONG;

  t = p;
  p = ((guchar*) p + SIZEOF_LONG);
  *t = 0;
#endif /* MEM_CHECK */

#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  size -= SIZEOF_LONG;

  t = p;
  p = ((guchar*) p + SIZEOF_LONG);
  *t = size;

#ifdef MEM_PROFILE
  if (size <= 4095)
    allocations[size-1] += 1;
  else
    allocations[4095] += 1;
  allocated_mem += size;
#endif /* MEM_PROFILE */
#endif /* MEM_PROFILE */


  return p;
}

gpointer
g_realloc (gpointer mem,
	   gulong   size)
{
  gpointer p;

#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  gulong *t;
#endif /* MEM_PROFILE || MEM_CHECK */


  if (size == 0)
    return NULL;


#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  size += SIZEOF_LONG;
#endif /* MEM_PROFILE || MEM_CHECK */

#ifdef MEM_CHECK
  size += SIZEOF_LONG;
#endif /* MEM_CHECK */


  if (!mem)
    p = (gpointer) malloc (size);
  else
    {
#if defined(MEM_PROFILE) || defined(MEM_CHECK)
      t = (gulong*) ((guchar*) mem - SIZEOF_LONG);
#ifdef MEM_PROFILE
      freed_mem += *t;
#endif /* MEM_PROFILE */
      mem = t;
#endif /* MEM_PROFILE || MEM_CHECK */

#ifdef MEM_CHECK
      t = (gulong*) ((guchar*) mem - SIZEOF_LONG);
      if (*t >= 1)
	g_warning ("trying to realloc freed memory\n");
      mem = t;
#endif /* MEM_CHECK */

      p = (gpointer) realloc (mem, size);
    }

  if (!p)
    {
      g_print ("could not reallocate %d bytes", (int)size);
    }


#ifdef MEM_CHECK
  size -= SIZEOF_LONG;

  t = p;
  p = ((guchar*) p + SIZEOF_LONG);
  *t = 0;
#endif /* MEM_CHECK */

#if defined(MEM_PROFILE) || defined(MEM_CHECK)
  size -= SIZEOF_LONG;

  t = p;
  p = ((guchar*) p + SIZEOF_LONG);
  *t = size;

#ifdef MEM_PROFILE
  if (size <= 4095)
    allocations[size-1] += 1;
  else
    allocations[4095] += 1;
  allocated_mem += size;
#endif /* MEM_PROFILE */
#endif /* MEM_PROFILE || MEM_CHECK */


  return p;
}

void
g_free (gpointer mem)
{
  if (mem)
    {
#if defined(MEM_PROFILE) || defined(MEM_CHECK)
      gulong *t;
      gulong size;
#endif /* MEM_PROFILE || MEM_CHECK */

#if defined(MEM_PROFILE) || defined(MEM_CHECK)
      t = (gulong*) ((guchar*) mem - SIZEOF_LONG);
      size = *t;
#ifdef MEM_PROFILE
      freed_mem += size;
#endif /* MEM_PROFILE */
      mem = t;
#endif /* MEM_PROFILE || MEM_CHECK */

#ifdef MEM_CHECK
      t = (gulong*) ((guchar*) mem - SIZEOF_LONG);
      if (*t >= 1)
	g_warning ("freeing previously freed memory\n");
      *t += 1;
      mem = t;

      memset ((guchar*) mem + 8, 0, size);
#else /* MEM_CHECK */
      free (mem);
#endif /* MEM_CHECK */
    }
}

#endif /* ! USE_DMALLOC */


void
g_mem_profile ()
{
#ifdef MEM_PROFILE
  gint i;

  for (i = 0; i < 4095; i++)
    if (allocations[i] > 0)
      g_print ("%lu allocations of %d bytes\n", allocations[i], i + 1);

  if (allocations[4095] > 0)
    g_print ("%lu allocations of greater than 4095 bytes\n", allocations[4095]);
  g_print ("%lu bytes allocated\n", allocated_mem);
  g_print ("%lu bytes freed\n", freed_mem);
  g_print ("%lu bytes in use\n", allocated_mem - freed_mem);
#endif /* MEM_PROFILE */
}

void
g_mem_check (gpointer mem)
{
#ifdef MEM_CHECK
  gulong *t;

  t = (gulong*) ((guchar*) mem - SIZEOF_LONG - SIZEOF_LONG);

  if (*t >= 1)
    g_warning ("mem: 0x%08x has been freed: %lu\n", (gulong) mem, *t);
#endif /* MEM_CHECK */
}

GMemChunk*
g_mem_chunk_new (gchar  *name,
		 gint    atom_size,
		 gulong  area_size,
		 gint    type)
{
  GRealMemChunk *mem_chunk;
  gulong rarea_size;

  mem_chunk = g_new (struct _GRealMemChunk, 1);
  mem_chunk->name = name;
  mem_chunk->type = type;
  mem_chunk->num_mem_areas = 0;
  mem_chunk->num_marked_areas = 0;
  mem_chunk->mem_area = NULL;
  mem_chunk->free_mem_area = NULL;
  mem_chunk->free_atoms = NULL;
  mem_chunk->mem_tree = NULL;
  mem_chunk->mem_areas = NULL;
  mem_chunk->atom_size = atom_size;

  if (mem_chunk->type == G_ALLOC_AND_FREE)
    mem_chunk->mem_tree = g_tree_new ((GCompareFunc) g_mem_chunk_area_compare);

  if (mem_chunk->atom_size % MEM_ALIGN)
    mem_chunk->atom_size += MEM_ALIGN - (mem_chunk->atom_size % MEM_ALIGN);

  mem_chunk->area_size = area_size;
  if (mem_chunk->area_size > MAX_MEM_AREA)
    mem_chunk->area_size = MAX_MEM_AREA;
  while (mem_chunk->area_size < mem_chunk->atom_size)
    mem_chunk->area_size *= 2;

  rarea_size = mem_chunk->area_size + sizeof (GMemArea) - MEM_AREA_SIZE;
  rarea_size = g_mem_chunk_compute_size (rarea_size);
  mem_chunk->area_size = rarea_size - (sizeof (GMemArea) - MEM_AREA_SIZE);

  /*
  mem_chunk->area_size -= (sizeof (GMemArea) - MEM_AREA_SIZE);
  if (mem_chunk->area_size < mem_chunk->atom_size)
    {
      mem_chunk->area_size = (mem_chunk->area_size + sizeof (GMemArea) - MEM_AREA_SIZE) * 2;
      mem_chunk->area_size -= (sizeof (GMemArea) - MEM_AREA_SIZE);
    }

  if (mem_chunk->area_size % mem_chunk->atom_size)
    mem_chunk->area_size += mem_chunk->atom_size - (mem_chunk->area_size % mem_chunk->atom_size);
    */

  mem_chunk->next = mem_chunks;
  mem_chunk->prev = NULL;
  if (mem_chunks)
    mem_chunks->prev = mem_chunk;
  mem_chunks = mem_chunk;

  return ((GMemChunk*) mem_chunk);
}

void
g_mem_chunk_destroy (GMemChunk *mem_chunk)
{
  GRealMemChunk *rmem_chunk;
  GMemArea *mem_areas;
  GMemArea *temp_area;

  assert (mem_chunk != NULL);

  rmem_chunk = (GRealMemChunk*) mem_chunk;

  mem_areas = rmem_chunk->mem_areas;
  while (mem_areas)
    {
      temp_area = mem_areas;
      mem_areas = mem_areas->next;
      g_free (temp_area);
    }

  if (rmem_chunk->next)
    rmem_chunk->next->prev = rmem_chunk->prev;
  if (rmem_chunk->prev)
    rmem_chunk->prev->next = rmem_chunk->next;

  if (rmem_chunk == mem_chunks)
    mem_chunks = mem_chunks->next;

  if (rmem_chunk->type == G_ALLOC_AND_FREE)
    g_tree_destroy (rmem_chunk->mem_tree);

  g_free (rmem_chunk);
}

gpointer
g_mem_chunk_alloc (GMemChunk *mem_chunk)
{
  GRealMemChunk *rmem_chunk;
  GMemArea *temp_area;
  gpointer mem;

  assert (mem_chunk != NULL);

  rmem_chunk = (GRealMemChunk*) mem_chunk;

  while (rmem_chunk->free_atoms)
    {
      /* Get the first piece of memory on the "free_atoms" list.
       * We can go ahead and destroy the list node we used to keep
       *  track of it with and to update the "free_atoms" list to
       *  point to its next element.
       */
      mem = rmem_chunk->free_atoms;
      rmem_chunk->free_atoms = rmem_chunk->free_atoms->next;

      /* Determine which area this piece of memory is allocated from */
      temp_area = g_tree_search (rmem_chunk->mem_tree,
				 (GSearchFunc) g_mem_chunk_area_search,
				 mem);

      /* If the area has been marked, then it is being destroyed.
       *  (ie marked to be destroyed).
       * We check to see if all of the segments on the free list that
       *  reference this area have been removed. This occurs when
       *  the ammount of free memory is less than the allocatable size.
       * If the chunk should be freed, then we place it in the "free_mem_area".
       * This is so we make sure not to free the mem area here and then
       *  allocate it again a few lines down.
       * If we don't allocate a chunk a few lines down then the "free_mem_area"
       *  will be freed.
       * If there is already a "free_mem_area" then we'll just free this mem area.
       */
      if (temp_area->mark)
        {
          /* Update the "free" memory available in that area */
          temp_area->free += rmem_chunk->atom_size;

          if (temp_area->free == rmem_chunk->area_size)
            {
              if (temp_area == rmem_chunk->mem_area)
                rmem_chunk->mem_area = NULL;

              if (rmem_chunk->free_mem_area)
                {
                  rmem_chunk->num_mem_areas -= 1;
                  rmem_chunk->num_marked_areas -= 1;

                  if (temp_area->next)
                    temp_area->next->prev = temp_area->prev;
                  if (temp_area->prev)
                    temp_area->prev->next = temp_area->next;
                  if (temp_area == rmem_chunk->mem_areas)
                    rmem_chunk->mem_areas = rmem_chunk->mem_areas->next;
                  if (temp_area == rmem_chunk->mem_area)
                    rmem_chunk->mem_area = NULL;

                  g_free (temp_area);
                }
              else
                rmem_chunk->free_mem_area = temp_area;
	    }
	}
      else
        {
          /* Update the number of allocated atoms count.
	   */
          temp_area->allocated += 1;

          /* The area wasn't marked...return the memory
	   */
	  goto outa_here;
        }
    }

  /* If there isn't a current mem area or the current mem area is out of space
   *  then allocate a new mem area. We'll first check and see if we can use
   *  the "free_mem_area". Otherwise we'll just malloc the mem area.
   */
  if ((!rmem_chunk->mem_area) ||
      ((rmem_chunk->mem_area->index + rmem_chunk->atom_size) > rmem_chunk->area_size))
    {
      if (rmem_chunk->free_mem_area)
        {
          rmem_chunk->mem_area = rmem_chunk->free_mem_area;
	  rmem_chunk->free_mem_area = NULL;
        }
      else
        {
	  rmem_chunk->mem_area = (GMemArea*) g_malloc (sizeof (GMemArea) -
						       MEM_AREA_SIZE +
						       rmem_chunk->area_size);

	  rmem_chunk->num_mem_areas += 1;
	  rmem_chunk->mem_area->next = rmem_chunk->mem_areas;
	  rmem_chunk->mem_area->prev = NULL;

	  if (rmem_chunk->mem_areas)
	    rmem_chunk->mem_areas->prev = rmem_chunk->mem_area;
	  rmem_chunk->mem_areas = rmem_chunk->mem_area;

	  if (rmem_chunk->type == G_ALLOC_AND_FREE)
	    g_tree_insert (rmem_chunk->mem_tree, rmem_chunk->mem_area, rmem_chunk->mem_area);
        }

      rmem_chunk->mem_area->index = 0;
      rmem_chunk->mem_area->free = rmem_chunk->area_size;
      rmem_chunk->mem_area->allocated = 0;
      rmem_chunk->mem_area->mark = 0;
    }
  else if (rmem_chunk->free_mem_area)
    {
      rmem_chunk->num_mem_areas -= 1;

      if (rmem_chunk->free_mem_area->next)
	rmem_chunk->free_mem_area->next->prev = rmem_chunk->free_mem_area->prev;
      if (rmem_chunk->free_mem_area->prev)
	rmem_chunk->free_mem_area->prev->next = rmem_chunk->free_mem_area->next;
      if (rmem_chunk->free_mem_area == rmem_chunk->mem_areas)
	rmem_chunk->mem_areas = rmem_chunk->mem_areas->next;

      if (rmem_chunk->type == G_ALLOC_AND_FREE)
	g_tree_remove (rmem_chunk->mem_tree, rmem_chunk->free_mem_area);

      g_free (rmem_chunk->free_mem_area);
      rmem_chunk->free_mem_area = NULL;
    }

  /* Get the memory and modify the state variables appropriately.
   */
  mem = (gpointer) &rmem_chunk->mem_area->mem[rmem_chunk->mem_area->index];
  rmem_chunk->mem_area->index += rmem_chunk->atom_size;
  rmem_chunk->mem_area->free -= rmem_chunk->atom_size;
  rmem_chunk->mem_area->allocated += 1;

outa_here:
  return mem;
}

void
g_mem_chunk_free (GMemChunk *mem_chunk,
		  gpointer   mem)
{
  GRealMemChunk *rmem_chunk;
  GMemArea *temp_area;
  GFreeAtom *free_atom;

  assert (mem_chunk != NULL);
  assert (mem != NULL);

  rmem_chunk = (GRealMemChunk*) mem_chunk;

  /* Don't do anything if this is an ALLOC_ONLY chunk
   */
  if (rmem_chunk->type == G_ALLOC_AND_FREE)
    {
      /* Place the memory on the "free_atoms" list
       */
      free_atom = (GFreeAtom*) mem;
      free_atom->next = rmem_chunk->free_atoms;
      rmem_chunk->free_atoms = free_atom;

      temp_area = g_tree_search (rmem_chunk->mem_tree,
				 (GSearchFunc) g_mem_chunk_area_search,
				 mem);

      temp_area->allocated -= 1;

      if (temp_area->allocated == 0)
	{
	  temp_area->mark = 1;
	  rmem_chunk->num_marked_areas += 1;

	  g_mem_chunk_clean (mem_chunk);
	}
    }
}

void
g_mem_chunk_clean (GMemChunk *mem_chunk)
{
  GRealMemChunk *rmem_chunk;
  GMemArea *mem_area;
  GFreeAtom *prev_free_atom;
  GFreeAtom *temp_free_atom;
  gpointer mem;

  assert (mem_chunk != NULL);

  rmem_chunk = (GRealMemChunk*) mem_chunk;

  if (rmem_chunk->type == G_ALLOC_AND_FREE)
    {
      prev_free_atom = NULL;
      temp_free_atom = rmem_chunk->free_atoms;

      while (temp_free_atom)
	{
	  mem = (gpointer) temp_free_atom;

	  mem_area = g_tree_search (rmem_chunk->mem_tree,
				    (GSearchFunc) g_mem_chunk_area_search,
				    mem);

          /* If this mem area is marked for destruction then delete the
	   *  area and list node and decrement the free mem.
           */
	  if (mem_area->mark)
	    {
	      if (prev_free_atom)
		prev_free_atom->next = temp_free_atom->next;
	      else
		rmem_chunk->free_atoms = temp_free_atom->next;
	      temp_free_atom = temp_free_atom->next;

	      mem_area->free += rmem_chunk->atom_size;
	      if (mem_area->free == rmem_chunk->area_size)
		{
		  rmem_chunk->num_mem_areas -= 1;
		  rmem_chunk->num_marked_areas -= 1;

		  if (mem_area->next)
		    mem_area->next->prev = mem_area->prev;
		  if (mem_area->prev)
		    mem_area->prev->next = mem_area->next;
		  if (mem_area == rmem_chunk->mem_areas)
		    rmem_chunk->mem_areas = rmem_chunk->mem_areas->next;
		  if (mem_area == rmem_chunk->mem_area)
		    rmem_chunk->mem_area = NULL;

		  if (rmem_chunk->type == G_ALLOC_AND_FREE)
		    g_tree_remove (rmem_chunk->mem_tree, mem_area);
		  g_free (mem_area);
		}
	    }
	  else
	    {
	      prev_free_atom = temp_free_atom;
	      temp_free_atom = temp_free_atom->next;
	    }
	}
    }
}

void
g_mem_chunk_reset (GMemChunk *mem_chunk)
{
  GRealMemChunk *rmem_chunk;
  GMemArea *mem_areas;
  GMemArea *temp_area;

  assert (mem_chunk != NULL);

  rmem_chunk = (GRealMemChunk*) mem_chunk;

  mem_areas = rmem_chunk->mem_areas;
  rmem_chunk->num_mem_areas = 0;
  rmem_chunk->mem_areas = NULL;
  rmem_chunk->mem_area = NULL;

  while (mem_areas)
    {
      temp_area = mem_areas;
      mem_areas = mem_areas->next;
      g_free (temp_area);
    }

  rmem_chunk->free_atoms = NULL;

  if (rmem_chunk->mem_tree)
    g_tree_destroy (rmem_chunk->mem_tree);
  rmem_chunk->mem_tree = g_tree_new ((GCompareFunc) g_mem_chunk_area_compare);
}

void
g_mem_chunk_print (GMemChunk *mem_chunk)
{
  GRealMemChunk *rmem_chunk;
  GMemArea *mem_areas;
  gulong mem;

  assert (mem_chunk != NULL);

  rmem_chunk = (GRealMemChunk*) mem_chunk;
  mem_areas = rmem_chunk->mem_areas;
  mem = 0;

  while (mem_areas)
    {
      mem += rmem_chunk->area_size - mem_areas->free;
      mem_areas = mem_areas->next;
    }

  g_print ("%s: %ld bytes using %d mem areas", rmem_chunk->name, mem, rmem_chunk->num_mem_areas);
}

void
g_mem_chunk_info ()
{
  GRealMemChunk *mem_chunk;
  gint count;

  count = 0;
  mem_chunk = mem_chunks;
  while (mem_chunk)
    {
      count += 1;
      mem_chunk = mem_chunk->next;
    }

  g_print ("%d mem chunks", count);

  mem_chunk = mem_chunks;
  while (mem_chunk)
    {
      g_mem_chunk_print ((GMemChunk*) mem_chunk);
      mem_chunk = mem_chunk->next;
    }
}

void
g_blow_chunks ()
{
  GRealMemChunk *mem_chunk;

  mem_chunk = mem_chunks;
  while (mem_chunk)
    {
      g_mem_chunk_clean ((GMemChunk*) mem_chunk);
      mem_chunk = mem_chunk->next;
    }
}


static gulong
g_mem_chunk_compute_size (gulong size)
{
  gulong power_of_2;
  gulong lower, upper;

  power_of_2 = 16;
  while (power_of_2 < size)
    power_of_2 <<= 1;

  lower = power_of_2 >> 1;
  upper = power_of_2;

  if ((size - lower) < (upper - size))
    return lower;
  return upper;
}

static gint
g_mem_chunk_area_compare (GMemArea *a,
			  GMemArea *b)
{
  return (a->mem - b->mem);
}

static gint
g_mem_chunk_area_search (GMemArea *a,
			 gchar    *addr)
{
  if (a->mem <= addr)
    {
      if (addr < &a->mem[a->index])
	return 0;
      return 1;
    }
  return -1;
}


typedef struct _GRealListAllocator GRealListAllocator;

struct _GRealListAllocator
{
  GMemChunk *list_mem_chunk;
  GSList    *free_list;
};


static GRealListAllocator *default_allocator = NULL;
static GRealListAllocator *current_allocator = NULL;

GListAllocator*
g_slist_set_allocator (GListAllocator* fallocator)
{
  GRealListAllocator* allocator = (GRealListAllocator *) fallocator;
  GRealListAllocator* old_allocator = current_allocator;

  if (allocator)
    current_allocator = allocator;
  else
    {
      if (!default_allocator)
	default_allocator = (GRealListAllocator*) g_list_allocator_new ();
      current_allocator = default_allocator;
    }

  if (!current_allocator->list_mem_chunk)
    current_allocator->list_mem_chunk = g_mem_chunk_new ("slist mem chunk",
							 sizeof (GSList),
							 1024,
							 G_ALLOC_ONLY);

  return (GListAllocator*) (old_allocator == default_allocator ? NULL : old_allocator);
}


GSList*
g_slist_alloc ()
{
  GSList *new_list;

  g_slist_set_allocator (NULL);
  if (current_allocator->free_list)
    {
      new_list = current_allocator->free_list;
      current_allocator->free_list = current_allocator->free_list->next;
    }
  else
    {
      new_list = g_chunk_new (GSList, current_allocator->list_mem_chunk);
    }

  new_list->data = NULL;
  new_list->next = NULL;

  return new_list;
}

void
g_slist_free (GSList *list)
{
  GSList *last;

  if (list)
    {
      last = g_slist_last (list);
      last->next = current_allocator->free_list;
      current_allocator->free_list = list;
    }
}

void
g_slist_free_1 (GSList *list)
{
  if (list)
    {
      list->next = current_allocator->free_list;
      current_allocator->free_list = list;
    }
}

GSList*
g_slist_append (GSList   *list,
		gpointer  data)
{
  GSList *new_list;
  GSList *last;

  new_list = g_slist_alloc ();
  new_list->data = data;

  if (!list)
    {
      list = new_list;
    }
  else
    {
      last = g_slist_last (list);
      assert (last != NULL);
      last->next = new_list;
    }

  return list;
}

GSList*
g_slist_prepend (GSList   *list,
		 gpointer  data)
{
  GSList *new_list;

  new_list = g_slist_alloc ();
  new_list->data = data;
  new_list->next = list;

  return new_list;
}

GSList*
g_slist_insert (GSList   *list,
		gpointer  data,
		gint      position)
{
  GSList *prev_list;
  GSList *tmp_list;
  GSList *new_list;

  prev_list = NULL;
  tmp_list = list;

  while (tmp_list && (position-- > 0))
    {
      prev_list = tmp_list;
      tmp_list = tmp_list->next;
    }

  if (!tmp_list && !prev_list)
    return list;

  new_list = g_slist_alloc ();

  if (!prev_list)
    {
      new_list->next = list;
      list = new_list;
    }
  else
    {
      new_list->next = prev_list->next;
      prev_list->next = new_list;
    }

  return list;
}

GSList*
g_slist_remove (GSList   *list,
		gpointer  data)
{
  GSList *tmp;
  GSList *prev;

  prev = NULL;
  tmp = list;

  while (tmp)
    {
      if (tmp->data == data)
	{
	  if (prev)
	    prev->next = tmp->next;
	  if (list == tmp)
	    list = list->next;

	  tmp->next = NULL;
	  g_slist_free (tmp);

	  break;
	}

      prev = tmp;
      tmp = tmp->next;
    }

  return list;
}

GSList*
g_slist_remove_link (GSList *list,
		     GSList *link)
{
  GSList *tmp;
  GSList *prev;

  prev = NULL;
  tmp = list;

  while (tmp)
    {
      if (tmp == link)
	{
	  if (prev)
	    prev->next = tmp->next;
	  if (list == tmp)
	    list = list->next;

	  tmp->next = NULL;
	  break;
	}

      prev = tmp;
      tmp = tmp->next;
    }

  return list;
}

GSList*
g_slist_reverse (GSList *list)
{
  GSList *tmp;
  GSList *prev;
  GSList *last;

  last = NULL;
  prev = NULL;

  while (list)
    {
      last = list;

      tmp = list->next;
      list->next = prev;

      prev = list;
      list = tmp;
    }

  return last;
}

GSList*
g_slist_nth (GSList *list,
	     gint    n)
{
  while ((n-- > 0) && list)
    list = list->next;

  return list;
}

GSList*
g_slist_find (GSList   *list,
	      gpointer  data)
{
  while (list)
    {
      if (list->data == data)
	break;
      list = list->next;
    }

  return list;
}

GSList*
g_slist_last (GSList *list)
{
  if (list)
    {
      while (list->next)
	list = list->next;
    }

  return list;
}

gint
g_slist_length (GSList *list)
{
  gint length;

  length = 0;
  while (list)
    {
      length++;
      list = list->next;
    }

  return length;
}

void
g_slist_foreach (GSList   *list,
		 GFunc     func,
		 gpointer  user_data)
{
  while (list)
    {
      (*func) (list->data, user_data);
      list = list->next;
    }
}

typedef struct _GRealTree  GRealTree;
typedef struct _GTreeNode  GTreeNode;

struct _GRealTree
{
  GTreeNode *root;
  GCompareFunc key_compare;
};

struct _GTreeNode
{
  gint balance;      /* height (left) - height (right) */
  GTreeNode *left;   /* left subtree */
  GTreeNode *right;  /* right subtree */
  gpointer key;      /* key for this node */
  gpointer value;    /* value stored at this node */
};


static GTreeNode* g_tree_node_new                   (gpointer        key,
						     gpointer        value);
static void       g_tree_node_destroy               (GTreeNode      *node);
static GTreeNode* g_tree_node_insert                (GTreeNode      *node,
						     GCompareFunc    compare,
						     gpointer        key,
						     gpointer        value,
						     gint           *inserted);
static GTreeNode* g_tree_node_remove                (GTreeNode      *node,
						     GCompareFunc    compare,
						     gpointer        key);
static GTreeNode* g_tree_node_balance               (GTreeNode      *node);
static GTreeNode* g_tree_node_remove_leftmost       (GTreeNode      *node,
						     GTreeNode     **leftmost);
static GTreeNode* g_tree_node_restore_left_balance  (GTreeNode      *node,
						     gint            old_balance);
static GTreeNode* g_tree_node_restore_right_balance (GTreeNode      *node,
						     gint            old_balance);
static gpointer   g_tree_node_lookup                (GTreeNode      *node,
						     GCompareFunc    compare,
						     gpointer        key);
static gint       g_tree_node_count                 (GTreeNode      *node);
static gint       g_tree_node_pre_order             (GTreeNode      *node,
						     GTraverseFunc   traverse_func,
						     gpointer        data);
static gint       g_tree_node_in_order              (GTreeNode      *node,
						     GTraverseFunc   traverse_func,
						     gpointer        data);
static gint       g_tree_node_post_order            (GTreeNode      *node,
						     GTraverseFunc   traverse_func,
						     gpointer        data);
static gpointer   g_tree_node_search                (GTreeNode      *node,
						     GSearchFunc     search_func,
						     gpointer        data);
static gint       g_tree_node_height                (GTreeNode      *node);
static GTreeNode* g_tree_node_rotate_left           (GTreeNode      *node);
static GTreeNode* g_tree_node_rotate_right          (GTreeNode      *node);
static void       g_tree_node_check                 (GTreeNode      *node);


static GMemChunk *node_mem_chunk = NULL;
static GSList *node_free_list = NULL;


GTree*
g_tree_new (GCompareFunc key_compare_func)
{
  GRealTree *rtree;

  rtree = g_new (GRealTree, 1);
  rtree->root = NULL;
  rtree->key_compare = key_compare_func;

  return (GTree*) rtree;
}

void
g_tree_destroy (GTree *tree)
{
  GRealTree *rtree;

  rtree = (GRealTree*) tree;

  g_tree_node_destroy (rtree->root);
  g_free (rtree);
}

void
g_tree_insert (GTree    *tree,
	       gpointer  key,
	       gpointer  value)
{
  GRealTree *rtree;
  gint inserted;

  rtree = (GRealTree*) tree;

  inserted = FALSE;
  rtree->root = g_tree_node_insert (rtree->root, rtree->key_compare,
				    key, value, &inserted);
}

void
g_tree_remove (GTree    *tree,
	       gpointer  key)
{
  GRealTree *rtree;

  rtree = (GRealTree*) tree;

  rtree->root = g_tree_node_remove (rtree->root, rtree->key_compare, key);
}

gpointer
g_tree_lookup (GTree    *tree,
	       gpointer  key)
{
  GRealTree *rtree;

  rtree = (GRealTree*) tree;

  return g_tree_node_lookup (rtree->root, rtree->key_compare, key);
}

void
g_tree_traverse (GTree         *tree,
		 GTraverseFunc  traverse_func,
		 GTraverseType  traverse_type,
		 gpointer       data)
{
  GRealTree *rtree;

  rtree = (GRealTree*) tree;

  switch (traverse_type)
    {
    case G_PRE_ORDER:
      g_tree_node_pre_order (rtree->root, traverse_func, data);
      break;

    case G_IN_ORDER:
      g_tree_node_in_order (rtree->root, traverse_func, data);
      break;

    case G_POST_ORDER:
      g_tree_node_post_order (rtree->root, traverse_func, data);
      break;
    }
}

gpointer
g_tree_search (GTree       *tree,
	       GSearchFunc  search_func,
	       gpointer     data)
{
  GRealTree *rtree;

  rtree = (GRealTree*) tree;

  if (rtree->root)
    return g_tree_node_search (rtree->root, search_func, data);
  return NULL;
}

gint
g_tree_height (GTree *tree)
{
  GRealTree *rtree;

  rtree = (GRealTree*) tree;

  if (rtree->root)
    return g_tree_node_height (rtree->root);
  return 0;
}

gint
g_tree_nnodes (GTree *tree)
{
  GRealTree *rtree;

  rtree = (GRealTree*) tree;

  if (rtree->root)
    return g_tree_node_count (rtree->root);
  return 0;
}


static GTreeNode*
g_tree_node_new (gpointer key,
		 gpointer value)
{
  GTreeNode *node;
  GSList *tmp_list;

  if (node_free_list)
    {
      tmp_list = node_free_list;
      node_free_list = node_free_list->next;

      node = tmp_list->data;

      {
	GListAllocator *tmp_allocator = g_list_set_allocator (NULL);
	g_slist_free_1 (tmp_list);
	g_list_set_allocator (tmp_allocator);
      }
    }
  else
    {
      if (!node_mem_chunk)
	node_mem_chunk = g_mem_chunk_new ("tree node mem chunk", sizeof (GTreeNode), 1024, G_ALLOC_ONLY);

      node = g_chunk_new (GTreeNode, node_mem_chunk);
    }

  node->balance = 0;
  node->left = NULL;
  node->right = NULL;
  node->key = key;
  node->value = value;

  return node;
}

static void
g_tree_node_destroy (GTreeNode *node)
{
  if (node)
    {
      node_free_list = g_slist_prepend (node_free_list, node);
      g_tree_node_destroy (node->right);
      g_tree_node_destroy (node->left);
    }
}

static GTreeNode*
g_tree_node_insert (GTreeNode    *node,
		    GCompareFunc  compare,
		    gpointer      key,
		    gpointer      value,
		    gint         *inserted)
{
  gint old_balance;
  gint cmp;

  if (!node)
    {
      *inserted = TRUE;
      return g_tree_node_new (key, value);
    }

  cmp = (* compare) (key, node->key);
  if (cmp == 0)
    {
      *inserted = FALSE;
      node->value = value;
      return node;
    }

  if (cmp < 0)
    {
      if (node->left)
	{
	  old_balance = node->left->balance;
	  node->left = g_tree_node_insert (node->left, compare, key, value, inserted);

	  if ((old_balance != node->left->balance) && node->left->balance)
	    node->balance -= 1;
	}
      else
	{
	  *inserted = TRUE;
	  node->left = g_tree_node_new (key, value);
	  node->balance -= 1;
	}
    }
  else if (cmp > 0)
    {
      if (node->right)
	{
	  old_balance = node->right->balance;
	  node->right = g_tree_node_insert (node->right, compare, key, value, inserted);

	  if ((old_balance != node->right->balance) && node->right->balance)
	    node->balance += 1;
	}
      else
	{
	  *inserted = TRUE;
	  node->right = g_tree_node_new (key, value);
	  node->balance += 1;
	}
    }

  if (*inserted)
    {
      if ((node->balance < -1) || (node->balance > 1))
	node = g_tree_node_balance (node);
    }

  return node;
}

static GTreeNode*
g_tree_node_remove (GTreeNode    *node,
		    GCompareFunc  compare,
		    gpointer      key)
{
  GTreeNode *garbage;
  GTreeNode *new_root;
  gint old_balance;
  gint cmp;

  if (!node)
    return NULL;

  cmp = (* compare) (key, node->key);
  if (cmp == 0)
    {
      garbage = node;

      if (!node->right)
	{
	  node = node->left;
	}
      else
	{
	  old_balance = node->right->balance;
	  node->right = g_tree_node_remove_leftmost (node->right, &new_root);
	  new_root->left = node->left;
	  new_root->right = node->right;
	  new_root->balance = node->balance;
	  node = g_tree_node_restore_right_balance (new_root, old_balance);
	}

      node_free_list = g_slist_prepend (node_free_list, garbage);
    }
  else if (cmp < 0)
    {
      if (node->left)
	{
	  old_balance = node->left->balance;
	  node->left = g_tree_node_remove (node->left, compare, key);
	  node = g_tree_node_restore_left_balance (node, old_balance);
	}
    }
  else if (cmp > 0)
    {
      if (node->right)
	{
	  old_balance = node->right->balance;
	  node->right = g_tree_node_remove (node->right, compare, key);
	  node = g_tree_node_restore_right_balance (node, old_balance);
	}
    }

  return node;
}

static GTreeNode*
g_tree_node_balance (GTreeNode *node)
{
  if (node->balance < -1)
    {
      if (node->left->balance > 0)
	node->left = g_tree_node_rotate_left (node->left);
      node = g_tree_node_rotate_right (node);
    }
  else if (node->balance > 1)
    {
      if (node->right->balance < 0)
	node->right = g_tree_node_rotate_right (node->right);
      node = g_tree_node_rotate_left (node);
    }

  return node;
}

static GTreeNode*
g_tree_node_remove_leftmost (GTreeNode  *node,
			     GTreeNode **leftmost)
{
  gint old_balance;

  if (!node->left)
    {
      *leftmost = node;
      return node->right;
    }

  old_balance = node->left->balance;
  node->left = g_tree_node_remove_leftmost (node->left, leftmost);
  return g_tree_node_restore_left_balance (node, old_balance);
}

static GTreeNode*
g_tree_node_restore_left_balance (GTreeNode *node,
				  gint       old_balance)
{
  if (!node->left)
    node->balance += 1;
  else if ((node->left->balance != old_balance) &&
	   (node->left->balance == 0))
    node->balance += 1;

  if (node->balance > 1)
    return g_tree_node_balance (node);
  return node;
}

static GTreeNode*
g_tree_node_restore_right_balance (GTreeNode *node,
				   gint       old_balance)
{
  if (!node->right)
    node->balance -= 1;
  else if ((node->right->balance != old_balance) &&
	   (node->right->balance == 0))
    node->balance -= 1;

  if (node->balance < -1)
    return g_tree_node_balance (node);
  return node;
}

static gpointer
g_tree_node_lookup (GTreeNode    *node,
		    GCompareFunc  compare,
		    gpointer      key)
{
  gint cmp;

  if (!node)
    return NULL;

  cmp = (* compare) (key, node->key);
  if (cmp == 0)
    return node->value;

  if (cmp < 0)
    {
      if (node->left)
	return g_tree_node_lookup (node->left, compare, key);
    }
  else if (cmp > 0)
    {
      if (node->right)
	return g_tree_node_lookup (node->right, compare, key);
    }

  return NULL;
}

static gint
g_tree_node_count (GTreeNode *node)
{
  gint count;

  count = 1;
  if (node->left)
    count += g_tree_node_count (node->left);
  if (node->right)
    count += g_tree_node_count (node->right);

  return count;
}

static gint
g_tree_node_pre_order (GTreeNode     *node,
		       GTraverseFunc  traverse_func,
		       gpointer       data)
{
  if ((*traverse_func) (node->key, node->value, data))
    return TRUE;
  if (node->left)
    {
      if (g_tree_node_pre_order (node->left, traverse_func, data))
	return TRUE;
    }
  if (node->right)
    {
      if (g_tree_node_pre_order (node->right, traverse_func, data))
	return TRUE;
    }

  return FALSE;
}

static gint
g_tree_node_in_order (GTreeNode     *node,
		      GTraverseFunc  traverse_func,
		      gpointer       data)
{
  if (node->left)
    {
      if (g_tree_node_in_order (node->left, traverse_func, data))
	return TRUE;
    }
  if ((*traverse_func) (node->key, node->value, data))
    return TRUE;
  if (node->right)
    {
      if (g_tree_node_in_order (node->right, traverse_func, data))
	return TRUE;
    }

  return FALSE;
}

static gint
g_tree_node_post_order (GTreeNode     *node,
			GTraverseFunc  traverse_func,
			gpointer       data)
{
  if (node->left)
    {
      if (g_tree_node_post_order (node->left, traverse_func, data))
	return TRUE;
    }
  if (node->right)
    {
      if (g_tree_node_post_order (node->right, traverse_func, data))
	return TRUE;
    }
  if ((*traverse_func) (node->key, node->value, data))
    return TRUE;

  return FALSE;
}

static gpointer
g_tree_node_search (GTreeNode   *node,
		    GSearchFunc  search_func,
		    gpointer     data)
{
  gint dir;

  if (!node)
    return NULL;

  do {
    dir = (* search_func) (node->key, data);
    if (dir == 0)
      return node->value;

    if (dir < 0)
      node = node->left;
    else if (dir > 0)
      node = node->right;
  } while (node && (dir != 0));

  return NULL;
}

static gint
g_tree_node_height (GTreeNode *node)
{
  gint left_height;
  gint right_height;

  if (node)
    {
      left_height = 0;
      right_height = 0;

      if (node->left)
	left_height = g_tree_node_height (node->left);

      if (node->right)
	right_height = g_tree_node_height (node->right);

      return MAX (left_height, right_height) + 1;
    }

  return 0;
}

static GTreeNode*
g_tree_node_rotate_left (GTreeNode *node)
{
  GTreeNode *left;
  GTreeNode *right;
  gint a_bal;
  gint b_bal;

  left = node->left;
  right = node->right;

  node->right = right->left;
  right->left = node;

  a_bal = node->balance;
  b_bal = right->balance;

  if (b_bal <= 0)
    {
      if (a_bal >= 1)
	right->balance = b_bal - 1;
      else
	right->balance = a_bal + b_bal - 2;
      node->balance = a_bal - 1;
    }
  else
    {
      if (a_bal <= b_bal)
	right->balance = a_bal - 2;
      else
	right->balance = b_bal - 1;
      node->balance = a_bal - b_bal - 1;
    }

  return right;
}

static GTreeNode*
g_tree_node_rotate_right (GTreeNode *node)
{
  GTreeNode *left;
  GTreeNode *right;
  gint a_bal;
  gint b_bal;

  left = node->left;
  right = node->right;

  node->left = left->right;
  left->right = node;

  a_bal = node->balance;
  b_bal = left->balance;

  if (b_bal <= 0)
    {
      if (b_bal > a_bal)
	left->balance = b_bal + 1;
      else
	left->balance = a_bal + 2;
      node->balance = a_bal - b_bal + 1;
    }
  else
    {
      if (a_bal <= -1)
	left->balance = b_bal + 1;
      else
	left->balance = a_bal + b_bal + 2;
      node->balance = a_bal + 1;
    }

  return left;
}

static void
g_tree_node_check (GTreeNode *node)
{
  gint left_height;
  gint right_height;
  gint balance;

  if (node)
    {
      left_height = 0;
      right_height = 0;

      if (node->left)
	left_height = g_tree_node_height (node->left);
      if (node->right)
	right_height = g_tree_node_height (node->right);

      balance = right_height - left_height;
      if (balance != node->balance)
	g_print ("g_tree_node_check: failed: %d ( %d )\n",
		 balance, node->balance);

      if (node->left)
	g_tree_node_check (node->left);
      if (node->right)
	g_tree_node_check (node->right);
    }
}


GListAllocator*
g_list_allocator_new ()
{
  GRealListAllocator* allocator = g_new (GRealListAllocator, 1);

  allocator->list_mem_chunk = NULL;
  allocator->free_list = NULL;

  return (GListAllocator*) allocator;
}

void
g_list_allocator_free (GListAllocator* fallocator)
{
  GRealListAllocator* allocator = (GRealListAllocator *) fallocator;

  if (allocator && allocator->list_mem_chunk)
    g_mem_chunk_destroy (allocator->list_mem_chunk);
  if (allocator)
    g_free (allocator);
}

GListAllocator*
g_list_set_allocator (GListAllocator* fallocator)
{
  GRealListAllocator* allocator = (GRealListAllocator *) fallocator;
  GRealListAllocator* old_allocator = current_allocator;

  if (allocator)
    current_allocator = allocator;
  else
    {
      if (!default_allocator)
	default_allocator = (GRealListAllocator*) g_list_allocator_new ();
      current_allocator = default_allocator;
    }

  if (!current_allocator->list_mem_chunk)
    current_allocator->list_mem_chunk = g_mem_chunk_new ("list mem chunk",
							 sizeof (GList),
							 1024,
							 G_ALLOC_ONLY);

  return (GListAllocator*) (old_allocator == default_allocator ? NULL : old_allocator);
}
