/*
 * dmnt.c -- Linux mount support functions for lsof
 */


/*
 * Copyright 1997 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */


#include "lsof.h"


/*
 * readmnt() - read mount table
 */

int
readmnt()
{
	char *dn = (char *)NULL;
	int err = 0;
	char *ln;
	FILE *mfp;
	struct mntent *mp;
	struct mounts *mtp;
	char *opt, *opte;
	struct stat sb;
/*
 * Open access to the mount table.
 */
	if (!(mfp = setmntent(MOUNTED, "r"))) {
	    (void) fprintf(stderr, "%s: can't access %s\n", Pn, MOUNTED);
	    return(0);
	}
/*
 * Read mount table entries.
 */
	while ((mp = getmntent(mfp))) {

	/*
	 * Interpolate a possible symbolic directory link.
	 */
	    if (dn)
		(void) free((FREE_P *)dn);
	    if (!(dn = mkstrcpy(mp->mnt_dir, (MALLOC_S *)NULL))) {
		err = 1;
		break;
	    }
	    if (!(ln = Readlink(dn))) {
		if (!Fwarn) {
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		}
		err = 2;
		continue;
	    }
	    if (ln != dn) {
		(void) free((FREE_P *)dn);
		dn = ln;
	    }
	/*
	 * Stat() the directory.
	 */
	    if (statsafely(dn, &sb)) {
		if (!Fwarn) {
		    (void) fprintf(stderr, "%s: WARNING: can't stat() ", Pn);
		    safestrprt(mp->mnt_type, stderr, 0);
		    (void) fprintf(stderr, " file system ");
		    safestrprt(mp->mnt_dir, stderr, 1);
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		}
		err = 2;
		if ((opt = strstr(mp->mnt_opts, "dev="))) {
		    (void) zeromem(&sb, sizeof(sb));
		    if ((opte = x2dev(opt + 4, (dev_t *)&sb.st_dev))) {
			sb.st_mode = S_IFDIR | 0777;
			if (!Fwarn) {
			    (void) fprintf(stderr,
				"      assuming \"%.*s\" from %s\n",
				(opte - opt), opt, MOUNTED);
			}
		    } else
			opt = (char *)NULL;
		}
		if (!opt)
		    continue;
	     }
	/*
	 * Allocate and fill a local mount structure.
	 */
	    if (!(mtp = (struct mounts *)malloc(sizeof(struct mounts)))) {
		err = 1;
		break;
	    }
	    mtp->dir = dn;
	    dn = (char *)NULL;
	    mtp->next = Mtab;
	    mtp->dev = sb.st_dev;
	    mtp->rdev = sb.st_rdev;
	    mtp->inode = sb.st_ino;
	    mtp->mode = sb.st_mode;
	    if (mp->mnt_type && strcasecmp(mp->mnt_type, "nfs") == 0) {
		HasNFS = 1;
		mtp->ty = N_NFS;
	    } else
		mtp->ty = N_REGLR;
	/*
	 * Interpolate a possible file system (mounted-on) device name link.
	 */
	    if (!(dn = mkstrcpy(mp->mnt_fsname, (MALLOC_S *)NULL))) {
		err = 1;
		break;
	    }
	    mtp->fsname = dn;
	    ln = Readlink(dn);
	    dn = (char *)NULL;
	/*
	 * Stat() the file system (mounted-on) name and add file system
	 * information to the local mount table entry.
	 */
	    if (!ln || statsafely(ln, &sb))
		sb.st_mode = 0;
	    mtp->fsnmres = ln;
	    mtp->fs_mode = sb.st_mode;
	    Mtab = mtp;
	}
	(void) endmntent(mfp);
/*
 * Clean up and handle errors.
 */
	if (dn)
	    (void) free((FREE_P *)dn);
	switch (err) {
	case 1:
	    (void) fprintf(stderr, "%s: no space for mount at ", Pn);
	    safestrprt(mp->mnt_fsname, stderr, 0);
	    (void) fprintf(stderr, " (");
	    safestrprt(mp->mnt_dir, stderr, 0);
	    (void) fprintf(stderr, ")\n");
	    return(0);
	case 2:
	    return(1);
	}
	return(1);
}
