/*
 * tnmUnixIcmp.c --
 *
 *	Make an ICMP request to a list of IP addresses. The UNIX
 *	implementation is based on ntping, a set uid root program
 *	which is used to access raw sockets. This module implements
 *	the communication between the Tnm extension and ntping.
 *
 * Copyright (c) 1993-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"

/*
 * The default filename where we will find the ntping binary. This
 * is normally overwritten in the Makefile.
 */

#ifndef NTPING
#define NTPING "/usr/local/bin/ntping"
#endif

/*
 * The following to variables hold the file descriptor used to
 * talk to the ntping process.
 */

static Tcl_Channel channel = NULL;

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
ForkNtping	_ANSI_ARGS_((Tcl_Interp *interp));


/*
 *----------------------------------------------------------------------
 *
 * ForkNtping --
 *
 *	This procedure is invoked to fork a ntping process and to set
 *	up a channel to talk to the ntping process.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new process and a new pipe is created.
 *
 *----------------------------------------------------------------------
 */

static int
ForkNtping(interp)
    Tcl_Interp *interp;
{
    int argc = 2;
    static char *argv[3] = { NULL, "-b", 0 };
    static char* ntping = NULL;

    if (! ntping) {
        ntping = getenv("TNM_NTPING");
	if (! ntping) {
	    ntping = NTPING;
	}
	ntping = ckstrdup(ntping);
    }
    argv[0] = ntping;

    channel = Tcl_OpenCommandChannel(interp, argc, argv, TCL_STDIN|TCL_STDOUT);
    if (channel == NULL) {
        if (ntping) {
	    ckfree(ntping);
	    ntping = NULL;
	}
	return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmIcmp --
 *
 *	XXX This one will change slightly. XXX
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmIcmp(interp, icmpPtr)
    Tcl_Interp *interp;
    TnmIcmpRequest *icmpPtr;
{
    Tcl_DString dst;
    int i, rc, len;
    char buffer[80];

    /*
     * Start ntping if not done yet.
     */

    if (channel == NULL) {
	if (ForkNtping(interp) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    Tcl_DStringInit(&dst);
    sprintf(buffer, "-t %d -r %d -d %d -s %d ", 
	    icmpPtr->timeout, icmpPtr->retries, icmpPtr->delay, icmpPtr->size);
    Tcl_DStringAppend(&dst, buffer, -1);

    switch (icmpPtr->type) {
    case TNM_ICMP_ECHO:
      break;
    case TNM_ICMP_MASK:
      Tcl_DStringAppend(&dst, "-mask ", -1);
      break;
    case TNM_ICMP_TIMESTAMP:
      Tcl_DStringAppend(&dst, "-timestamp ", -1);
      break;
    case TNM_ICMP_TTL:
      sprintf(buffer, "-ttl %d ", icmpPtr->ttl);
      Tcl_DStringAppend(&dst, buffer, -1);
      break;
    case TNM_ICMP_TRACE:
      sprintf(buffer, "-trace %d ", icmpPtr->ttl);
      Tcl_DStringAppend(&dst, buffer, -1);
      break;
    }

    for (i = 0; i < icmpPtr->argc; i++) {
	Tcl_DStringAppend(&dst, icmpPtr->argv[i], -1);
	Tcl_DStringAppend(&dst, " ", 1);
    }
    Tcl_DStringAppend(&dst, "\n", 1);

    rc = Tcl_Write(channel, Tcl_DStringValue(&dst), Tcl_DStringLength(&dst));
    if (rc > 0) {
	if (Tcl_Flush(channel) != TCL_OK) {
	    rc = -1;
	}
    }
    Tcl_DStringFree(&dst);

    if (rc < 0) {
	Tcl_AppendResult(interp, "ntping: ", Tcl_PosixError(interp),
			 (char *) NULL);
        return TCL_ERROR;
    }

    len = Tcl_Gets(channel, &dst);
    if (len < 0) {
        Tcl_AppendResult(interp, "reading icmp result failed: ",
                         Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_DStringResult(interp, &dst);
    return TCL_OK;
}
