/*
 * tnmMibFrozen.c --
 *
 *	Save and load MIB-Definitions in/from a frozen-format file.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/*
 * Strings are collected in a hashtable for every parsed mib. This allows
 * us to write all strings in one big chunk so that we do not need to
 * allocate every single string. The offset (== size) of the string pool
 * is maintained in poolOffset.
 */

static Tcl_HashTable *poolHashTable = NULL;
static int poolOffset = 0;

#define IDY_MAGIC	"Nase"

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
PoolInit		_ANSI_ARGS_((void));

static void
PoolDelete		_ANSI_ARGS_((void));

static void
PoolAddString		_ANSI_ARGS_((char *string));

static int
PoolGetOffset		_ANSI_ARGS_((char *string));

static void
PoolSave		_ANSI_ARGS_((FILE *fp));

static void
SaveEnum		_ANSI_ARGS_((Tnm_MibEnum *enumPtr, FILE *fp));

static void
SaveTC			_ANSI_ARGS_((Tnm_MibTC *tcPtr, int *i, FILE *fp));

static void
SaveNode		_ANSI_ARGS_((Tnm_MibNode *nodePtr, int *i, FILE *fp));

static void
CollectData		_ANSI_ARGS_((int *numEnums, int *numTcs,
				     int *numNodes, Tnm_MibNode *nodePtr));
static void
SaveData		_ANSI_ARGS_((FILE *fp, int numEnums, int numTcs, 
				     int numNodes, Tnm_MibNode *nodePtr));


/*
 *----------------------------------------------------------------------
 *
 * PoolInit --
 *
 *	This procedure initializes the hash table that is used to
 *	create a string pool. The pool is used to eliminate 
 *	duplicated strings.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
PoolInit()
{
    poolOffset = 0;
    if (poolHashTable == NULL) {
	poolHashTable = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
    }
    Tcl_InitHashTable(poolHashTable, TCL_STRING_KEYS);
}

/*
 *----------------------------------------------------------------------
 *
 * PoolDelete --
 *
 *	This procedure clears the memory used to create the string
 *	pool. Very simple and really not worth the comments. Thats
 *	why this one is getting a bit longer as usual. :-)
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
PoolDelete()
{
    if (poolHashTable) {
        Tcl_DeleteHashTable(poolHashTable);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PoolAddString --
 *
 *	This procedure adds a string to the pool if it is not yet 
 *	there. The value is initialized to mark this entry as used.
 *	The total offset is incremented to get total memory required 
 *	for the string pool. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
PoolAddString(s)
    char *s;
{
    Tcl_HashEntry *entryPtr;
    int isnew;

    if (! s) return;

    entryPtr = Tcl_CreateHashEntry(poolHashTable, s, &isnew);
    if (! isnew) {
	return;
    }
    Tcl_SetHashValue(entryPtr, 1);
    poolOffset += strlen(s) + 1;
}

/*
 *----------------------------------------------------------------------
 *
 * PoolGetOffset --
 *
 *	This procedure returns the offset to the given string in the 
 *	string pool or 0 if the string is not in the pool.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
PoolGetOffset(s)
    char *s;
{
    Tcl_HashEntry *entryPtr;

    if (! s) return 0;

    entryPtr = Tcl_FindHashEntry(poolHashTable, s);
    if (entryPtr) {
        return (int) Tcl_GetHashValue(entryPtr);
    } else {
	return 0;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PoolSave --
 *
 *	This procedure writes the string pool to the given file pointer.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
PoolSave(fp)
    FILE *fp;
{
    Tcl_HashSearch searchPtr;
    Tcl_HashEntry *entryPtr;

    /*
     * Save magic and size:
     */

    poolOffset += strlen(IDY_MAGIC) + 1;
    fwrite((char *) &poolOffset, sizeof(int), 1, fp);
    fwrite(IDY_MAGIC, 1, strlen(IDY_MAGIC) + 1, fp);

    /*
     * Save the strings in the pool:
     */

    poolOffset = strlen(IDY_MAGIC) + 1;
    entryPtr = Tcl_FirstHashEntry(poolHashTable, &searchPtr);
    while (entryPtr) {
	char *s = Tcl_GetHashKey(poolHashTable, entryPtr);
	int len = strlen(s) + 1;
	Tcl_SetHashValue(entryPtr, poolOffset);
	fwrite(s, 1, len, fp);
	poolOffset += len;
	entryPtr = Tcl_NextHashEntry(&searchPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * CollectData --
 *
 *	This procedure collects the strings (by adding them to the pool)
 *	and counts the number of enums, tcs and nodes to be saved.
 *
 * Results:
 *	Returns the number of enums, tcs and nodes.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
CollectData(numEnums, numTcs, numNodes, nodePtr)
    int *numEnums, *numTcs, *numNodes;
    Tnm_MibNode *nodePtr;
{
    Tnm_MibNode *ptr;
    Tnm_MibEnum *enumPtr;
    Tnm_MibTC *tcPtr;

    *numEnums = *numTcs = *numNodes = 0;
    for (ptr = nodePtr; ptr; (*numNodes)++, ptr = ptr->nextPtr) {
	PoolAddString(ptr->label);
	PoolAddString(ptr->parentName);
	PoolAddString(ptr->fileName);
	PoolAddString(ptr->moduleName);
	PoolAddString(ptr->index);
	if (ptr->tc) {
	    (*numTcs)++;
	    PoolAddString(ptr->tc->name);
	    PoolAddString(ptr->tc->fileName);
	    PoolAddString(ptr->tc->moduleName);
	    PoolAddString(ptr->tc->displayHint);
	    for (enumPtr = ptr->tc->enumList; 
		 enumPtr; (*numEnums)++, enumPtr = enumPtr->nextPtr) {
		PoolAddString(enumPtr->label);
	    }
	}
    }
    for (tcPtr = tnm_MibTCList; 
	 tcPtr != tnm_MibTCSaveMark; tcPtr = tcPtr->nextPtr) {
        (*numTcs)++;
	PoolAddString(tcPtr->name);
	PoolAddString(tcPtr->fileName);
	PoolAddString(tcPtr->moduleName);
	PoolAddString(tcPtr->displayHint);
	for (enumPtr = tcPtr->enumList; 
	     enumPtr; (*numEnums)++, enumPtr = enumPtr->nextPtr) {
	    PoolAddString(enumPtr->label);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SaveEnum --
 *
 *	This procedure writes a Tnm_MibEnum structure. The char*
 *	pointers are used to store the offset into the string pool.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
SaveEnum(enumPtr, fp)
    Tnm_MibEnum *enumPtr;
    FILE *fp;
{
    Tnm_MibEnum en;

    memcpy((char *) &en, (char *) enumPtr, sizeof(Tnm_MibEnum));
    en.label = (char *) PoolGetOffset(enumPtr->label);
    en.nextPtr = (Tnm_MibEnum *) (enumPtr->nextPtr ? 1 : 0);

    fwrite((char *) &en, sizeof(en), 1, fp);
}

/*
 *----------------------------------------------------------------------
 *
 * SaveTC --
 *
 *	This procedure writes a Tnm_MibTC structure. The char*
 *	pointers are used to store the offset into the string pool.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
SaveTC(tcPtr, i, fp)
    Tnm_MibTC *tcPtr;
    int *i;
    FILE *fp;
{
    Tnm_MibTC tc;

    memcpy((char *) &tc, (char *) tcPtr, sizeof(Tnm_MibTC));
    tc.name = (char *) PoolGetOffset(tcPtr->name);
    tc.fileName = (char *) PoolGetOffset(tcPtr->fileName);
    tc.moduleName = (char *) PoolGetOffset(tcPtr->moduleName);
    tc.displayHint = (char *) PoolGetOffset(tcPtr->displayHint);
    if (tcPtr->enumList) {
	Tnm_MibEnum *e;
	tc.enumList = (Tnm_MibEnum *) (*i + 1);
	for (e = tcPtr->enumList; e; (*i)++, e = e->nextPtr) ;
    }
    tc.nextPtr = (Tnm_MibTC *) (tcPtr->nextPtr ? 1 : 0);

    fwrite((char *) &tc, sizeof(Tnm_MibTC), 1, fp);
}

/*
 *----------------------------------------------------------------------
 *
 * SaveNode --
 *
 *	This procedure writes a Tnm_MibNode structure. The char*
 *	pointers are used to store the offset into the string pool.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
SaveNode(nodePtr, i, fp)
    Tnm_MibNode *nodePtr;
    int *i;
    FILE *fp;
{
    struct Tnm_MibNode no;

    memcpy((char *) &no, (char *) nodePtr, sizeof(Tnm_MibNode));
    no.label = (char *) PoolGetOffset(nodePtr->label);
    no.parentName = (char *) PoolGetOffset(nodePtr->parentName);
    no.fileName = (char *) PoolGetOffset(nodePtr->fileName);
    no.moduleName = (char *) PoolGetOffset(nodePtr->moduleName);
    no.index = (char *) PoolGetOffset(nodePtr->index);
    no.childPtr = 0;
    if (nodePtr->tc) {
	no.tc = (Tnm_MibTC *) ++(*i);
    }
    no.nextPtr = (Tnm_MibNode *) (nodePtr->nextPtr ? 1 : 0);
    
    fwrite((char *) &no, sizeof(Tnm_MibNode), 1, fp);
}

/*
 *----------------------------------------------------------------------
 *
 * SaveData --
 *
 *	This procedure writes the node list as an index file to fp.
 *
 *	int (#enum elements)
 *  		followed by #enum structs
 * 	int (#tc elements)
 *		followed by #tc structs
 *      int (#node elements)
 *		followed by n Tnm_MibNode structs.
 *
 *	string pointer are saved as offsets relative to the pool,
 *	enum pointer are saved as offset to the saved enums (+ 1)
 *	tc pointer dito.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
SaveData(fp, numEnums, numTCs, numNodes, nodePtr)
    FILE *fp;
    int numEnums, numTCs, numNodes;
    Tnm_MibNode *nodePtr;
{
    Tnm_MibNode *nPtr;
    Tnm_MibEnum *ePtr;
    Tnm_MibTC *tPtr;
    int i;
    
    /*
     * Save numEnums Tnm_MibEnum structures:
     */

    fwrite((char *) &numEnums, sizeof(int), 1, fp);
    for (nPtr = nodePtr; nPtr; nPtr = nPtr->nextPtr) {
	if (nPtr->tc) {
	    for (ePtr = nPtr->tc->enumList; ePtr; ePtr = ePtr->nextPtr) {
		SaveEnum(ePtr, fp);
	    }
	}
    }
    for (tPtr=tnm_MibTCList; tPtr != tnm_MibTCSaveMark; tPtr = tPtr->nextPtr) {
	for (ePtr = tPtr->enumList; ePtr; ePtr = ePtr->nextPtr) {
	    SaveEnum(ePtr, fp);
	}
    }

    /*
     * Save numTCs Tnm_MibTC structures:
     */

    fwrite((char *) &numTCs, sizeof(int), 1, fp);
    for (i = 0, nPtr = nodePtr; nPtr; nPtr = nPtr->nextPtr) {
	if (nPtr->tc) {
	    SaveTC(nPtr->tc, &i, fp);
	}
    }
    for (tPtr=tnm_MibTCList; tPtr != tnm_MibTCSaveMark; tPtr = tPtr->nextPtr) {
	SaveTC(tPtr, &i, fp);
    }
    
    /*
     * Save numNodes Tnm_MibNode structures:
     */

    fwrite((char *) &numNodes, sizeof(int), 1, fp);
    for (i = 0, nPtr = nodePtr; nPtr; nPtr = nPtr->nextPtr) {
	SaveNode(nPtr, &i, fp);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibWriteFrozen --
 *
 *	This procedure writes a frozen MIB file. See the description
 *	of Tnm_MibReadFrozen() for an explanation of the format.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_MibWriteFrozen(fp, nodePtr)
    FILE *fp;
    Tnm_MibNode *nodePtr;
{
    int numEnums, numTcs, numNodes; 
    PoolInit();
    CollectData(&numEnums, &numTcs, &numNodes, nodePtr);
    PoolSave(fp);
    SaveData(fp, numEnums, numTcs, numNodes, nodePtr);
    PoolDelete();
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibReadFrozen --
 *
 *	This procedure reads a frozen MIB file that was written by 
 *	Tnm_MibWriteFrozen(). The expected format is:
 *
 *	int (stringpool size)
 *		pool_size bytes
 *	int (number of enum structs)
 *		#enum structs
 *	int (number of textual conventions)
 *		#tc structs
 *	int (number of nodes)
 *		#node structs
 *
 * Results:
 *	A pointer to the MIB root node or NULL if there was an error.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tnm_MibNode*
Tnm_MibReadFrozen(fp)
    FILE *fp;
{
    Tnm_MibNode *root = NULL;
    int poolSize;
    char *pool;
    int numEnums;
    Tnm_MibEnum *enums = NULL;
    int numTcs;
    Tnm_MibTC *tcs = NULL;
    int numNodes;
    Tnm_MibNode *nodes;

    /*
     * First, read the string space: 
     */

    if (1 != fread((char *) &poolSize, sizeof(int), 1, fp)) {
	TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			   "error reading string pool size...\n");
	return NULL;
    }

    pool = ckalloc(poolSize);
    if (poolSize != fread(pool, 1, poolSize, fp)) {
	TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			   "error reading string pool...\n");
	return NULL;
    }

    /*
     * Next, read the enumerations:
     */

    if (1 != fread((char *) &numEnums, sizeof(int), 1, fp)) {
	TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			   "error reading enum counter...\n");
	return NULL;
    }

    if (numEnums > 0) {
	Tnm_MibEnum *enumPtr;
	int i;
	enums = (Tnm_MibEnum *) ckalloc(numEnums * sizeof(Tnm_MibEnum));
	if (numEnums != fread(enums, sizeof(Tnm_MibEnum), numEnums, fp)) {
	    TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			       "error reading enums...\n");
	    ckfree((char *) enums);
	    return NULL;
	}
    
	/* 
	 * Adjust the string and chain pointers:
	 */
	
	for (i = 0, enumPtr = enums; i < numEnums; i++, enumPtr++) {
	    enumPtr->label = (int) enumPtr->label + pool;
	    enumPtr->nextPtr = enumPtr->nextPtr ? enumPtr + 1 : 0;
	}
    }

    /*
     * Next, read the textual conventions: 
     */

    if (1 != fread((char *) &numTcs, sizeof(int), 1, fp)) {
	TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			   "error reading tc counter...\n");
	return NULL;
    }
    if (numTcs > 0) {
	Tnm_MibTC *tcPtr;
	int i;

	tcs = (Tnm_MibTC *) ckalloc(numTcs * sizeof(Tnm_MibTC));
	if (numTcs != fread(tcs, sizeof(Tnm_MibTC), numTcs, fp)) {
	    TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			       "error reading tcs...\n");
	    ckfree((char *) tcs);
	    return NULL;
	}
	
	/* 
	 * Adjust string and enum pointers: 
	 */

	for (i = 0, tcPtr = tcs; i < numTcs; i++, tcPtr++) {
	    tcPtr->name = (int) tcPtr->name + pool;
	    if (tcPtr->fileName) {
	        tcPtr->fileName = (int) tcPtr->fileName + pool;
	    }
	    if (tcPtr->moduleName) {
	        tcPtr->moduleName = (int) tcPtr->moduleName + pool;
	    }
	    if (tcPtr->displayHint) {
		tcPtr->displayHint = (int) tcPtr->displayHint + pool;
	    }
	    if (tcPtr->enumList) {
		tcPtr->enumList = (int) tcPtr->enumList + enums - 1;
	    }
	    if (tcPtr->name[0] != '_') {
		Tnm_MibAddTC(tcPtr);
	    }
	}	
    }

    /*
     * Next, read the MIB nodes:
     */

    if (1 != fread((char *) &numNodes, sizeof(int), 1, fp)) {
	TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			   "error reading node counter...\n");
	return NULL;
    }

    if (numNodes > 0) {
	Tnm_MibNode *ptr;
	int i;

	nodes = (Tnm_MibNode *) ckalloc(numNodes * sizeof(Tnm_MibNode));
	if (numNodes != fread(nodes, sizeof(Tnm_MibNode), numNodes, fp)) {
	    TnmWriteLogMessage(NULL, TNM_LOG_DEBUG,
			       "error reading nodes...\n");
	    ckfree((char *) nodes);
	    return NULL;
	}
	
	/*
	 * Adjust string and tc pointers:
	 */

	for (i = 0, ptr = nodes; i < numNodes; i++, ptr++) {
	    ptr->label = (int) ptr->label + pool;
	    ptr->parentName = (int) ptr->parentName + pool;
	    if (ptr->fileName) {
	        ptr->fileName = (int) ptr->fileName + pool;
	    }
	    if (ptr->moduleName) {
	        ptr->moduleName = (int) ptr->moduleName + pool;
	    }
	    if (ptr->index) {
	        ptr->index = (int) ptr->index + pool;
	    }
	    if (ptr->tc) {
	        ptr->tc = (int) ptr->tc + tcs - 1;
	    }
	    ptr->nextPtr = ptr->nextPtr ? ptr + 1 : 0;
	}
	root = nodes;
    }
    
    return root;
}
