/* filename: rlpr-client.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr-client.c,v 1.2 1997/02/22 08:56:45 meem Exp meem $
 * contents: common routines which can be used by client-side rlpr commands.
 *
 * Time-stamp: <1997/04/27 01:11:02 cdt -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996, 1997 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <stdio.h>		/* for FILE, EOF           */
#include <stdlib.h>		/* for getenv()            */
#include <assert.h>
#include <unistd.h>		/* for geteuid(), close()  */
#include <string.h>		/* for strchr(), ..        */

#include "getopt.h"

#include "rlpr-client.h"	
#include "rlpr-msg.h"		/* rlpr messaging facility */
#include "rlpr-util.h"		/* rlpr utility routines   */
#include "rlprd.h"		/* for RLPRD_DST_PORT      */
#include "rlpr-dbfile.h"	/* for gethostfromq, etc.. */
#include "rlpr-rfc1179.h"	/* rfc1179 constants       */

static rlpr_client_info rlpr_client_props = { 0, /* all args */ };

static void rlpr_client_init(void)
{
  char * tmp;
  
  tmp = getenv("PRINTER");
  rlpr_client_props.printer   = tmp ? tmp : getenv("LPDEST");

  tmp = getenv(PROXYHOST);	/* initialize default proxyhost */
  rlpr_client_props.proxyhost = tmp ? tmp : getenv(PROXYHOST2);
  
  rlpr_client_props.printhost = getenv(PRINTHOST);

  rlpr_client_props.localhost = rlpr_malloc(MAXHOSTNAMELEN);
  if (get_local_hostname(rlpr_client_props.localhost, MAXHOSTNAMELEN) == -1)
    rlpr_msg(FATAL, NO_ERRNO, "unable to resolve your local hostname!");
}

static void rlpr_client_fini(void)
{
  /* if the printer variable is set with an @ sign in it, assume it's a 
   * host@printer form and forget any previous value for printhost.
   */
  
  if (rlpr_client_props.printer && strchr(rlpr_client_props.printer, '@')) {
    rlpr_client_props.printhost = strchr(rlpr_client_props.printer, '@');
    *rlpr_client_props.printhost++ = '\0';
  }

  /* resolve remainder of printer/printhost information */
  
  if (rlpr_client_props.printhost == NULL && rlpr_client_props.printer == NULL)
    rlpr_msg(FATAL, NO_ERRNO, "no printhost or printer set");

  if (rlpr_client_props.printhost == NULL)
    if (!(rlpr_client_props.printhost = gethostfromq(rlpr_client_props.printer)))
      rlpr_msg(FATAL, NO_ERRNO, "unable to resolve hostname from printer name");

  if (rlpr_client_props.printer == NULL)
    if (!(rlpr_client_props.printer = getqfromhost(rlpr_client_props.printhost)))
      rlpr_msg(FATAL, NO_ERRNO, "unable to resolve printer name from hostname");
  
  if (rlpr_client_props.dst_port == 0) /* destination port still unset */
    rlpr_client_props.dst_port = rlpr_client_props.proxyhost ?
                                 RLPRD_DST_PORT : LPD_DST_PORT;
  
  /* now we need to go through all the options and make sure they're
   * not too large.. if they are, we may overflow the limits of some
   * poorly-coded lpd's
   */

  rlpr_msg(DEBUG, NO_ERRNO, "checking user-supplied params in client module...");

  if (rlpr_client_props.localhost)
    assert(strlen(rlpr_client_props.localhost) < MAXHOSTNAMELEN);
  if (rlpr_client_props.printhost)
    assert(strlen(rlpr_client_props.printhost) < MAX_HOST_LEN);
  if (rlpr_client_props.proxyhost)
    assert(strlen(rlpr_client_props.proxyhost) < MAXHOSTNAMELEN);
  if (rlpr_client_props.printer)
    assert(strlen(rlpr_client_props.printer)   < MAX_QUEUE_LEN);

  rlpr_msg(DEBUG, NO_ERRNO, "params passed in client module");
} 


void rlpr_client_args(int argc, char * const argv[])
{
  extern int                 opterr, optind;
  extern const char   *      rlpr_opts;
  extern const struct option rlpr_long_opts[];
  int c;                              /* the number of arguments read */

  rlpr_client_init();
  opterr = 0;                         /* do not report errors          */
  optind = 0;			      /* reset to start of argv        */

  while ((c = getopt_long(argc, argv, rlpr_opts, rlpr_long_opts, NULL)) != EOF)
    /* ignore any options we don't recognize */
    switch(c) {

    case 'P':                         /* printer queue name (FALLTHRU) */
    case 'Q':                         /* i think this is a better name */
      rlpr_client_props.printer = optarg;
      break;
    case 'N':			      /* don't bind to a port at all */
      rlpr_client_props.no_bind = 1;
      break;
    case 'H':                         /* for overriding PRINTHOST */
      rlpr_client_props.printhost = optarg;
      break;
    case 'X':                         /* for overriding PROXY  */
      rlpr_client_props.proxyhost  = optarg;
      break;
    case -201:                        /* port number to connect to */
      rlpr_client_props.dst_port   = atoi(optarg); 
      break;
    }

  rlpr_client_fini();
}


void * rlpr_client_get(enum rlpr_client_field field) {
  switch (field) {
  case RLPR_CLIENT_PRINTER:
    return rlpr_client_props.printer;
  case RLPR_CLIENT_PROXYHOST:
    return rlpr_client_props.proxyhost;
  case RLPR_CLIENT_PRINTHOST:
    return rlpr_client_props.printhost;
  case RLPR_CLIENT_LOCALHOST:
    return rlpr_client_props.localhost;
  }
  return NULL;
}


int rlpr_client_open(void)
{
  struct sockaddr_in sin_local;       /* to bind a socket to a number */
  int    sockfd;
  
  /* rlpr_client_props gets really annoying and bloats the code here
   * -- let's change to something shorter
   */

  rlpr_client_info * rcp = &rlpr_client_props;
  
  init_sockaddr(&sin_local, rcp->localhost, 0);
  
  /* obtain a socket descriptor
   *
   * if connecting to a proxy, or connecting to a SVR4 [NCR & Solaris 2] or
   * WinNT machine, we don't need to bind to a privileged port.  The user can
   * specify this with the --no-bind option.   This avoids using up all the
   * lpd source ports whin the 2MSL timeout in the TIME_WAIT tcp state.
   */

  if (rcp->proxyhost || rcp->no_bind) {
    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
      rlpr_msg(FATAL, ERRNO, "socket");

  } else {  /* we probably need root, however it's possible we can get by w/o it */
    toggle_euid();

    if (geteuid() != 0)
      rlpr_msg(WARNING, NO_ERRNO, "cannot bind to privileged port - continuing...");

    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
      rlpr_msg(FATAL, ERRNO, "socket");

    if (geteuid() == 0)
      if (bind_try_range(&sin_local, LO_LPD_DST_PORT, HI_LPD_DST_PORT, sockfd) < 0)
	rlpr_msg(FATAL, ERRNO, 
		 "bind to ports %hi-%hi", LO_LPD_DST_PORT, HI_LPD_DST_PORT);

    toggle_euid();                    /* lose root */
  }

  { /* we either want to connect to the proxy or the actual lpd. we
     * will try to connect to the machine specified by the RLPR_PROXYHOST
     * variable if it's set
     */

    struct sockaddr_in sin_to;        /* who we actually connect() to */
    struct sockaddr_in sin_local;     /* who we are... */
    int    i = sizeof(sin_local);

    init_sockaddr(&sin_to, rcp->proxyhost ? rcp->proxyhost : rcp->printhost,
		  rcp->dst_port);

    if (connect(sockfd, (struct sockaddr *) &sin_to, sizeof(sin_to)) < 0)
      rlpr_msg(FATAL, ERRNO, "connect to port %hi", ntohs(sin_to.sin_port));

    if (getsockname(sockfd, (struct sockaddr *) &sin_local, &i) < 0)
      rlpr_msg(FATAL, ERRNO, "getsockname on local endpoint");

    rcp->src_port = ntohs(sin_local.sin_port);

    rlpr_msg(DEBUG, NO_ERRNO, "connected %s:%d -> %s:%d", rcp->localhost, 
	     rcp->src_port,   rcp->proxyhost ?
	     rcp->proxyhost : rcp->printhost, rcp->dst_port);
    
    if (rcp->proxyhost) {       /* if proxy, send out final destination */
      rlpr_msg(DEBUG, NO_ERRNO, "directing proxy to %s..", rcp->printhost);
      safe_writen(sockfd, rcp->printhost, strlen(rcp->printhost));
      safe_writen(sockfd, "\n", 1);
    }
  }

  return sockfd;
}


void rlpr_client_close(int sockfd)
{
  if (close(sockfd) < 0)
    rlpr_msg(FATAL, ERRNO, "close on socket connection");
  rlpr_msg(DEBUG, NO_ERRNO, "connection closed");
}
