/*								-*- C++ -*-
 * $Id: WIN_menu.cpp,v 1.2 1997-01-15 14:59:07+01 mho Exp $
 *
 * Purpose: simple menu class
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "WIN_menu.h"
#endif

#define  Uses_XtIntrinsic
#define  Uses_wxGDI
#define  Uses_wxMenu
#include "wx.h"
#define  Uses_ShellWidget
#define  Uses_MenuWidget
#include <widgets.h>

// forward declaration
char *wxGetLabelAndKey(char *label, char **clean_label, char **clean_key);

//-----------------------------------------------------------------------------
// constructor and destructor
//-----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(wxMenu, wxItem)

wxMenu::wxMenu(char *_title, wxFunction _func)
{
    __type = wxTYPE_MENU;

    callback = _func;
    top = topdummy = title = last = 0;
    owner = 0;
    // if a title is associated with a menu, it may not be removed
    if (_title) {
	Append(-1, _title);
	title = top;
	((menu_item*)title)->type = MENU_TEXT;
	AppendSeparator();
	AppendSeparator();
    } else {
	Append(-1, NULL); // to have something if associated to another menu
	topdummy = top;
    }

    WXGC_IGNORE(owner);
}

wxMenu::~wxMenu(void)
{
    menu_item *item = (menu_item*)top;

    while (item) {
	menu_item *temp = item;
	item = item->next;
	if (temp->label != wxEmptyString)
	    XtFree(temp->label);		 // delete label
	if (temp->contents)			 // has submenu?
	    delete ((wxMenu*)(temp->user_data)); // delete wxMenu
	XtFree((char*)temp);			 // delete menu_item
    }
}

//-----------------------------------------------------------------------------
// create and popup menu, will be destroyed by wxMenuCallback
//-----------------------------------------------------------------------------

Bool PopupMenu(wxWindow *parent, wxMenu *menu, int x, int y)
{
    return menu->PopupMenu(parent, x, y);
}

Bool wxMenu::PopupMenu(wxWindow *parent, int root_x, int root_y)
{
    FWidget() = XtVaCreatePopupShell
	("popup", overrideShellWidgetClass, 
	 (parent ? parent->HWidget() : wxAPP_TOPLEVEL), NULL);
    HWidget() = XtVaCreateManagedWidget
	("menu", xfwfMenuWidgetClass, FWidget(),
	 XtNmenu,       top,
	 XtNfont,       font.GetInternalFont(),
	 XtNforeground, fg.GetPixel(wxAPP_COLOURMAP),
	 XtNbackground, bg.GetPixel(wxAPP_COLOURMAP),
	 NULL);
    XtRealizeWidget(FWidget());
    XtAddCallback(HWidget(), XtNonSelect, wxMenu::EventCallback, this);

    XfwfPopupMenuAtPos(HWidget(), root_x, root_y);

    return TRUE;
}

//-----------------------------------------------------------------------------
// add items to menu
//-----------------------------------------------------------------------------

void wxMenu::Append(int id, char *label, char *help, Bool checkable)
{
    menu_item *item = 0;
    // create new menu item or use topdummy
    if (topdummy) {
	item = (menu_item*)topdummy;
	if (item->label != wxEmptyString)
	    XtFree(item->label);
	topdummy = 0;
    } else {
	item = XtNew(menu_item);
	// chain or initialize menu_item list
	if (last) {
	    menu_item *prev = (menu_item*)last;
	    prev->next = item;
	    last = (wxMenuItem*)item;
	} else {
	    top = last = (wxMenuItem*)item;
	}
    }
    // initialize menu_item
    item->underline = wxGetLabelAndKey(label, &item->label, &item->key_binding);
    item->help_text = help;
    item->ID        = id; 
    item->enabled   = TRUE;
    item->set       = FALSE;
    item->contents  = NULL;
    item->next      = NULL;
    item->user_data = NULL;
    item->type      = checkable ? MENU_TOGGLE : MENU_BUTTON;
}

void wxMenu::Append(int id, char *label, wxMenu *submenu, char *help)
{
    if (submenu->owner)
	return;

    // do the same thing as if appending a "button"
    Append(id, label, help, FALSE);
    // change data for submenu
    menu_item *item = (menu_item*)last;
    item->type      = MENU_CASCADE;
    item->contents  = (menu_item*)submenu->top;
    item->user_data = (void*)submenu;

    submenu->owner  = (wxMenuItem **)&item->contents;
}

void wxMenu::AppendSeparator(void)
{
    // do the same thing as if appending a "button"
    Append(-1, NULL, NULL, FALSE);
    // change data for separator
    menu_item *item = (menu_item*)last;
    item->type      = MENU_SEPARATOR;
}

void wxMenu::Delete(int id)
{
    menu_item *found, *prev;

    prev = NULL;
    for (found = (menu_item*)top; found; found = found->next) {
	if (found->ID == id)
	    break;
	prev = found;
    }
    if (found) {
	if (!prev) {
	    top = (wxMenuItem*)found->next;
	    if (!top) {
		last = 0;
		Append(-1, NULL); /* Reinstate topdummy */
	    }
	    if (owner)
		*owner = top;
	} else {
	    prev->next = found->next;
	    if (!found->next)
		last = (wxMenuItem*)prev;
	}
	if (found->label != wxEmptyString)
	    XtFree(found->label);		 // delete label
	if (found->contents)			 // If there's a submenu, let it go.
	    ((wxMenu *)found->user_data)->owner = NULL;
	XtFree((char*)found);			 // delete menu_item
    }
}

//-----------------------------------------------------------------------------
// modify items
//-----------------------------------------------------------------------------

void wxMenu::Check(int id, Bool flag)
{
    menu_item *found = (menu_item*)FindItemForId(id);
    if (found)
	found->set = flag;
}

Bool wxMenu::Checked(int id)
{
    menu_item *found = (menu_item*)FindItemForId(id);
    if (found)
	return found->set;
    return FALSE;
}

void wxMenu::Enable(int id, Bool flag)
{
    menu_item *found = (menu_item*)FindItemForId(id);
    if (found)
	found->enabled = flag;
}

char *wxMenu::GetHelpString(int id)
{
    menu_item *found = (menu_item*)FindItemForId(id);
    if (found)
	return found->help_text;
    return NULL;
}

char *wxMenu::GetLabel(int id)
{
    menu_item *found = (menu_item*)FindItemForId(id);
    if (found)
	return found->label;
    return NULL;
}

char *wxMenu::GetTitle(void)
{
    if (title)
	return ((menu_item*)title)->label;
    return NULL;
}

void wxMenu::SetHelpString(int id, char *help)
{
    menu_item *found = (menu_item*)FindItemForId(id);
    if (found)
	found->help_text = help;
}

void wxMenu::SetLabel(int id, char *label)
{
    menu_item *found = (menu_item*)FindItemForId(id);
    if (found) {
	if (found->label != wxEmptyString) XtFree(found->label);
	found->underline = wxGetLabelAndKey(label, &found->label, &found->key_binding);
    }
}

void wxMenu::SetTitle(char *label)
{
    if (title) {
	menu_item *item = (menu_item*)title;
	if (item->label != wxEmptyString) XtFree(item->label);
	item->underline = wxGetLabelAndKey(label, &item->label, &item->key_binding);
    }
}

//-----------------------------------------------------------------------------
// find items by ID or by label
//-----------------------------------------------------------------------------

int wxMenu::FindItem(char *itemstring)
{
    char *label, *key;
    int  answer = -1;

    wxGetLabelAndKey(itemstring, &label, &key);
    for (menu_item *item = (menu_item*)top; item; item=item->next) {
	if (!strcmp(label, item->label)) { // label found
	    answer = item->ID;
	    break; // found
	}
	if (item->contents) // has submenu => search in submenu
	    if ((answer = ((wxMenu*)item->user_data)->FindItem(label)) > -1)
		break; // found
    }
    if (label != wxEmptyString) XtFree(label); // key is part of label
    return answer;
}

int wxMenu::FindItem(char* itemstring, wxMenu **submenu)
{
    char*   label;
    char*   key;
    int     answer = -1;

    *submenu = NULL;

    wxGetLabelAndKey(itemstring, &label, &key);
    for (menu_item *item = (menu_item*)top; item; item=item->next) {
	if (!strcmp(label, item->label)) { // label found
	    answer   = item->ID;
	    *submenu = (wxMenu*)item->user_data;
	    break;
	}
    }
    if (label != wxEmptyString) XtFree(label); // key is part of label
    return answer;
}

wxMenuItem *wxMenu::FindItemForId(int id, wxMenu **req_menu)
{
    menu_item *answer=NULL;

    for (menu_item *item = (menu_item*)top; item; item=item->next) {
	if (id == item->ID) { // id found
	    answer = item;
	    break; // found
	}
	if (item->contents) // has submenu => search in submenu
	    if ((answer =
		 (menu_item*)((wxMenu*)item->user_data)->FindItemForId(id)))
		break; // found
    }
    if (req_menu)
	*req_menu = (wxMenu*)answer->user_data;
    return ((wxMenuItem*)answer);
}

//-----------------------------------------------------------------------------
// callback for wxMenu::PopupMenu
//-----------------------------------------------------------------------------

void wxMenu::EventCallback(Widget WXUNUSED(w), XtPointer dclient, XtPointer dcall)
{
    wxMenu    *menu  = (wxMenu*)dclient;
    menu_item *item  = (menu_item*)dcall;

    if (item) {
	if (item->type == MENU_TOGGLE)
	    item->set = (!item->set);

	wxCommandEvent event(wxEVENT_TYPE_MENU_COMMAND);

	event.eventObject = menu;
	event.commandInt = item->ID;

	// call callback function
	if (menu->callback)
	    (void)(*(menu->callback))(*menu, event);
    }

    // destroy widgets
    XtDestroyWidget(menu->FWidget());
    menu->FWidget() = menu->HWidget() = 0;
}

//-----------------------------------------------------------------------------
// Utility function to separate label and key from a menu string
//-----------------------------------------------------------------------------

char *wxGetLabelAndKey(char *label, char **clean_label, char **clean_key)
{
    char *amp, *key, *underline = NULL;

    // make private copy (if no label reference to wxEmptyString)
    *clean_label = (char*)XtNewString(label);
    if ( !(*clean_label) )
	*clean_label = (char*)wxEmptyString;
    // erase ampersands
    if ((amp = strchr(*clean_label, '&'))) {
	memmove(amp, amp+1, strlen(amp+1));
	(*clean_label)[strlen(*clean_label)-1] = '\0';
	underline = amp;
    }
    // is there a key binding? -> split
    if ((key=strchr(*clean_label, '\t')))
	*key++ ='\0';
    // point to key binding in private copy
    *clean_key = key;

    return underline; // the character to be underlined
}

