// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  KWDialog.h

  The Koala Widget API (kWAPI)

  Stephane Rehel
  August 4 1996
*/

#ifndef __KWDialog_h
#define __KWDialog_h

#ifndef __kw_h
#include "kw.h"
#endif

#ifndef __SIList_h
#include "tools/SIList.h"
#endif

#ifndef __OString_h
#include "tools/OString.h"
#endif

#ifndef __IRectangle_h
#include "tools/IRectangle.h"
#endif

class VDialog;
class VWidget;
class SystemWindow;
class MLKeyboardHandler;
class MLEventGrabber;
class SystemWindow;
class VWindow;
class VButton;
class VCartouche;
class VScrollingCartouche;
class MLFont;

typedef unsigned int KWDialogID;

/////////////////////////////////////////////////////////////////////////////

class KWDialog: public KWCallbackClass
{
  friend VDialog; // for handleCancelValidateKey()
  friend VWidget; // for cancelID and validateID
  friend class KWDialogAllocator;
  friend class VScrollingDialogs;
  friend class VScrollingCartouche;

public:
  // exit code from Validate/Cancel dialogs
  enum
    {
    CANCEL= 0,
    VALIDATE= 1
    };

protected:
  KWDialogID dialogID;

  MLKeyboardHandler* previousKH;
  MLEventGrabber* previousEG;

  VCartouche* cartouche;
  VScrollingCartouche* scartouche;
  IBOOL scrolling;

  VDialog* dialog;

  KWCallbackClass* callbackClass;

  int exitCode;
  IBOOL quitMe;

  IBOOL validateCancel;
  int validateID, cancelID;

  struct KWGroup
    {
    int size;
    KWidget* widgets;
    };
  int nGroups;
  KWGroup* groups;
  OString name;
  IBOOL create_unmapped;

public:
  KWDialog();
  virtual ~KWDialog();

  void setScrollingCartouche();

  // parent may be NULL for root
  IBOOL          create( SystemWindow* parent, int x, int y,
                         const unsigned char* dialogMap,
                         KWCallbackClass* _callbackClass,
                         IBOOL map_it = ITRUE );

  // parent may be NULL for root
  IBOOL          create( SystemWindow* parent,
                         const unsigned char* dialogMap,
                         KWCallbackClass* _callbackClass,
                         IBOOL map_it = ITRUE );

  void           drawDialog    ();
  int            run           ();
  void           quit          ( int _exitCode = 0 );

  const OString& getName       () const { return name; }
  KWDialogID     getID         () const { return dialogID; }
  void           map           ( KWidget widget );
  void           unmap         ( KWidget widget );
  IBOOL          mapped        ( KWidget widget );

  void           setText       ( KWidget widget, const OString& text );
  void           setChecked    ( KWidget widget,  IBOOL yes = ITRUE );
  void           setAvailable  ( KWidget widget, IBOOL yes = ITRUE );
  void           focus         ( KWidget widget );
  void           setValue      ( KWidget widget, double value );
  void           setInteger    ( KWidget widget, int i );
  IBOOL          setListText   ( KWidget widget, int i, const OString& text );
  IBOOL          appendListText( KWidget widget, const OString& text );
  IBOOL          insertListText( KWidget widget, int i, const OString& text );
  IBOOL          deleteListText( KWidget widget, int i );
  void           clearList     ( KWidget widget );
  void*          getListUserData( KWidget widget, int i );
  int            getListUserDataIndex( KWidget widget, void* user_data );

  int            appendFolder    ( KWidget widget,
                                   int folder_id,
                                   const OString& text,
                                   void* user_data = 0 );
  IBOOL          appendFolderItem( KWidget widget,
                                   int folder_id,
                                   const OString& text,
                                   void* user_data = 0 );
  int            insertFolder    ( KWidget widget,
                                   int folder_id,
                                   int i,
                                   const OString& text,
                                   void* user_data = 0 );
  IBOOL          insertFolderItem( KWidget widget,
                                   int folder_id,
                                   int i,
                                   const OString& text,
                                   void* user_data = 0 );
  IBOOL          setFolderData( KWidget widget,
                                int folder_id,
                                int i, // i==0 for folder text
                                const OString& text,
                                void* user_data = 0 );
  void*          getFolderUserData( KWidget widget, int folder_id, int i );
  int            getFolderUserDataIndex( KWidget widget,
                                         int folder_id,
                                         void* user_data );
  int            getFolderID( KWidget widget, int i );
  int            getFolderID( KWidget widget, int folder_id, int i );
  int            getFolderLength( KWidget widget, int folder_id );
  IBOOL          deleteFolder( KWidget widget, int folder_id );

  void           openFolder      ( KWidget widget,
                                   int folder_id,
                                   IBOOL yes = ITRUE );
  void           closeFolder     ( KWidget widget,
                                   int folder_id );

  void           setValueEditRanges( KWidget widget,
                                     double minValue,
                                     double maxValue,
                                     int nDigits );
  void           setRangeBarRanges( KWidget widget,
                                    double minValue, double maxValue );
  void           repaintArea( KWidget widget );

  IRectangle     getRectangle  ( KWidget widget );
  const OString& getText       ( KWidget widget );
  IBOOL          getChecked    ( KWidget widget );
  IBOOL          getAvailable  ( KWidget widget );
  KWidget        getRadioChecked( KWidget radioFromGroup );
  double         getValue      ( KWidget widget );
  int            getInteger    ( KWidget widget );
  int            getListLength ( KWidget widget );
  const OString&
                 getListText   ( KWidget widget, int i );
  int            getListTextIndex( KWidget widget, const OString& text );

  void           activateCard  ( KWidget card );
  KWidget        getActiveCard ();

  static KWDialog* find( KWDialogID id );

  VWidget* getWidget( KWidget widget );

private:
  void validateDialog();
  void removeWidget( VWidget* w );
  KWGroup* getRadioGroup( KWidget widget );

  IBOOL handleValidateCancelKey( Key key );

  IBOOL kwDialogCallback( KWEvent& event );

public:
  static OString** splitString( MLFont* font,
                                const OString& string,
                                int max_width,
                                int nLines );
};

/////////////////////////////////////////////////////////////////////////////

// in KWError.C
// title + 4-line-message + cancel button
void KWError( const OString& title, const OString& message );

// in KWError.C
// title + 4-line-message + cancel button
void KWErrorf( const OString& title, const char* format, ... );

// in KWWarningOkCancel.C
// title + 4-line-message + cancel button
// return IBOOL is Ok pressed
IBOOL KWWarningOkCancel( const OString& message );
IBOOL KWWarningOkCancelf( const char* format, ... );

/////////////////////////////////////////////////////////////////////////////

#endif // ifndef __KWDialog_h
