%{
/*
Copyright (c) 1997 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: idlscan.l,v 1.8 1997/11/21 23:36:21 janssen Exp $
*/
/* This code was contributed by Martin von Lwis */
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include <ctype.h>
#include <iluptype.h>
#include "iluidl.h"
#include "idlparser.h"

struct include_file{
  YY_BUFFER_STATE state;
  char *file;
  int lineno;
};
#define MAX_INCLUDE_DEPTH 20
struct include_file include_stack[MAX_INCLUDE_DEPTH];
static int include_stack_ptr;

static void OpenInclude(char *statement,int brackets);
static void PopInclude();
static void inc_lineno();
static void name_setline(IDLName); 

char *current_cpp;
void CppAppend(char*);
void CppProcess();
void CppDefine(string s);
string CppDefinitionOf(string s);
boolean IfDefed(string s,int len);
boolean CppReplace(string s);

static int ifnesting=0,iffailed=0;
static int cpp_continue_state;

typedef struct cpp_define_s{
	string name;
	string value;
}*cpp_define;

list CppDefines;

#define YY_USER_INIT init_scanner();
void init_scanner();

static IDLDefinition pragma_id(char*);
static IDLDefinition pragma_version(char*);
static IDLDefinition pragma_prefix(char*);

static IDLValue new_string(char*);
static IDLValue new_float(char*,char*,int,char*,int,char*);

%}
/* CPP is exclusive: in CPP mode, normal rules are not considered */
%x CPP
%x CPPGET
%x COMMENT
%x FAILED
%option noyywrap
/* nested comments and such using yy_push_state */
%option stack
ID	[[:alpha:]_][[:alnum:]_]*
DEC	[[:digit:]]+
HEX	0x[[:xdigit:]]+
EXP	[Ee][+-]?[[:digit:]]+
COM	\/\*
%%
{COM}		yy_push_state(COMMENT);
<COMMENT>{
	\*\/	yy_pop_state();
	.	/*skip*/
	\n	inc_lineno();
	<<EOF>>	fprintf(stderr,"unterminated comment\n");
}

^[ \t]*#[ \t]*	BEGIN(CPPGET);
<CPPGET>{
	[^\\\n]*	CppAppend(yytext);
	\\\n	inc_lineno();
	\\	CppAppend(yytext);
	\n	inc_lineno();CppProcess();
}
<CPP>{
	{COM}	yy_push_state(COMMENT);
	^include[ \t]+\<[^>]+>[ \t]*	{
		OpenInclude(yytext,1);
	}
	^include[ \t]+\"[^\"]+\"[ \t]*	{
		OpenInclude(yytext,0);
	}
	^ifdef[ \t]+{ID}	{
					if(!IfDefed(yytext,6)){
						cpp_continue_state=FAILED;
						iffailed++;
					}
					ifnesting++;
				}
	^ifndef[ \t]+{ID}	{	if(IfDefed(yytext,7)){
						cpp_continue_state=FAILED;
						iffailed++;
					}
					ifnesting++;
				}
	^else[ \t]*	cpp_continue_state=FAILED;iffailed++;
	^endif[ \t]*	{
				ifnesting--;
				if(ifnesting<0){
					fprintf(stderr,"unbalanced #endif\n");
					exit(1);
				}
			}
	^define[ \t]+{ID}.*	{
					CppDefine(yytext);
			}
	^pragma[ \t]+ID[ \t].*	{
				idllval.definition=pragma_id(yytext);
				if(idllval.definition)
				  return PRAGMA_ID;
				else
				  exit(1);
			}
	^pragma[ \t]+version[ \t].* {
				idllval.definition=pragma_version(yytext);
				if(idllval.definition)
				  return PRAGMA_VERSION;
				else
				  exit(1);
			}
	^pragma[ \t]+prefix[ \t].*  {
				idllval.definition=pragma_prefix(yytext);
				if(idllval.definition)
				  return PRAGMA_PREFIX;
				else
				  exit(1);
			}
	.+	fprintf(stderr,"Unknown preprocessor directive %s\n",yytext);
	<<EOF>>		PopInclude();BEGIN(cpp_continue_state);
}

<FAILED>{
	^#[ \t]*ifdef	ifnesting++;iffailed++;
	^#[ \t]*else	if(--iffailed)iffailed++;else{BEGIN(0);}
	^#[ \t]*endif	ifnesting--;iffailed--;if(!iffailed){BEGIN(0);}
	.		/*skip*/
	\n		inc_lineno();
}

module		return MODULE;
interface	return INTERFACE;
const		return CONST;
typedef		return TYPEDEF;
readonly	return READONLY;
attribute	return ATTRIBUTE;
exception	return EXCEPTION;
oneway		return ONEWAY;
inout		return INOUT;
in		return IN;
out		return OUT;
raises		return RAISES;
context		return CONTEXT;

float		return FLOAT_T;
double		return DOUBLE_T;
long		return LONG_T;
short		return SHORT_T;
unsigned	return UNSIGNED_T;
char		return CHAR_T;
wchar		return WCHAR_T; /* wide characters as in orbos/96-05-04 */
boolean		return BOOLEAN_T;
void		return VOID_T;
octet		return OCTET_T;
any		return ANY_T;
string		return STRING_T;
wstring		return WSTRING_T;
struct		return STRUCT;
union		return UNION;
switch		return SWITCH;
case		return CASE;
default		return DEFAULT;
enum		return ENUM;
sequence	return SEQUENCE;
fixed		return FIXED;
native		return NATIVE; /* native support as in orbos/97-04-04 */
Object		return OBJECT_T;

TRUE		{	idllval.value=new_value();
			idllval.value->tag=idl_boolean;
			idllval.value->u.BOOL=TRUE;
			return BOOL_TRUE;
		}
FALSE		{	idllval.value=new_value();
			idllval.value->tag=idl_boolean;
			idllval.value->u.BOOL=FALSE;
			return BOOL_FALSE;
		}

{DEC}		{	idllval.value=new_value();
			idllval.value->tag=idl_int;
			idllval.value->u.INT=strtol(yytext,0,10);
			return INTEGER_L;
		}
{HEX}		{	idllval.value=new_value();
			idllval.value->tag=idl_int;
			idllval.value->u.INT=strtol(yytext,0,16);
			return INTEGER_L;
		}
[[:digit:]]+{EXP}	{char *e;
			e=strchr(yytext,'e');if(!e)e=strchr(yytext,'E');
			idllval.value=new_float(yytext,yytext,e-yytext,
						0,0,e+1);
			return FLOAT_L;
			}
[[:digit:]]+\.[[:digit:]]*{EXP}?	{
			char *f,*e;
			f=strchr(yytext,'.');
			e=strchr(f+1,'e');
			if(!e)e=strchr(f+1,'E');
			idllval.value=new_float(yytext,yytext,f-yytext,
						f+1,e?e-f-1:strlen(f+1),e);
			return FLOAT_L;
			}
[[:digit:]]*\.[[:digit:]]+{EXP}?	{
			char *f,*e;
			f=strchr(yytext,'.');
			e=strchr(f+1,'e');
			if(!e)e=strchr(f+1,'E');
			idllval.value=new_float(yytext,yytext,f-yytext,
						f+1,e?e-f-1:strlen(f+1),e);
			return FLOAT_L;
			}
\'[[:print:]]\'			{	idllval.value=new_value();
					idllval.value->u.CHAR=yytext[1];
					idllval.value->tag=idl_char;
					return CHAR_L;
				}
\'\\[0-7]{1,3}\'		{idllval.value=new_value();
				 idllval.value->u.CHAR=strtol(yytext+1,NULL,8);
				 idllval.value->tag=idl_char;
				 return CHAR_L;
				}
\'\\x[[:xdigit:]]{1,2}\'	{idllval.value=new_value();
				 idllval.value->u.CHAR=strtol(yytext+2,NULL,16);
				 idllval.value->tag=idl_char;
				return CHAR_L;
				}
\'\\.\'				{	char c;
					switch(yytext[2]){
					case 'n': c='\n';break;
					case 't': c='\t';break;
					case 'v': c='\v';break;
					case 'b': c='\b';break;
					case 'r': c='\r';break;	
					case 'f': c='\f';break;
					case 'a': c='\a';break;
					case '\\': c='\\';break;
					case '\?': c='\?';break;
					case '\'': c='\'';break;
					case '\"': c='\"';break;
					default: c=yytext[2];break;
					}
					idllval.value=new_value();
					idllval.value->u.CHAR=c;
					idllval.value->tag=idl_char;
					return CHAR_L;
				}
\"([^\"\n\t\v\b\r\f\a]|\\\")*\"	{
					idllval.value=new_string(yytext);
					return STRING_L;
				}
{ID}		{ if(!CppReplace(yytext)){
			idllval.name=new_name();
			idllval.name->name=ilu_strdup(yytext);
			name_setline(idllval.name);
			return IDENT;
		  }
		}

::		return SCOPE;
>>		return RSHIFT;
\<\<		return LSHIFT;
[-|^&+*/%]	return yytext[0]; /* arithmetic operators */;
[(){},:;=<>]	return yytext[0];
\[		return yytext[0];
\]		return yytext[0];

[ \t]		;
\n		inc_lineno();
.		return yytext[0];

\/\/.*$		; /* C++ comment */

<INITIAL><<EOF>>	{
		if(--include_stack_ptr==0)
    				yyterminate();
                 else{
		   yy_switch_to_buffer(include_stack[include_stack_ptr].state);
		   BEGIN(0);
		 }
		}
%%

void init_scanner()
{
	CppDefines=iluparser_new_list();
}

/* yy_scan_bytes fr defines */

void OpenInclude(char *statement,int brackets)
{
  char *file, *path = 0;
  char *start,*stop;
  if(brackets){
    start=strchr(statement,'<');
    assert(start);
    stop=strchr(start+1,'>');
    assert(stop);
  }else{
    start=strchr(statement,'"');
    assert(start);
    stop=strchr(start+1,'"');
    assert(stop);
  }
  *stop='\0';
  file=ilu_strdup(start+1);
  assert(include_stack_ptr<MAX_INCLUDE_DEPTH);
#if defined(WIN32)
#define DIR_SEPARATOR_CHAR '\\'
  if (file[0] != '\\' && file[1] != ':') {
#elif defined(macintosh)
#define DIR_SEPARATOR_CHAR ':'
  if (*file == DIR_SEPARATOR_CHAR) {
#else
#define DIR_SEPARATOR_CHAR '/'
  if (file[0] != DIR_SEPARATOR_CHAR) {
#endif
     /* if file is not an absolute path name then we should first
      * look in the directory of the file that contained the include
      * directive.
      */
     char *current_file, *last_separator;
     int i;
     for (i = include_stack_ptr - 1; include_stack[i].file == 0; --i)
	/*EMPTY*/;
     current_file = include_stack[i].file;
     last_separator = strrchr(current_file, DIR_SEPARATOR_CHAR);
     if (last_separator) {
	*last_separator = 0;
	path = iluparser_FindFileInDir(current_file, file);
	*last_separator = DIR_SEPARATOR_CHAR;
     }
  }
  if (!path)
     path = iluparser_FindFileInIncludes(file);
  if (!path) {
     fprintf (stderr, "file \"%s\" not found in include path.\n", file);
     exit(1);
  }
  include_stack[include_stack_ptr].state=YY_CURRENT_BUFFER;
  include_stack[include_stack_ptr].file=path;
  include_stack[include_stack_ptr].lineno=1;
  include_stack_ptr++;
  idlin=fopen(path,"r");
  if(!idlin){perror(path);exit(1);}
  yy_switch_to_buffer(yy_create_buffer(idlin,YY_BUF_SIZE));
  /* new file is scanned like top-level */
  BEGIN(0);
}

void PopInclude()
{
  --include_stack_ptr;
  assert(include_stack_ptr>=0);
  yy_switch_to_buffer(include_stack[include_stack_ptr].state);
  if (include_stack[include_stack_ptr].file)
    iluparser_Free(include_stack[include_stack_ptr].file);
}

void CppAppend(char* s)
{
  char *n;
  if(current_cpp){
    n=iluparser_Malloc(strlen(current_cpp)+strlen(s)+1);
    strcpy(n,current_cpp);
    strcat(n,s);
    iluparser_Free(current_cpp);
    current_cpp=n;
  }else{
    current_cpp=ilu_strdup(s);
  }
}

void CppProcess()
{
  /* FIXME: could avoid copy, should delete buffer, string vs file */
  include_stack[include_stack_ptr].state=YY_CURRENT_BUFFER;
  include_stack[include_stack_ptr].file=0;
  include_stack_ptr++;
  /* this already does the switch_to_buffer */
  yy_scan_string(current_cpp);
  iluparser_Free(current_cpp);
  current_cpp=0;
  /* where to go on after we're done */
  cpp_continue_state=INITIAL;
  BEGIN(CPP);
}

void CppDefine(string s)
{
  string name=ilu_strdup(s+7); /* skip "define " */
  string end;
  cpp_define def;
  while(isspace(*name))name++;
  name=ilu_strdup(name);
  for(end=name+1;*end && isidlidentchar(*end);end++);
  *end='\0';
  end++;
  def=iluparser_Malloc(sizeof(struct cpp_define_s));
  def->name=name;
  def->value=end;
  list_insert(CppDefines,def);
}

static boolean define_find(cpp_define item,string s)
{
  return strcmp(item->name,s)==0;
}

string CppDefinitionOf(string s)
{
  cpp_define def=list_find(CppDefines,(iluparser_FindProc)define_find,s);
  return def ? def->value : NULL;
}

boolean IfDefed(string s,int len)
{
  string name=s+len; /* skip "ifdef " */
  string end;
  boolean result;
  while(!isidlidentchar(*name))name++;
  /* Fixme: do ifdef lookup with name and length */
  name=ilu_strdup(name);
  for(end=name+1;*end && isidlidentchar(*end);end++);
  *end='\0';
  result = CppDefinitionOf(name)!=NULL;
  iluparser_Free(name);
  return result;
}

boolean CppReplace(string s)
{
  string r=CppDefinitionOf(s);
  if(!r)return 0;
  if(include_stack_ptr==MAX_INCLUDE_DEPTH){
    fprintf(stderr,"Recursive macro expansion for %s\n",s);
    exit(1);
  }
  include_stack[include_stack_ptr].state=YY_CURRENT_BUFFER;
  include_stack[include_stack_ptr].file=0;
  include_stack_ptr++;
  yy_scan_string(r);
  return 1;
}

static IDLDefinition
name_error(char *str)
{
  fprintf(stderr,"No name given in %s\n",str);
  return 0;
}

static IDLDefinition
value_error(char *str)
{
  fprintf(stderr,"Invalid value in %s\n",str);
  return 0;
}

static IDLName
pragma_name(char* start,char* end,IDLName scope)
{
  IDLName result;
  char *it;
  for(it=start;it!=end;it++)
    if(*it==':')break;
  if(*it==':'){
    if(it[1]!=':' || it[2]==':'){
      fprintf(stderr,"Invalid scope in %*s\n",end-start,start);
      return 0;
    }
  }
  result=new_name();
  result->name=iluparser_Malloc(it-start+1);
  memcpy(result->name,start,it-start);
  result->name[it-start]='\0';
  result->scope=scope;
  name_setline(result);
  result->no_ordering=1;
  if(it!=end)
    return pragma_name(it+2,end,result);
  return result;
}

static IDLDefinition 
pragma_id(char* str)
{
  IDLDefinition result;
  char *name_start,*name_end,*value_start,*value_end;
  char *it;
  it=strstr(str,"ID");
  it+=2;
  while(*it && isspace(*it))it++;
  if(!*it)
    return name_error(str);
  name_start=it;
  while(*it && (isalnum(*it)||(*it=='_')||(*it==':')))it++;
  name_end=it;
  if(name_end==name_start)
    return name_error(str);
  while(*it && isspace(*it))it++;
  if(*it!='"')
    return value_error(str);
  it++;
  value_start=it;
  while(*it && *it!='"')it++;
  if(*it!='"' || it==value_start)
    return value_error(str);
  value_end=it;
  result=new_definition();
  result->tag=PRAGMA_IDtag;
  result->name=pragma_name(name_start,name_end,0);

  result->u.pragma=iluparser_Malloc(value_end-value_start+1);
  memcpy(result->u.pragma,value_start,value_end-value_start);
  result->u.pragma[value_end-value_start]='\0';
  return result;
}

static IDLDefinition 
pragma_version(char* str)
{
  IDLDefinition result=new_definition();
  char *name_start,*name_end,*value_start,*value_end;
  char *it;
  it=strstr(str,"version");
  it+=7;
  while(*it && isspace(*it))it++;
  if(!*it)
    return name_error(str);
  name_start=it;
  while(*it && (isalnum(*it)||(*it=='_')||(*it==':')))it++;
  name_end=it;
  if(name_end==name_start)
    return name_error(str);
  while(*it && isspace(*it))it++;
  if(!isdigit(*it))
    return value_error(str);
  value_start=it;
  while(*it && (isdigit(*it)||*it=='.'))it++;
  if(it==value_start)
    return value_error(str);
  value_end=it;
  result->tag=PRAGMA_VERSIONtag;
  result->name=pragma_name(name_start,name_end,0);

  result->u.pragma=iluparser_Malloc(value_end-value_start+1);
  memcpy(result->u.pragma,value_start,value_end-value_start);
  result->u.pragma[value_end-value_start]='\0';
  return result;
}
  
static IDLDefinition 
pragma_prefix(char* str)
{
  IDLDefinition result;
  char *value_start,*value_end;
  char *it;
  it=strstr(str,"prefix");
  it+=6;
  while(*it && isspace(*it))it++;
  if(*it!='"')
    return value_error(str);
  it++;
  value_start=it;
  while(*it && *it!='"')it++;
  if(*it!='"' || it==value_start)
    return value_error(str);
  value_end=it;
  result=new_definition();
  result->tag=PRAGMA_PREFIXtag;
  result->u.pragma=iluparser_Malloc(value_end-value_start+1);
  memcpy(result->u.pragma,value_start,value_end-value_start);
  result->u.pragma[value_end-value_start]='\0';
  return result;
}

void
idlsetinitialfile(char* file)
{
  include_stack[0].file=file;
  include_stack[0].lineno=1;
  include_stack_ptr=1;
}

static int
current_file_index()
{
  int i;
  for(i=include_stack_ptr-1;i>=0;i--)
    if(include_stack[i].file)
      return i;
  fprintf(stderr,"Error:Include stack underflow\n");
  exit(1);
}

static void
inc_lineno()
{
  int i=current_file_index();
  include_stack[i].lineno++;
}

static void
name_setline(IDLName n)
{
  int i=current_file_index();
  n->file=include_stack[i].file;
  n->line=include_stack[i].lineno;
}

char*
idlcurrentfile()
{
  int i=current_file_index();
  return include_stack[i].file;
}

int
idlcurrentline()
{
  int i=current_file_index();
  return include_stack[i].lineno;
}

static IDLValue
new_string(char *s)
{
  char *s1;
  IDLValue result=new_value();
  result->tag=idl_string;
  result->u.string=s1=ilu_strdup(s+1);
  /* skip first ", check for last " */
  while(*(++s+1)){
    if(*s=='\\')
      switch(*++s){
      case 'n': *s1='\n';break;
      case 't': *s1='\t';break;
      case 'v': *s1='\v';break;
      case 'b': *s1='\b';break;
      case 'r': *s1='\r';break;	
      case 'f': *s1='\f';break;
      case 'a': *s1='\a';break;
      case '\\': *s1='\\';break;
      case '\?': *s1='\?';break;
      case '\'': *s1='\'';break;
      case '\"': *s1='\"';break;
      default: *s1=*s;break;
      }
    else
      *s1=*s;
    s1++;
  }
  *s1='\0';
  return result;
}

static IDLValue
new_float(char *text,char *i,int ilen,char* f,int flen,char *e)
{
	IDLValue ret=new_value();
	ret->tag=idl_float;
	ret->u.FLOAT.val=strtod(text,0);
	ret->u.FLOAT.sign=0;
	/* 0 is not allowed for the integer part in a constantvalue_s */
	ret->u.FLOAT.integer=ilen?strndup(i,ilen):"0";
	ret->u.FLOAT.fraction=flen?strndup(f,flen):0;
	/* e is like "E+10" */
	ret->u.FLOAT.exponent=e?strtol(e+1,0,10):0;
	return ret;
}
