/*
 * Log module for OpenGate
 * 
 * Copyright (c) Tony Genovese 2000
 * Copyright (c) Egoboo Ltd. 2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is RAS Log
 *
 * The Initial Developer of the Original Code is Tony Genovese
 *
 * $Log: Log.cxx,v $
 * Revision 1.6  2000/08/04 14:13:39  aunitt
 * Updated to new Q931 naming conventions in 1.1beta3 of OpenH323
 *
 * Revision 1.5  2000/07/02 19:51:03  aunitt
 * Fix to logging of invalid IP addresses, thanks to Nick Golovin.
 *
 * Revision 1.4  2000/05/23 10:12:58  aunitt
 * Fixed windows compile error.
 *
 * Revision 1.3  2000/05/19 12:09:48  aunitt
 * Added logging of tunneled H.245 messages.
 *
 * Revision 1.2  2000/05/15 19:04:07  aunitt
 * Added IP address to logging output.
 * Improved Q.931 logging by adding decode of UU field.
 *
 * Revision 1.1  2000/05/12 14:12:00  aunitt
 * Reworked log module.
 *
 *
 */

//////////////////////////////////////////////////////////////////////
//
// Log.cpp: implementation of the OpengateLog class.
//
//////////////////////////////////////////////////////////////////////

#include <iostream.h>
#include <ptlib.h>
#if (_MSC_VER >= 1200)
#include <q931.h>               // To stop "ostream" ambiguous symbol error
#include <h245.h>
#endif
#include "AddrUtils.h"
#include "Log.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

OpengateLog::OpengateLog() : Cfg("Log")
{

	PTime now;
	PString DateStamp = now.AsString("hh:mm:ss dd/MM/yyyy");
	PString StartupMsg = "Starting logging: ";
	LogFilename = Cfg.GetString("File", "OpengateLog.txt");
		
	// Check to see if logging has been requested
	// If none then do not open log file

	LogMutex.Wait();
	if (!LogFile.IsOpen()) {
		LogFile.Open(LogFilename, PFile::ReadWrite);
		LogFile.SetPosition(0, PFile::End);
	}
	LogFile << StartupMsg;
	LogFile << DateStamp;
	LogFile.WriteLine("");  //Cheating really want to flush buffer...
	LogMutex.Signal();

}

OpengateLog::~OpengateLog()
{
	LogFile.Close();
}

void OpengateLog::WriteHeader()
{
    PTime now;
    PString DateStamp = now.AsString("hh:mm:ss dd/MM/yyyy");
    LogMutex.Wait();
   	LogFile << DateStamp << " ";
}

void OpengateLog::WriteHeader( Direction Dir )
{
    WriteHeader();
    if ( Dir == Sending )
        LogFile << "Sent ";
    else
        LogFile << "Recv ";
}

void OpengateLog::WriteHeader( Direction                    Dir,
                               const PIPSocket::Address &   Address,
                               WORD                         Port
                             )
{
    WriteHeader( Dir );
    LogFile << Address << ":" << Port << "\t";
}

void OpengateLog::WriteHeader( Direction                        Dir,
                               const H225_TransportAddress &    Addr
                             )
{
    if ( ( Addr.GetTag() == H225_TransportAddress::e_ipAddress ) &&
           Addr.IsValid()
       )
    {
        const H225_TransportAddress_ipAddress & IPAddr = Addr;
        PIPSocket::Address  Address;
        WORD                Port;
        AddrUtils::ConvertToIPAddress( IPAddr, Address, Port );
        WriteHeader( Dir, Address, Port );
    }
    else
    {
        WriteHeader( Dir );
    }
}

void OpengateLog::WriteTrailer()
{
    LogFile.WriteLine(""); //adding CRLF and insuring buffer is flushed.
	LogMutex.Signal();
}


//Log PString style messages to the Log file

void OpengateLog::LogNote(const PString &note)
{
  if ( GetLogLevel() > OpengateLog::None )
  {
    WriteHeader();
    LogFile << note;
    WriteTrailer();
  }
}

// Log H225_RasMessage style messages to the RAS log file

void OpengateLog::LogRasMsg(const H225_RasMessage &         msg,
                            Direction                       Dir,
                            const H225_TransportAddress &   Addr
                           )
{
  if ( GetLogLevel() > OpengateLog::None )
  {
    WriteHeader(Dir,Addr);
    if ( GetLogLevel() == OpengateLog::Info )
        LogFile << "H.225 " << msg.GetTagName();
    else
    {
        LogFile.WriteLine("H.225");
        LogFile << setprecision(2) << msg;
    }
    WriteTrailer();
  }
}

void OpengateLog::LogQ931Msg(const Q931 &               msg,
                             Direction                  Dir,
	                         const PIPSocket::Address & Address,
	                         WORD                       Port
        	                )
{
  if ( GetLogLevel() > OpengateLog::None )
  {
    WriteHeader(Dir,Address,Port);
    if ( GetLogLevel() == OpengateLog::Info )
        LogFile << "Q.931 " << msg.GetMessageTypeName();
    else
    {
        LogFile.WriteLine("Q.931");
        LogFile << setprecision(2) << msg;
        if ( msg.HasIE(Q931::UserUserIE) )
        {
            LogFile.WriteLine("");
            LogFile.WriteLine("UserUserField:");
            H225_H323_UserInformation UUField;
            PPER_Stream Strm = msg.GetIE(Q931::UserUserIE);
            if ( UUField.Decode(Strm) )
            {
                LogFile << setprecision(2) << UUField;

                H225_H323_UU_PDU & UU_PDU = UUField.m_h323_uu_pdu;
                if ( UU_PDU.HasOptionalField( H225_H323_UU_PDU::e_h245Control ) )
                {
                    for( PINDEX i=0; i < UU_PDU.m_h245Control.GetSize(); ++i )
                    {
                        PPER_Stream H245Strm = UU_PDU.m_h245Control[i].GetValue();
       	                H245_MultimediaSystemControlMessage H245Mesg;
                        H245Mesg.Decode( H245Strm );
                        LogFile.WriteLine("");
                        LogFile.WriteLine("Tunneled H.245:");
                        LogFile << setprecision(2) << H245Mesg;
                    }
                }
            }
        }
    }
    WriteTrailer();
  }
}

void OpengateLog::LogH245Msg(const H245_MultimediaSystemControlMessage &msg,
                             Direction                                  Dir,
   	                         const PIPSocket::Address &                 Address,
	                         WORD                                       Port
                            )
{
  if ( GetLogLevel() > OpengateLog::None )
  {
    WriteHeader(Dir,Address,Port);
    if ( GetLogLevel() == OpengateLog::Info )
    {
        LogFile << "H.245 " << msg.GetTagName();
        switch( msg.GetTag() )
        {
            case H245_MultimediaSystemControlMessage::e_request     :
                {
                    const H245_RequestMessage & Request = msg;
                    LogFile << " " << Request.GetTagName();
                }
                break;
            case H245_MultimediaSystemControlMessage::e_response    :
                {
                    const H245_ResponseMessage & Response = msg;
                    LogFile << " " << Response.GetTagName();
                }
                break;
            case H245_MultimediaSystemControlMessage::e_command     :
                {
                    const H245_CommandMessage & Command = msg;
                    LogFile << " " << Command.GetTagName();
                }
                break;
            case H245_MultimediaSystemControlMessage::e_indication  :
                {
                    const H245_IndicationMessage & Indication = msg;
                    LogFile << " " << Indication.GetTagName();
                }
                break;
            default : // Do nothing
                break;
        }
    }
    else
    {
        LogFile.WriteLine("H.245");
        LogFile << setprecision(2) << msg;
    }
    WriteTrailer();
  }
}