// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;
using System.Text;

using Internal.Runtime;
using Internal.Text;
using Internal.TypeSystem;

using Debug = System.Diagnostics.Debug;

namespace ILCompiler.DependencyAnalysis
{
    /// <summary>
    /// Represents a subset of <see cref="EETypeNode"/> that is used to describe GC static field regions for
    /// types. It only fills out enough pieces of the EEType structure so that the GC can operate on it. Runtime should
    /// never see these.
    /// </summary>
    internal class GCStaticEETypeNode : ObjectNode, ISymbolDefinitionNode
    {
        private GCPointerMap _gcMap;
        private TargetDetails _target;

        public GCStaticEETypeNode(TargetDetails target, GCPointerMap gcMap)
        {
            _gcMap = gcMap;
            _target = target;
        }

        protected override string GetName(NodeFactory factory) => this.GetMangledName(factory.NameMangler);

        public override ObjectNodeSection Section
        {
            get
            {
                if (_target.IsWindows)
                    return ObjectNodeSection.ReadOnlyDataSection;
                else
                    return ObjectNodeSection.DataSection;
            }
        }

        public override bool StaticDependenciesAreComputed => true;

        public void AppendMangledName(NameMangler nameMangler, Utf8StringBuilder sb)
        {
            sb.Append("__GCStaticEEType_").Append(_gcMap.ToString());
        }

        int ISymbolDefinitionNode.Offset
        {
            get
            {
                int numSeries = _gcMap.NumSeries;
                return numSeries > 0 ? ((numSeries * 2) + 1) * _target.PointerSize : 0;
            }
        }

        int ISymbolNode.Offset => 0;

        public override bool IsShareable => true;

        public override ObjectData GetData(NodeFactory factory, bool relocsOnly)
        {
            ObjectDataBuilder dataBuilder = new ObjectDataBuilder(factory, relocsOnly);
            dataBuilder.RequireInitialPointerAlignment();
            dataBuilder.AddSymbol(this);

            // +1 for SyncBlock (in CoreRT static size already includes EEType)
            Debug.Assert(factory.Target.Abi == TargetAbi.CoreRT);
            int totalSize = (_gcMap.Size + 1) * _target.PointerSize;

            // We only need to check for containsPointers because ThreadStatics are always allocated
            // on the GC heap (no matter what "HasGCStaticBase" says).
            // If that ever changes, we can assume "true" and switch this to an assert.

            bool containsPointers = _gcMap.NumSeries > 0;
            if (containsPointers)
            {
                GCDescEncoder.EncodeStandardGCDesc(ref dataBuilder, _gcMap, totalSize, 0);
            }

            Debug.Assert(dataBuilder.CountBytes == ((ISymbolDefinitionNode)this).Offset);

            dataBuilder.EmitShort(0); // ComponentSize is always 0

            short flags = 0;
            if (containsPointers)
                flags |= (short)EETypeFlags.HasPointersFlag;

            dataBuilder.EmitShort(flags);

            totalSize = Math.Max(totalSize, _target.PointerSize * 3); // minimum GC eetype size is 3 pointers
            dataBuilder.EmitInt(totalSize);

            // Related type: System.Object. This allows storing an instance of this type in an array of objects.
            dataBuilder.EmitPointerReloc(factory.NecessaryTypeSymbol(factory.TypeSystemContext.GetWellKnownType(WellKnownType.Object)));

            return dataBuilder.ToObjectData();
        }

        protected internal override int ClassCode => 1304929125;

        protected internal override int CompareToImpl(SortableDependencyNode other, CompilerComparer comparer)
        {
            return _gcMap.CompareTo(((GCStaticEETypeNode)other)._gcMap);
        }
    }
}
