/*
 *  ACM - Landing gear module
 *  Copyright (C) 2008  Umberto Salsi
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

/**
 * Aircraft gear dynamics, including steering, aerodynamic drag, drag over the
 * terrain.
 * @file
 */

#ifndef _gear_h
#define _gear_h

#include "pm.h"

#ifdef gear_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * Allocates the gear state data structure c->gear. The gear is initialized
 * up and locked, zero steering angle, no brakes applied.
 */
EXTERN void gear_allocate(craft * c);

/**
 * Release the gear state data structure c->gear and sets that field to
 * null. Does nothing if already NULL.
 */
EXTERN void gear_free(craft *c);

/**
 * Set initial handle and gear state, no animation, no sound.
 * @param c
 */
EXTERN void gear_up(craft *c);

/**
 * Set initial handle and gear state, no animation, no sound.
 * @param c
 */
EXTERN void gear_down(craft *c);

EXTERN _BOOL gear_isHandleDown(craft * c);

/** Move gear handle with gear animation and sound. */
EXTERN void gear_handle_up(craft *c);
/** Move gear handle with gear animation and sound. */
EXTERN void gear_handle_down(craft *c);
/** Move gear handle with gear animation and sound. */
EXTERN void gear_handle_toggle(craft *c);

/**
 * Returns the nose landing gear position: 0 = up, 1 = moving, 2 = down.
 */
EXTERN int gear_nosePosition(craft * c);

/**
 * Returns the left landing gear position: 0 = up, 1 = moving, 2 = down.
 */
EXTERN int gear_leftPosition(craft * c);

/**
 * Returns the right landing gear position: 0 = up, 1 = moving, 2 = down.
 */
EXTERN int gear_rightPosition(craft * c);

/**
 * Returns true if all wheel are in ground contact.
 */
EXTERN _BOOL gear_allWheelsGroundContact(craft * c);

/**
 * Returns true if at least one wheel is in ground contact.
 */
EXTERN _BOOL gear_someWheelGroundContact(craft * c);

/**
 * Returns true if the nose/tail wheel is in ground contact.
 */
EXTERN _BOOL gear_noseWheelGroundContact(craft * c);

/**
 * Returns true if any of the two main wheels is in ground contact.
 */
EXTERN _BOOL gear_mainWheelsGroundContact(craft * c);

EXTERN _BOOL gear_brakesEngaged(craft * c);
EXTERN void gear_brakesEngage(craft * c);
EXTERN void gear_brakesDisengage(craft * c);

/**
 * Return the rest altitude of the CM over the terrain (ft) and the rest
 * pitch (RAD) at which the plane can be gently deployed over the runway. The
 * spring compression are set in order to sustain the total gross weight
 * empty+fuel+payload.
 * 
 * ATTENTION. Fuel quantity and payload must be already set before calling
 * this function.
 */
EXTERN void gear_ground_altitude_pitch(craft *c, double *altitude, double *pitch);

/**
 * Returns the gear contribute to the drag factor.
 * @return Gear contribute to the drag factor.
 */
EXTERN double gear_get_drag(craft * c);

/**
 * Return the total force and moment due to the friction of the wheels on
 * the ground, components in aircraft XYZ.  Return NULL on success, or the
 * reason of the crash.
 */
EXTERN char * gear_ground_dynamics(craft * c, VPoint *gearF, VPoint *gearM);

#undef EXTERN
#endif
