/***************************************************************************
 $RCSfile: ctmemorycard.h,v $
                             -------------------
    cvs         : $Id: ctmemorycard.h,v 1.5 2003/04/24 01:43:29 aquamaniac Exp $
    begin       : Wed Aug 29 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 Changes

 */


#ifndef CTMEMORYCARD_H
#define CTMEMORYCARD_H

class CTMemoryCard;

#include <string>

#include <chipcard/ctcard.h>
#include <chipcard/cterror.h>


/**
 * This class provides access to memory chip cards.
 * It allows you to do some very basic things with memory cards, like
 * reading from them or writing to them.
 * @author Martin Preuss<martin@libchipcard.de>
 * @ingroup gcards
 * @short Basic class for memory cards
 */
class CHIPCARD_API CTMemoryCard : public CTCard {
private:
  string _cmdSelectFile;
  string _cmdSelectMF;

public:
  /** @name Constructors/Destructor
   *
   * Methods to retrieve the private members of this class.
   */
  //@{
  /**
   *
   */
  CTMemoryCard(const CTCard &c);

  virtual ~CTMemoryCard();
  //@}

  /** @name Opening and closing operations
   *
   * Methods to connect and disconnect the card. Most other methods
   * only work if the card is open.
   */
  //@{
  /**
   * When this method is called normally the card is already opened by
   * means of @ref openCard(). However, it is likely that openCard() has
   * been called assuming this card is only a CTCard() object, nothing
   * special. You may then call THIS method here on all known card classes
   * to check for the type of this card.
   * This method must assume that any file on the card is already selected,
   * so it is in most cases a good idea to select the MF as the first action
   * of this method.
   * If the executing class finds that the card inserted is not of a type
   * it can handle then it MUST return an error.
   */
  virtual CTError reopenCard();
  //@}

  /** @name Informational methods
   *
   * These methods tell you about the type and the status of the card.
   */
  //@{
  /**
   * This method returns a short name of the class which handles this card.
   * A HBCI card for example returns "HBCICard". So you can use this method
   * to check for the type of the card.
   */
  virtual string cardType();

  /**
   * This method returns a comma separated list of all card types this card
   * inherits including the type of this card, e.g. a HBCI card would
   * return "CTProcessorCard, HBCICard". If you overload this method in your
   * own class (you SHOULD) you simply have to call this method of the
   * class it directly inherits. That class will then do the same, so that
   * there is a list upon return.
   */
  virtual string cardTypes();
  //@}

  /** @name Commands
   *
   * These methods allow you to send a command to the card. Of course the
   * card needs to be already open.
   */
  //@{
  /**
   * Lets the card verify the pin. Some cards may need this to allow
   * writing to it.
   * @author Martin Preuss<martin@libchipcard.de>
   * @param pin hmm ;-)
   * @return CTError object that holds the result (call isOk() to see if
   * there was an error)
   */
  CTError verifyPin(const string &pin);

  /**
   * Changes the pin on the card.
   * @author Martin Preuss<martin@libchipcard.de>
   * @param oldpin the pin
   * @param newpin new pin
   * @return CTError object that holds the result (call isOk() to see if
   * there was an error)
   */
  CTError changePin(const string &oldpin,
		    const string &newpin);

  /**
   * Selects a file.
   * Please not that some cards only allow this method to be called
   * with a fid=0x3f00.
   * @author Martin Preuss<martin@libchipcard.de>
   * @param fid the id of the file to select (0x3f00 for MF)
   * @return CTError object that holds the result (call isOk() to see if
   * there was an error)
   */
  CTError selectFile(unsigned short fid);

  /**
   * Selects the master file.
   * On success the FCI (File Control Information) is returned.
   * The FCI may contain some usefull information about access rights
   * you have for the file.
   * For a closer information about FCI see the file "geldkarte.pdf"
   * which may be obtained from the Chaos Computer Club (CCC).
   * Please note that some cards do not support this command. In that
   * case you have to use the basic method @ref CTCard::selectFile().
   * @author Martin Preuss<martin@libchipcard.de>
   * @return CTError object that holds the result (call isOk() to see if
   * there was an error)
   * @param fcp a reference to a string to receive the FCI
   */
  CTError selectMF(string &fcp);

  //@}

};




#endif



