#****************************************************************************
#*                             WaveEnum.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements a window for editing enum sets for traces...
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#****************************************************************************


namespace eval WaveEnum {

    variable d_configSpec {
        { -sdbr            "" }
    }
}

#********************************************************************
#* WaveEnum
#********************************************************************
proc WaveEnum::WaveEnum {path args} {
    global   CallbackTypes
    variable d_configSpec

    set path [frame $path]

    array set $path {_dummy _dummy}
    upvar #0 $path data

    foreach cs $d_configSpec {
        set data([lindex $cs 0]) [lindex $cs 1]
    }

    eval BaseClass::configure $path $args

    set data(w:panedWin) [PanedWindow $path.panedWin]

    ConstructSigListPane $path
    ConstructEnumListPane $path

    pack $data(w:panedWin) -expand yes -fill both

    set data(curr_sig) ""
    set data(curr_sel) ""
    set data(modified) 0
    set data(change_sel) 0

    set data(sdbr_update_cb) [callback add \
        $CallbackTypes(SDBR_SIGLIST_UPDATE) \
        $data(-sdbr) [list WaveEnum::SdbrUpdateCB $path]]

    bind $path <Destroy> "WaveEnum::Destroy $path"

    UpdateSigList $path

    return $path
}

#********************************************************************
#* Destroy
#********************************************************************
proc WaveEnum::Destroy {w} {
    upvar #0 $w data

    callback destroy $data(sdbr_update_cb)
}

#********************************************************************
#* SdbrUpdateCB
#********************************************************************
proc WaveEnum::SdbrUpdateCB args {
    set w [lindex $args 0]
    upvar #0 $w data

    UpdateSigList $w
}

#********************************************************************
#* ConstructSigListPane
#********************************************************************
proc WaveEnum::ConstructSigListPane {w} {
    upvar #0 $w data

    set data(w:SigPane) [$data(w:panedWin) add -weight 1]
   
    set scrollc [ScrollContainer::ScrollContainer $data(w:SigPane).scrollc]
    set frame [$scrollc subwidget frame]

    set data(w:SigList) [listbox $frame.sigList -exportselection false]

    set vsb [$scrollc subwidget vsb]
    set hsb [$scrollc subwidget hsb]

    $data(w:SigList) configure -yscrollcommand "$vsb set"
    $data(w:SigList) configure -xscrollcommand "$hsb set"

    $vsb configure -command "$data(w:SigList) yview"
    $hsb configure -command "$data(w:SigList) xview"

    bind $data(w:SigList) "<<ListboxSelect>>" [list WaveEnum::SigListBrowse $w]

    pack $data(w:SigList) -expand yes -fill both
    pack $scrollc -expand yes -fill both
}

#********************************************************************
#* EnumListReturnBind
#********************************************************************
proc WaveEnum::EnumListReturnBind {w} {
    upvar #0 $w data

    set sel_l [split [$data(w:EnumList) curselection] {,}]

    set newr [lindex $sel_l 0]
    set newc [lindex $sel_l 1]

    if {$newr == "" || $newc == ""} {
        return -code break
    }

    $data(w:EnumList) selection clear "$newr,$newc"

    incr newr

    set rows [$data(w:EnumList) cget -rows]
    if {$newr >= $rows} {
        $data(w:EnumList) configure -rows [expr $newr + 1]
    }

    $data(w:EnumList) activate "$newr,$newc"
    $data(w:EnumList) selection set "$newr,$newc"
    $data(w:EnumList) see "$newr,$newc"

    return -code break
}


#********************************************************************
#* EnumListTabBind
#********************************************************************
proc WaveEnum::EnumListTabBind {w} {
    upvar #0 $w data

    set sel_l [split [$data(w:EnumList) curselection] {,}]

    set newr [lindex $sel_l 0]
    set newc [lindex $sel_l 1]

    if {$newr == "" || $newc == ""} {
        return -code break
    }

    $data(w:EnumList) selection clear "$newr,$newc"

    if {$newc == 1} {
        set newr [expr [lindex $sel_l 0] + 1]
        set newc 0
    } else { 
        set newr [lindex $sel_l 0]
        set newc 1
    }

    set rows [$data(w:EnumList) cget -rows]
    if {$newr >= $rows} {
        $data(w:EnumList) configure -rows [expr $newr + 1]
    }

    $data(w:EnumList) activate "$newr,$newc"
    $data(w:EnumList) selection set "$newr,$newc"
    $data(w:EnumList) see "$newr,$newc"
    return -code break
}

#********************************************************************
#* ValidateEnumCommand
#********************************************************************
proc WaveEnum::ValidateEnumCommand {w c s} {
    upvar #0 $w data
    set ret true

    if {$c == 0} {
        #**** Expect an integer
        if {[regexp {^[0-9]*$} $s] != 1} {
            bell
            set ret false
        } else {
            set data(modified) true
        }
    } else {
        if {[regexp {^[^ \t]*$} $s] != 1} {
            bell
            set ret false
        } else {
            set data(modified) true
        }
    }

    return $ret
}

#********************************************************************
#* ConstructEnumListPane
#********************************************************************
proc WaveEnum::ConstructEnumListPane {w} {
    upvar #0 $w data
    global $w.data_arr

    set data(w:ValList) [$data(w:panedWin) add -weight 1]

    set data(w:EnumList) [iviTkTable $data(w:ValList).enumList \
        -cols 2 -rows 0 \
        -cache 1 \
        -titlerows 0 \
        -titlecols 0 \
        -colstretchmode last \
        -resizeborders col   \
        -validate yes \
        -validatecommand "WaveEnum::ValidateEnumCommand $w %c %S"]

    bind $data(w:EnumList) "<Return>" [list WaveEnum::EnumListReturnBind $w]
    bind $data(w:EnumList) "<Tab>" [list WaveEnum::EnumListTabBind $w]

    set data(w:EnumListVsb) [scrollbar $data(w:ValList).vsb -orient vertical \
        -command "$data(w:EnumList) yview"]
    $data(w:EnumList) configure -yscrollcommand "$data(w:EnumListVsb) set"
   
    pack $data(w:EnumListVsb) -side right -fill y
    pack $data(w:EnumList) -expand yes -fill both
}

#********************************************************************
#* UpdateSigEnum
#********************************************************************
proc WaveEnum::UpdateSigEnum {w sig} {
    upvar #0 $w data

    set rows [$data(w:EnumList) cget -rows]

    set val_list [$data(w:EnumList) get "0,0" "$rows,1"]

    eval $sig set_enum $val_list
}

#********************************************************************
#*  Apply
#********************************************************************
proc WaveEnum::Apply {w} {
    upvar #0 $w data

    if {$data(curr_sig) != ""} {
        UpdateSigEnum $w $data(curr_sig)
    }
}

#********************************************************************
#* SigListBrowse
#********************************************************************
proc WaveEnum::SigListBrowse {w} {
    upvar #0 $w data

    if {$data(curr_sig) != ""} {
        UpdateSigEnum $w $data(curr_sig)
    }

    set csel  [$data(w:SigList) curselection]
    set sig [lindex [$data(-sdbr) list] $csel]
    set data(curr_sig) $sig
    set data(modified) 0
    set data(change_sel) 1

    UpdateEnumList $w $sig

    $data(w:SigList) selection set $csel
}

#********************************************************************
#* UpdateSigList
#********************************************************************
proc WaveEnum::UpdateSigList {w} {
    upvar #0 $w data

    if {$data(-sdbr) == ""} {
        return
    }

    if {$data(change_sel) == 1} {
        set data(change_sel) 0
        return
    }

    set sel_idx [$data(w:SigList) curselection]
    set sel_name ""
    if {$sel_idx != ""} {
        set sel_name [$data(w:SigList) get $sel_idx]
    }

    set signals [$data(-sdbr) list]

    $data(w:SigList) delete 0 end

    set data(curr_sel) ""
    for {set i 0} {$i < [llength $signals]} {incr i} {
        set sig [lindex $signals $i]

        if {[$sig cget -separator]} {
        } else {
            $data(w:SigList) insert end [$sig cget -sig_fullname]
            if {[$sig cget -has_enum]} {
                $data(w:SigList) itemconfigure end -foreground red
            }

            if {$sel_name == [$sig cget -sig_fullname]} {
                $data(w:SigList) selection set $sel_idx
                $data(w:SigList) see $sel_idx
                set data(curr_sel) $sel_idx
            }
        }
    }
}

#********************************************************************
#* UpdateEnumList
#********************************************************************
proc WaveEnum::UpdateEnumList {w sig} {
    upvar #0 $w data
    upvar #0 $w.data_arr data_arr

    $data(w:EnumList) clear all

    if {[$sig cget -has_enum] == 1} {
        set enum [$sig get_enum]

        set y 0
        $data(w:EnumList) config -cols 2 -rows [expr [llength $enum] / 2]

        foreach {ival sval} $enum {
            $data(w:EnumList) set $y,0 $ival
            $data(w:EnumList) set $y,1 $sval

            incr y
        }
    } else {
        $data(w:EnumList) config -rows 2
    }
}

