/****************************************************************************
 *                         IviRemoteSimPlugin.cc
 *                         
 * Author: Matthew Ballance
 * Desc:   Implements the base of the plugin that runs within the remote
 *         simulator
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "IviRemoteSimPlugin.h"
#include "RemoteAppClientConnection.h"
#include "IviRemoteCmdServer.h"
#include "IviRemoteProto.h"
#include "RemoteAppConnectionListener.h"
#include "iviSplitString.h"
#include "DesignDB.h"
#include "IviRemoteSDBClnt.h"
#include "ShmDFIO.h"
#include "DFIO.h"
#include "DFIOMgr.h"
#include "vpi_user.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#undef  DEBUG_IVI_REMOTE_SIM_PLUGIN

#define FP stderr

#ifdef DEBUG_IVI_REMOTE_SIM_PLUGIN
#define DBG_MSG(x) fprintf x ; fflush(FP)
#else
#define DBG_MSG(x)
#endif

/********************************************************************/
/** \class IviRemotePluginCmdChannel
 *  \brief Implements a command-channel listener
 ********************************************************************/
class IviRemotePluginCmdChannel : public RemoteAppConnectionListener {

    public:
        IviRemotePluginCmdChannel(IviRemoteSimPlugin *plugin) :
            RemoteAppConnectionListener(REMOTE_CMD_IDX)
        {
            d_plugin = plugin;
        }

        virtual void Receive(Uint32 idx, Uint32 len, Uchar *data)
        {
            Uint32 cmd = data[0]|(data[1]<<8)|(data[2]<<16)|(data[3]<<24);

            d_plugin->d_recv = true;
            d_plugin->d_cmd  = cmd;
            d_plugin->d_len  = len;
            memcpy(d_plugin->d_data, data, len);
        }

    private:
        IviRemoteSimPlugin        *d_plugin;
};

/********************************************************************
 * IviPluginConnection
 ********************************************************************/
class IviPluginConnection : public RemoteAppClientConnection {
    public:

        IviPluginConnection(Tcl_Interp *interp, 
                IviRemoteSimPlugin *plugin, Int32 port) :
            RemoteAppClientConnection(interp, port), d_plugin(plugin) { ; }

        virtual ~IviPluginConnection() { ; }

        virtual void HandleDisconnect() 
        {
            d_plugin->HandleDisconnect();
            fprintf(stderr, "NOTE: IVI plugin simulator exiting...\n");
            exit(0);
        }

        IviRemoteSimPlugin          *d_plugin;
};

/********************************************************************
 * IviRemoteSimPlugin()
 ********************************************************************/
IviRemoteSimPlugin::IviRemoteSimPlugin(Tcl_Interp *interp)
{
    Int32                port = 0;
    Uchar                buf[16];
    Uint32               idx=0;
    int                  ret;

    DBG_MSG((FP, "Connecting to port \"%s\"\n", getenv("IVI_SRV_PORT")));

    d_interp = interp;


    if (getenv("IVI_SRV_PORT")) {
        port = strtoul(getenv("IVI_SRV_PORT"), 0, 10);
    } else {
        fprintf(stderr, "ERROR: IVI VPI plugin couldn't locate IVI's port "
                "- IVI_SRV_PORT unset\n");
        _exit(1);
    }

    if (port) {
        d_conn = new IviPluginConnection(d_interp, this, port);

        d_cmdServer = new RemoteAppTclCmdListener(d_interp);
        d_conn->AddListener(new IviRemotePluginCmdChannel(this));
        d_conn->AddListener(d_cmdServer);

        DBG_MSG((FP, "----> IviRemoteSimPlugin::Connect()\n"));
        d_conn->Connect();
        DBG_MSG((FP, "<---- IviRemoteSimPlugin::Connect()\n"));
    }

    /**** Send "Hello" message ****/
    RemoteAppConnection::WriteUint32(IVI_REMOTE_INIT, buf, idx);
    d_conn->Send(REMOTE_CMD_IDX, idx, buf);

    /**** Wait for the instance-naming command ****/
    if ((ret = WaitCmd()) < 0) {
        DBG_MSG((FP, "ERROR: Problem receiving Instance Name\n"));
    }

    if (ret < 0 || d_cmd != IVI_REMOTE_INST_NAME) {
        DBG_MSG((FP, "ERROR: Didn't receive instance name\n"));
        return;
    }

    d_instName = (char *)&d_data[4];
    DBG_MSG((FP, "InstName = %s\n", (char *)&d_data[4]));

    d_ok = true;
}

/********************************************************************
 * HandleDisconnect()
 ********************************************************************/
void IviRemoteSimPlugin::HandleDisconnect()
{
}

/********************************************************************
 * ConstructSim()
 ********************************************************************/
IviSim *IviRemoteSimPlugin::ConstructSim()
{
    IviSim  *sim = 0;
#if 0
    int      argc;
    char   **argv;

    sprintf(buf, "sim %s", d_instName.value());
    iviSplitString(buf, &argc, &argv);
    sim = new IviRemoteIviSim(d_interp, argc, argv, pv);
    iviFreeStringArr(argv);
#endif

    return sim;
}

/********************************************************************
 * ConstructDDB()
 ********************************************************************/
DesignDB *IviRemoteSimPlugin::ConstructDDB()
{
    DesignDB     *ddb;
    int           argc;
    char        **argv;
    char          buf[1024];

    sprintf(buf, "ddb %s.ddb -sim %s", d_instName.value(), 
            d_instName.value());
    iviSplitString(buf, &argc, &argv);
    ddb = new DesignDB(d_interp, argc, argv);
    iviFreeStringArr(argv);

    return ddb;
}

/********************************************************************
 * ConstructDFIO()
 ********************************************************************/
DFIO *IviRemoteSimPlugin::ConstructDFIO()
{
    int           argc;
    char        **argv;
    DFIO         *dfio;
    char          buf[1024];

    sprintf(buf, "shm_dfio %s.dfio -file ivi.wav", d_instName.value());
    iviSplitString(buf, &argc, &argv);
    dfio = DFIOMgr_NewDFIO("shm_dfio_writer", d_interp, argc, argv);
    iviFreeStringArr(argv);

    return dfio;
}

/********************************************************************
 * ConstructSDB()
 ********************************************************************/
SigDB *IviRemoteSimPlugin::ConstructSDB()
{
    int           argc;
    char        **argv;
    SigDB        *sdb;
    char          buf[1024];

    sprintf(buf, "sdb %s.sdb -ddb %s.ddb -dfio %s.dfio", 
            d_instName.value(), d_instName.value(), d_instName.value());
    iviSplitString(buf, &argc, &argv);
    sdb = new IviRemoteSDBClnt(d_conn, d_interp, argc, argv);
    iviFreeStringArr(argv);

    return sdb;
}

/********************************************************************
 * SignalReadyForSimulation()
 ********************************************************************/
void IviRemoteSimPlugin::SignalReadyForSimulation()
{
    Uchar buf[16];
    Uint32 idx=0;

    DBG_MSG((FP, "SignalReadyForSimulation()\n"));

    fflush(stdout);

    ConstructSimObjs();

    RemoteAppConnection::WriteUint32(IVI_REMOTE_LOAD_DONE, buf, idx);

    /**** Find the simulation resolution ****/
    Int32 res  = d_iviSim->getSimResolution();
    Int32 mult = d_iviSim->getResMultiplier();

    RemoteAppConnection::WriteUint32(res, buf, idx);
    RemoteAppConnection::WriteUint32(mult, buf, idx);

    d_conn->Send(REMOTE_CMD_IDX, idx, buf);

    ProcessCmds(true);
}

/********************************************************************
 * SignalEndOfSimulation()
 ********************************************************************/
void IviRemoteSimPlugin::SignalEndOfSimulation(Uint64 simTime)
{
    Uchar               buf[16];
    Uint32              idx=0;
    Int32               scale = d_iviSim->getResMultiplier();

    DBG_MSG((FP, "End of Simulation (%d:%d)\n", simTime.high, 
                simTime.low));

    fflush(stdout);

    RemoteAppConnection::WriteUint32(IVI_REMOTE_STOP, buf, idx);
    RemoteAppConnection::WriteUint32(1, buf, idx); /* done with simulation */
    RemoteAppConnection::WriteUint32(simTime.low*scale, buf, idx);
    RemoteAppConnection::WriteUint32(simTime.high, buf, idx);

    d_conn->Send(REMOTE_CMD_IDX, idx, buf);

    ProcessCmds(true);
}

/********************************************************************
 * WaitCmd()
 ********************************************************************/
int IviRemoteSimPlugin::WaitCmd()
{
    int ret = 0;

    d_recv = false;
    do {
        ret = d_conn->Poll(REMOTE_POLL_FOREVER);
    } while ((ret > 0) && !d_recv);

    return ret;
}

/********************************************************************
 * ProcessCmds()
 ********************************************************************/
int IviRemoteSimPlugin::ProcessCmds(bool wait)
{
    int          ret=0;
    bool         done = false;
    s_cb_data    cbdata;
    s_vpi_time   time_s, tt;
    Uint32       max_time = 0;
    Uchar        buf[16];
    Uint32       idx=0;

    do {
        fflush(stdout);

        if ((ret = WaitCmd()) <= 0) {
            break;
        }

        fflush(stdout);

        switch (d_cmd) {
            case IVI_REMOTE_RUN:
                done = true;

                max_time = d_data[4] | (d_data[5]<<8) | 
                    (d_data[6]<<16) | (d_data[7]<<24);

                DBG_MSG((FP, "\tIVI_REMOTE_RUN: %d\n", max_time));

                /*** Now, modify max_time based on the resolution
                 *** multiplier
                 ***/
                max_time /= d_iviSim->getResMultiplier();

                if (!max_time) {
                    max_time = 1;
                }
                BeginRun(max_time);
                break;

            case IVI_REMOTE_CLOSE:
                RemoteAppConnection::WriteUint32(IVI_REMOTE_CLOSE_DONE, 
                        buf, idx);
                d_conn->Send(REMOTE_CMD_IDX, idx, buf);

                /**** Now, wait for a IVI_REMOTE_CLOSE_EXIT cmd before
                 **** actually exiting...
                 ****/
                while (1) {
                    int cmdret = WaitCmd();

                    if (cmdret <= 0) {
                        break;
                    } else if (d_cmd == IVI_REMOTE_CLOSE_EXIT) {
                        break;
                    }
                }

                exit(0);
                break;



            default:
                fprintf(stderr, "IviRemoteSimPlugin::ProcessCmds() - "
                        "Unknown command \"%d\"\n", d_cmd);
                break;
        }

    } while (!done);

    return ret;
}

/********************************************************************
 * SignalEndOfRun()
 ********************************************************************/
void IviRemoteSimPlugin::SignalEndOfRun(Uint64 simTime)
{
    Uchar         buf[16];
    Uint32        idx=0;
    Int32         scale = d_iviSim->getResMultiplier();
  
    /*** TODO: Make this 64-bit-time safe ***/
    RemoteAppConnection::WriteUint32(IVI_REMOTE_STOP, buf, idx);
    RemoteAppConnection::WriteUint32(0, buf, idx);
    RemoteAppConnection::WriteUint32(simTime.low*scale, buf, idx);
    RemoteAppConnection::WriteUint32(simTime.high, buf, idx);

    d_conn->Send(REMOTE_CMD_IDX, idx, buf);

    /**** Wait for the next command ****/
    ProcessCmds(true);
}

/********************************************************************
 * ConstructSimObjs()
 ********************************************************************/
int IviRemoteSimPlugin::ConstructSimObjs()
{
    char               **argv;
    int                  argc;
    char                 buf[1024];

    DBG_MSG((FP, "----> ConstructSimObjs()\n"));

    d_iviSim = ConstructSim();

    /**** Now, construct the DesignDB ****/
    d_ddb = ConstructDDB();
    d_ddb->scanDesign();

    /**** Add the ShmDFIOWriter ****/
    d_dfio = ConstructDFIO();

    if (!d_dfio) {
        fprintf(stderr, "ERROR: cannot construct dfio\n");
    }

//    d_dfio->setTimeUnit(d_iviSim->vpi_get(vpiTimePrecision, 0));

    d_dfio->setTimeUnit(d_iviSim->getSimResolution());
    d_dfio->setUnitMultiplier(d_iviSim->getResMultiplier());

    /**** Construct the SDB ****/
    d_sdb = ConstructSDB();

    /**** Now, configure Sim with '-ddb' and '-sdb' ****/
    sprintf(buf, "%s configure -ddb %s.ddb -sdb %s.sdb", d_instName.value(), 
            d_instName.value(), d_instName.value());
    iviSplitString(buf, &argc, &argv);
    d_iviSim->InstCmd(argc, argv);
    iviFreeStringArr(argv);

    DBG_MSG((FP, "<---- ConstructSimObjs()\n"));
}


