/*****************************************************************************
 *                                vector.h
 * Author: Matthew Ballance
 * Desc:   simple vector class...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#ifndef INCLUDED_VECTOR_H
#define INCLUDED_VECTOR_H
#include "types.h"
#include <stdio.h>

template <class vType> class Vector {

    /******************************************************
     * Public methods
     ******************************************************/
public:
    Vector(void);
    virtual ~Vector(void); 

    virtual inline vType *idx(Uint32 idx) {
        if (idx < numObjs) return (vType *)objArray[idx];
        else               return (vType *)0;
     }

    virtual inline void setIdx(Uint32 idx, vType *obj) {
        vType     **tmp;

        if (idx < numObjs) {
            objArray[idx] = obj;
        }
    }

    virtual inline Uint32 length(void) { return numObjs; }

    /**** Allow shrinking, but no expansion
     ****/
    virtual inline Uint32 setLength(Uint32 newlen) { 
        numObjs = (newlen <= numObjs)?newlen:numObjs;
        return numObjs;
    }

    virtual void Vector<vType>::empty(void) { 
        numObjs = 0;
    }

    virtual void Vector<vType>::append(vType *newObj);
    virtual void Vector<vType>::append(Vector<vType> *aVect);
    virtual void Vector<vType>::append(Vector<vType> *aVect, 
        Uint32 startIdx, Uint32 endIdx);

    virtual int Vector<vType>::remove(vType *rm_el);

    virtual void Vector<vType>::setEqual(Vector<vType> *aVect);

    virtual void Vector<vType>::swap(void);
    virtual void Vector<vType>::swap(Uint32 startIdx, Uint32 endIdx);

    vType **get_storage() { return objArray; }

private:
    vType       **objArray;
    Uint32       maxObjs;
    Uint32       numObjs;
};

/************************************************************
 * Constructor()
 ************************************************************/
template <class vType> Vector<vType>::Vector(void)
{
    objArray = new vType*[8];
    maxObjs  = 8;
    numObjs  = 0;
}

/************************************************************
 * append
 ************************************************************/
template <class vType> void Vector<vType>::append(vType *newObj)
{
    vType    **tmp;
    Uint32     i;

    if (maxObjs <= numObjs) {
       tmp = objArray;
       objArray = new vType*[8+numObjs];
       for (i=0; i<numObjs; i++) {
           objArray[i] = tmp[i];
       }
       delete[] tmp;
       maxObjs += 8;
    }

    objArray[numObjs++] = newObj;
}

/************************************************************
 * append(Vector<vType>)
 ************************************************************/
template <class vType> void Vector<vType>::append(Vector<vType> *aVect)
{
    Uint32 i; 
    for (i=0; i<aVect->numObjs; i++) {
        append(aVect->objArray[i]);
    }
}

/************************************************************
 * append(Vector<vType>, startIdx, endIdx)
 ************************************************************/
template <class vType> void Vector<vType>::append(
    Vector<vType> *srcVect, Uint32 startIdx, Uint32 endIdx)
{
    Uint32 i;
    for (i=startIdx; i<=endIdx; i++) {
        append(srcVect->objArray[i]);
    }
}

/************************************************************
 * remove()
 ************************************************************/
template <class vType> int Vector<vType>::remove(vType *rm_el)
{
    Uint32 i, removed=-1;

    for (i=0; i<numObjs; i++) {
        if (objArray[i] == rm_el) {

            /**** Move elements from the end of the list down... 
             **** arr[x] = arr[x+1]
             ****  x = i:(numObjs-1)
             ****/
            for (Uint32 x=i; x<(numObjs-1); x++) {
                objArray[x] = objArray[x+1];
            }
            removed=0;
            break;
        }
    }

    if (i == numObjs) {
        fprintf(stderr, "VECTOR ERROR :: Element %x not on vector %x\n",
                rm_el, this);
    } else {
        numObjs--;
    }

    return removed;
}

/************************************************************
 * swap()
 ************************************************************/
template <class vType> void Vector<vType>::swap(void)
{
    swap(0, (numObjs)?(numObjs-1):0);
}

/************************************************************
 * swap(start, end)
 * NOTE: start, end are indices
 ************************************************************/
template <class vType> void Vector<vType>::swap(Uint32 start, Uint32 end)
{
    vType   *tmp;
    Uint32   i;

    if (start >= end) {
       return;
    }

    for (i=0; i<((end-start+1)/2); i++) {
        tmp = objArray[start+i];
        objArray[start+i] = objArray[end-i];
        objArray[end-i] = tmp;
    }
}

/************************************************************
 * setEqual()
 ************************************************************/
template <class vType> void Vector<vType>::setEqual(Vector<vType> *aVect)
{
    Uint32 i;
    empty();
    for (i=0; i<aVect->numObjs; i++) {
        append(aVect->objArray[i]);
    }
}

/************************************************************
 * Destructor()
 ************************************************************/
template <class vType> Vector<vType>::~Vector(void)
{

}

#endif

